/*
 * knights_engine.cpp
 *
 * Copyright (c) Stephen Thompson, 2009.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "control.hpp"
#include "dummy_callbacks.hpp"
#include "dungeon_map.hpp"
#include "event_manager.hpp"
#include "gore_manager.hpp"
#include "home_manager.hpp"
#include "knights_config.hpp"
#include "knights_engine.hpp"
#include "magic_map.hpp"
#include "mediator.hpp"
#include "mini_map.hpp"
#include "monster_manager.hpp"
#include "player.hpp"
#include "round.hpp"
#include "stuff_bag.hpp"
#include "task_manager.hpp"
#include "view_manager.hpp"

class KnightsEngineImpl {
public:
    KnightsEngineImpl()
        : view_manager(2),   // 2 players
          initial_update_needed(true),
          premapped(false)
    { } 

    // Keep a reference on the configuration
    boost::shared_ptr<const KnightsConfig> config;
    
    // Game state
    boost::shared_ptr<DungeonMap> dungeon_map;
    std::vector<boost::shared_ptr<Player> > players;

    // "Manager" classes (these handle various types of update to the
    // game world, and in some cases hold game state as well).    
    EventManager event_manager;      // processes "script events"
    GoreManager gore_manager;        // places blood splats
    HomeManager home_manager;        // handles securing of homes
    MonsterManager monster_manager;  // controls monster generation
    StuffManager stuff_manager;      // holds contents of stuff bags
    TaskManager task_manager;        // holds all "tasks" waiting to run.
    ViewManager view_manager;        // sends updates of dungeon views / mini maps to clients.

    std::vector<std::pair<const ItemType *, std::vector<int> > > starting_gears;
    bool initial_update_needed;
    bool premapped;
    void doInitialUpdateIfNeeded();
};


KnightsEngine::KnightsEngine(boost::shared_ptr<const KnightsConfig> config,
                             const MenuSelections &msel,
                             int house_col_0, int house_col_1)
    : pimpl(new KnightsEngineImpl)
{
    // Create the mediator, add stuff to it
    Mediator::createInstance(pimpl->event_manager, pimpl->gore_manager,
                             pimpl->home_manager, pimpl->monster_manager,
                             pimpl->stuff_manager, pimpl->task_manager,
                             pimpl->view_manager, config->getConfigMap());

    // Most initialization work is delegated to the KnightsConfig object
    std::vector<boost::shared_ptr<Quest> > quests;
    bool premapped = false;
    config->initializeGame(msel,
                           pimpl->dungeon_map,
                           quests,
                           pimpl->home_manager,
                           pimpl->players,
                           pimpl->stuff_manager,
                           pimpl->gore_manager,
                           pimpl->monster_manager,
                           pimpl->event_manager,
                           pimpl->premapped,
                           pimpl->starting_gears,
                           pimpl->task_manager,
                           house_col_0,
                           house_col_1);
    
    Mediator::instance().setMap(pimpl->dungeon_map);
    
    for (int i = 0; i < pimpl->players.size(); ++i) {
        Mediator::instance().addPlayer(*pimpl->players[i]);
    }
    
    // Make sure we save a copy of the config.
    pimpl->config = config;
}

KnightsEngine::~KnightsEngine()
{
    // To prevent MediatorUnavailable exceptions we create a "dummy" callbacks object
    DummyCallbacks dcb;
    Mediator::instance().setCallbacks(&dcb);

    // Clear the map out first. This makes sure that (for example)
    // Knight dtor does not attempt to call methods on the deleted Player objects.
    if (pimpl->dungeon_map) {
        pimpl->dungeon_map->clearAll();
        pimpl->dungeon_map.reset();
    }

    // Now delete the pimpl
    pimpl.reset();

    // Finally, release the mediator.
    Mediator::destroyInstance();
}


//
// update
//

void KnightsEngine::update(int time_delta, KnightsCallbacks &callbacks)
{
    Mediator::instance().setCallbacks(&callbacks); 
    pimpl->doInitialUpdateIfNeeded();
    pimpl->task_manager.advanceToTime(pimpl->task_manager.getGVT() + time_delta);
    Mediator::instance().setCallbacks(0);
}

void KnightsEngineImpl::doInitialUpdateIfNeeded()
{
    if (initial_update_needed) {

        // Set each player's mini map size
        const int width = dungeon_map->getWidth();
        const int height = dungeon_map->getHeight();
        for (int i = 0; i < players.size(); ++i) {
            players[i]->getMiniMap().setSize(width, height);
        }

        for (int i = 0; i < players.size(); ++i) {
            // Transfer the starting gear to the Player
            for (int j = 0; j < starting_gears.size(); ++j) {
                players[i]->addStartingGear(*starting_gears[j].first, starting_gears[j].second);
            }
            
            // Spawn the player
            players[i]->respawn();
        }
        
        // Do premapping if wanted
        if (premapped) {
            for (int i = 0; i < players.size(); ++i) {
                MagicMapping(players[i]->getKnight());
            }
        }

        // Send out special quest message if needed
        for (int i = 0; i < players.size(); ++i) {
            Mediator::instance().getCallbacks().getStatusDisplay(i).setQuestMessage(players[i]->getQuestMessage());
        }
        
        // Send out initial quest icons
        for (int i = 0; i < players.size(); ++i) {
            std::vector<StatusDisplay::QuestIconInfo> icons;
            players[i]->getQuestIcons(icons);
            Mediator::instance().getCallbacks().getStatusDisplay(i).setQuestIcons(icons);
        }

        initial_update_needed = false;
    }
}


//
// setControl
//

void KnightsEngine::setControl(int player, const UserControl *control)
{
    const Control * ctrl = static_cast<const Control*>(control);
    boost::shared_ptr<Player> p = pimpl->players.at(player);
    p->setControl(ctrl);
}
