/*
 * knights_config_impl.hpp
 *
 * Copyright (c) Stephen Thompson, 2009.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef KNIGHTS_CONFIG_IMPL_HPP
#define KNIGHTS_CONFIG_IMPL_HPP

#include "colour_change.hpp"
#include "config_map.hpp"
#include "dungeon_layout.hpp"
#include "item_type.hpp"
#include "map_support.hpp"
#include "menu.hpp"
#include "menu_constraints.hpp"
#include "monster_type.hpp"
#include "overlay.hpp"
#include "segment_set.hpp"

// kfile
#include "kfile.hpp"

// coercri
#include "gfx/color.hpp"

#include "boost/scoped_ptr.hpp"
#include "boost/shared_ptr.hpp"
#include <string>
#include <vector>

class Action;
class Anim;
class ColourChange;
class Control;
class DungeonDirective;
class DungeonGenerator;
class EventManager;
class GoreManager;
class Graphic;
class HomeManager;
class ItemGenerator;
class ItemType;
class MenuInt;
class MenuItem;
class MenuSelections;
class MonsterManager;
class Player;
class Quest;
class Segment;
class Sound;
class StuffManager;
class Tile;
class UserControl;

class KnightsConfigImpl {
public:

    //
    // ctor, dtor
    //

    explicit KnightsConfigImpl(const std::string &config_filename);
    ~KnightsConfigImpl();


    //
    // types used below
    //

    struct DungeonBlock {
        BlockType bt;
        std::string exits;
    };

    struct SegmentTileData {
        std::vector<boost::shared_ptr<Tile> > tiles;
        std::vector<const ItemType*> items;
    };


    //
    // stuff forwarded from KnightsConfig
    //

    void getAnims(std::vector<const Anim*> &anims) const;
    void getGraphics(std::vector<const Graphic*> &graphics) const;
    void getOverlays(std::vector<const Overlay*> &overlays) const;
    void getSounds(std::vector<const Sound*> &sounds) const;
    void getStandardControls(std::vector<const UserControl*> &controls) const;
    void getOtherControls(std::vector<const UserControl*> &controls) const;
    void initializeGame(const MenuSelections &msel,
                        boost::shared_ptr<DungeonMap> &dungeon_map,
                        std::vector<boost::shared_ptr<Quest> > &quests,
                        HomeManager &home_manager,
                        std::vector<boost::shared_ptr<Player> > &players,
                        StuffManager &stuff_manager,
                        GoreManager &gore_manager,
                        MonsterManager &monster_manager,
                        EventManager &event_manager,
                        bool &premapped,
                        std::vector<std::pair<const ItemType *, std::vector<int> > > &starting_gears,
                        TaskManager &task_manager,
                        int house_col_0,
                        int house_col_1) const;
    const Menu & getMenu() const { return menu; }
    const MenuConstraints & getMenuConstraints() const { return menu_constraints; }
    int getApproachOffset() const;
    void getHouseColours(std::vector<Coercri::Color> &result) const;
    std::string getQuestDescription(int quest_num) const;
    boost::shared_ptr<const ConfigMap> getConfigMap() const { return config_map; }

    
    //
    // Interface to KFile
    //

    KConfig::KFile * getKFile() { return kf.get(); }  // NULL if file not opened

    MapAccess popAccessCode(MapAccess dflt);
    void popAccessTable(MapAccess out[], MapAccess dflt);
    Action * popAction();
    Action * popAction(Action *dflt);
    Anim * popAnim(bool batmode = false);
    Anim * popAnim(Anim *dflt, bool batmode = false);
    BlockType popBlockType();
    bool popBool();
    bool popBool(bool dflt);
    Control * popControl();
    Control * popControl(Control *dflt);
    void popControlSet(std::vector<const Control*> &which_control_set);
    DungeonBlock popDungeonBlock();
    DungeonDirective * popDungeonDirective();
    DungeonLayout * popDungeonLayout();
    Graphic * popGraphic();
    Graphic * popGraphic(Graphic *dflt);
    void popHouseColours(const ConfigMap &config_map);
    void popHouseColoursMenu();
    ItemGenerator * popItemGenerator();
    ItemSize popItemSize();
    ItemSize popItemSize(ItemSize dflt);
    ItemType * popItemType();
    ItemType * popItemType(ItemType *dflt);
    MapDirection popMapDirection();
    MapDirection popMapDirection(MapDirection dflt);
    void popMenu();
    void popMenuSpace();
    MenuItem popMenuItem();
    MenuInt * popMenuInt();
    void popMenuValue(int val, MenuItem &menu_item);
    string popMenuValueName(int val);
    void popMenuValueDirective(const string &key, int value);
    Overlay * popOverlay();
    Overlay * popOverlay(Overlay *dflt);
    void popOverlayOffsets();
    void popPotionRenderer();
    int popProbability();  // pop a float, multiply by 100 and round to nearest int. (Error if result <0 or >100.)
    int popProbability(int dflt);
    void popQuestDescriptions();
    RandomDungeonLayout * popRandomDungeonLayout();
    Colour popRGB();
    Segment * popSegment(const std::vector<SegmentTileData> &tile_map, int &category);
    Segment * popSegmentData(const std::vector<SegmentTileData> &tile_map, int w, int h);
    void popSegmentList(const std::vector<SegmentTileData> &tile_map);
    void popSegmentRooms(Segment &);
    void popSegmentSet();
    void popSegmentSwitches(Segment &);
    void popSegmentTiles(std::vector<SegmentTileData> &tile_map);
    void popSegmentTile(SegmentTileData &seg_tile, bool &items_yet);
    void popSkullRenderer();
    Sound * popSound();
    struct StairInfo {
        bool is_stair;
        bool special;
        MapDirection down_direction;
    };
    StairInfo popStairsDown();
    boost::shared_ptr<Tile> popTile();
    boost::shared_ptr<Tile> popTile(boost::shared_ptr<Tile> dflt);
    int popTileItems(const MapAccess acc[]);  // acc is just there to provide a default answer.
    void popTileList(std::vector<boost::shared_ptr<Tile> > &output);

    int getSegmentCategory(const std::string &); // returns -1 if empty string given
    int getTileCategory(const std::string &);    // ditto
    void useSegmentCategory(int sc) { doUse(sc, segment_categories.size(), segment_categories_used); }
    void useTileCategory(int tc) { doUse(tc, tile_categories.size(), tile_categories_used); }
    
    KConfig::RandomIntContainer & getRandomIntContainer() { return random_ints; }

private:
    //
    // helper functions
    //

    DungeonLayout * doPopDungeonLayout(int w, int h);
    void defineSegmentCategory(int sc) { doUse(sc, segment_categories.size(), segment_categories_defined); }
    void defineTileCategory(int tc) { doUse(tc, tile_categories.size(), tile_categories_defined); }
    void doUse(int item, int sz, std::vector<bool> &container);
    void processTile(const std::vector<SegmentTileData> &tmap, Segment &seg, int x, int y);
    void readMenu(const Menu &menu, const MenuSelections &msel, DungeonGenerator &dgen,
                  std::vector<boost::shared_ptr<Quest> > &quests) const;
    const SegmentTileData & tileFromInt(const std::vector<SegmentTileData> &tile_map, int t);
    shared_ptr<Tile> makeDeadKnightTile(boost::shared_ptr<Tile>, const ColourChange &);
    
    //
    // helper classes
    //

    class Sentry {
    public:
        Sentry(KnightsConfigImpl &c) : cfg(c) { }
        ~Sentry();
        KnightsConfigImpl &cfg;
    };
    friend class Sentry;

    struct Popper {
        Popper(KConfig::KFile &k) : kf(k) { }
        ~Popper();
        KConfig::KFile &kf;
    };

    
private:
    // Storage for all 'basic' game objects, i.e. stuff loaded directly from the config file.
    // (Deleted by ~KnightsConfigImpl.)

    std::map<const Value *, Action *> actions;
    std::map<const Value *, Anim *> anims;
    std::map<const Value *, Control *> controls;
    std::map<const Value *, DungeonDirective *> dungeon_directives;
    std::map<const Value *, DungeonLayout *> dungeon_layouts;
    std::map<const Value *, Graphic *> graphics;
    std::map<const Value *, ItemGenerator *> item_generators;
    std::map<const Value *, ItemType *> item_types;
    std::vector<ItemType *> special_item_types;
    std::map<const Value *, MenuInt *> menu_ints;
    std::map<const Value *, Overlay *> overlays;
    std::map<const Value *, RandomDungeonLayout *> random_dungeon_layouts;
    KConfig::RandomIntContainer random_ints;
    std::map<const Value *, Segment *> segments;
    std::map<std::string, Sound *> sounds;
    std::map<const Value *, boost::shared_ptr<Tile> > tiles;
    SegmentSet segment_set;

    // Menu
    Menu menu;
    MenuConstraints menu_constraints;
    std::vector<DungeonDirective*> global_dungeon_directives;
    typedef std::map<int, std::vector<DungeonDirective*> > val_directives_map;
    typedef std::map<std::string, val_directives_map> key_val_directives_map;
    key_val_directives_map menu_dungeon_directives;
    typedef std::map<int, std::vector<boost::shared_ptr<Quest> > > val_quests_map;
    typedef std::map<std::string, val_quests_map> key_val_quests_map;
    key_val_quests_map menu_quests;

    // Various things
    boost::shared_ptr<Tile> wall_tile, horiz_door_tile[2], vert_door_tile[2];
    std::vector<ColourChange> house_colours_normal, house_colours_invulnerable;  // One entry per House
    std::vector<Coercri::Color> house_col_vector;
    std::vector<boost::shared_ptr<ColourChange> > secured_cc;
    Anim * knight_anim;   // In colours of house 0, ie house_colours[0] is always 'empty'.
    std::vector<Anim *> knight_anims;   // copies of 'knight_anim' with appropriate colour changes
    ItemType * default_item;
    std::vector<const Control *> control_set;
    const Graphic *stuff_bag_graphic;
    std::vector<std::string> standard_quest_descriptions;

    // Monsters, gore
    const Graphic *blood_icon;
    std::vector<boost::shared_ptr<Tile> > blood_tiles, dead_knight_tiles, dead_bat_tiles;
    boost::shared_ptr<Tile> dead_zombie_tile;
    std::vector<boost::shared_ptr<Tile> > other_zombie_tiles;
    std::vector<boost::shared_ptr<Tile> > bat_pit_tiles;
    boost::scoped_ptr<MonsterType> vampire_bat_type, zombie_type;
    int bat_speed, zombie_speed;
    const KConfig::RandomInt * bat_health;
    const KConfig::RandomInt * zombie_health;
    const Anim * bat_anim;
    const Anim * zombie_anim;
    const ItemType * zombie_weapon;
    std::vector<boost::shared_ptr<Tile> > zombie_ai_avoid;
    const ItemType * zombie_ai_hit;
    const ItemType * zombie_ai_fear;

    // The generic hook system
    std::map<std::string, const Action *> hooks;

    // Some maps from names (in the config system) to integer id codes.
    // These are non-empty only during loading of the config file.
    std::map<std::string, int> menu_item_names;
    std::map<std::string, int> segment_categories;
    std::map<std::string, int> tile_categories;
    std::vector<bool> segment_categories_used;
    std::vector<bool> segment_categories_defined;
    std::vector<bool> tile_categories_used;
    std::vector<bool> tile_categories_defined;
    
    // Config Map.
    boost::shared_ptr<ConfigMap> config_map;

    // Overlay offsets
    struct OverlayData {
        int ofsx, ofsy;
        MapDirection dir;
    };
    OverlayData overlay_offsets[Overlay::N_OVERLAY_FRAME][4];
    
    // kf is non-null only during the constructor.
    boost::shared_ptr<KConfig::KFile> kf;


    // extra graphics for the dead knight tiles. added 30-May-2009
    std::vector<boost::shared_ptr<Graphic> > dead_knight_graphics;
};

#endif
