/*
 * knight.hpp
 *
 * Knight: represents a player (knight) in the dungeon.
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef KNIGHT_HPP
#define KNIGHT_HPP

#include "creature.hpp"
#include "potion_magic.hpp"

#include <list>
#include <map>
#include <vector>
using namespace std;

class DispelObserver;
class DungeonView;
class ItemType;
class Player;
class StuffContents;
class Task;

class Knight : public Creature {
public:
    // if backpack_capacities is non-null, then it points to a map
    // giving the maximum number for each itemtype. (if an itemtype is
    // not in the map then the number held is assumed to be
    // unlimited.) (the map is not copied!)
    Knight(Player &pl, const map<const ItemType *, int> * backpack_capacities,
           int health, MapHeight height, const ItemType * default_item,
           const Anim * lower_anim, int spd);
    virtual ~Knight();

    Player & getPlayer() { return player; }
    const Player & getPlayer() const { return player; }
    

    //
    // Magic
    //
    // "Potion magic" controls invisibility, strength, quickness,
    // regeneration, super (only one of which can be active at a
    // time). Invulnerability and poison immunity can be turned on and
    // off independently. In each case "stop_time" gives the gvt after
    // which the effect runs out.
    //
    void setPotionMagic(PotionMagic, int stop_time);
    void setInvulnerability(bool i, int stop_time);
    void setPoisonImmunity(bool pi, int stop_time);
    void setSenseKnight(bool sk, int stop_time);
    void setRevealLocation(bool rl, int stop_time);
    void setReveal2(bool rl);
    void setCrystalBall(bool cb);  // uses a counter.
    void dispelMagic();  // cancel all magic effects
    PotionMagic getPotionMagic() const;
    bool getInvulnerability() const;
    bool getPoisonImmunity() const;
    bool getSenseKnight() const;
    bool getRevealLocation() const;
    bool getCrystalBall() const;

    void startHomeHealing();
    void stopHomeHealing();

    // we may add "dispel observers":
    // (There is no "rm" function but they are automatically removed
    // when the weak pointer expires.)
    void addDispelObserver(weak_ptr<DispelObserver> o) { dispel_list.push_front(o); }
    
    
    // overridden from creature:
    virtual void onDeath(DeathMode);  // calls dropAllItems, also places a knight corpse
    virtual bool hasStrength() const;
    virtual bool hasQuickness() const;

    virtual const char * getWeaponDownswingHook() const { return "HOOK_WEAPON_DOWNSWING"; }

    //
    // Inventory 
    //
    // -- item in hand is handled by Creature (but we override
    // Creature::setItemInHand to ensure that the default item is
    // handled properly).
    //
    // -- "backpack" items are stored as (itemtype, number_held) pairs.
    //
    virtual void setItemInHand(const ItemType *i);
    bool canDropHeld() const { return getItemInHand() && getItemInHand() != default_item; }
    
    int getBackpackCount() const { return backpack.size(); }   // get no of itemtypes in backpack
    const ItemType & getBackpackItem(int idx) const { return *backpack[idx].first; }   // get itemtype
    int getNumCarried(int idx) const { return backpack[idx].second; }                  // get number held (by index)
    int getNumCarried(const ItemType &itype) const;     // get number held (by itemtype)
    
    bool canAddToBackpack(const ItemType &itype) const;  // true if (at least one of) "itype" can be added
    int addToBackpack(const ItemType &itype, int no_to_add);  // returns number *actually* added.
    void rmFromBackpack(const ItemType &itype, int no_to_rm);
    
    void dropAllItems(bool move_back = false);  // drops the entire inventory.

    
    // 
    // override damage(), poison() to check for invulnerability and/or poison immunity
    // (and also to tell DungeonView about updated health, and to run damage hooks).
    //
    virtual void damage(int amount, int stun_until = -1, bool inhibit_squelch = false);
    virtual void addToHealth(int amount);
    virtual void poison();


    // throwing of backpack items (as opposed to item-in-hand).
    // canThrowItem: true if Creature::canThrow() and you are carrying itemtype and itemtype->canThrow().
    bool canThrowItem(const ItemType &itemtype, bool strict) const;
    void throwItem(const ItemType &itemtype);
    

protected:
    virtual bool isParalyzed() const;
    virtual void throwAwayItem(const ItemType *);
    
private:
    void unloadBackpack(StuffContents &sc);
    int getMaxNo(const ItemType &) const;
    int backpackFind(const ItemType &) const;
    void resetMagic();
    void resetSpeed();
    void setResetMagicTask(int);
    friend class ResetMagicTask;
    
private:
    typedef vector<pair<const ItemType *, int> > BackpackType;
    BackpackType backpack;
    const ItemType * default_item;
    const map<const ItemType*, int> * b_capacity;
    Player & player;
    PotionMagic potion_magic; int potion_stop_time;
    int invuln_stop_time;
    int poison_immun_stop_time;
    int sense_kt_stop_time;
    int reveal_locn_stop_time;
    int reveal_2;
    int crystal_ball_count;
    shared_ptr<Task> regeneration_task, home_healing_task;
    list<weak_ptr<DispelObserver> > dispel_list;
};

#endif
