/*
 * item_respawn_task.cpp
 * 
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "dungeon_map.hpp"
#include "item_generator.hpp"
#include "item_respawn_task.hpp"
#include "knight.hpp"
#include "mediator.hpp"
#include "player.hpp"
#include "rng.hpp"
#include "room_map.hpp"
#include "stuff_bag.hpp"
#include "task_manager.hpp"
#include "tile.hpp"

ItemRespawnTask::ItemRespawnTask(const ItemGenerator *item_gen_,
                                 const ItemType *lockpicks_,
                                 int interval_,
                                 int time_base_,
                                 int odds_)
    : item_gen(item_gen_), lockpicks(lockpicks_),
      interval(interval_), time_base(time_base_), odds(odds_)
{ }

int ItemRespawnTask::countLockpicks() const
{
    if (!lockpicks) return 0;
    
    int result = 0;
    const Mediator & mediator = Mediator::instance();

    // Count how many players are holding lockpicks
    for (vector<Player*>::const_iterator it = mediator.getPlayers().begin(); it != mediator.getPlayers().end(); ++it) {
        shared_ptr<Knight> kt((*it)->getKnight());
        if (kt && kt->getNumCarried(*lockpicks) > 0) {
            ++result;
        }
    }

    // Count how many stuff bags contain lockpicks
    result += mediator.getStuffManager().getNumBagsContaining(*lockpicks);

    return result;
}

void ItemRespawnTask::execute(TaskManager &tm)
{
    const int gvt = tm.getGVT();

    // Randomly decide if we should generate an item    
    if (g_rng.getInt(0, gvt/interval) >= time_base && g_rng.getInt(0, odds) == 0) {

        // Select an item to generate.
        const pair<const ItemType *, int> item_to_generate = item_gen->get();

        // If it is the lock picks then we must count how many lock picks are "in play" (ie. held by
        // knights, or part of stuff bags).
        // If this number is 1 there is a 50-50 chance of generating more lock picks;
        // if it is 2 or more then no further lock picks will be generated.

        bool can_proceed = true;
        
        if (item_to_generate.first == lockpicks) {
            const int lockpicks_in_play = countLockpicks();
            if (lockpicks_in_play >= 2) can_proceed = false;
            else if (lockpicks_in_play) can_proceed = g_rng.getBool();  // 50-50 chance
        }

        if (can_proceed) {
            const Mediator& mediator(Mediator::instance());
            shared_ptr<DungeonMap> dmap = mediator.getMap();
            if (dmap) { // safety check.
            
                // try 5 times, to give it a fighting chance. if failed to place after 5 attempts then give up.
                for (int i=0; i<5; ++i) {
                    const int x = g_rng.getInt(0, dmap->getWidth());
                    const int y = g_rng.getInt(0, dmap->getHeight());
                    const MapCoord mc(x,y);
                    
                    // don't place items if there is an item already there:
                    if (dmap->getItem(mc)) continue;
                    
                    // don't place items if the tile blocks items:
                    vector<shared_ptr<Tile> > tiles;
                    dmap->getTiles(mc, tiles);
                    if (find_if(tiles.begin(), tiles.end(), BlocksItems()) != tiles.end()) continue;
                    
                    // don't place items in the same room that a knight is currently in:
                    bool visible_to_kt = false;
                    for (vector<Player*>::const_iterator it = mediator.getPlayers().begin(); it != mediator.getPlayers().end(); ++it) {
                        const MapCoord kt_pos = (*it)->getKnightPos();
                        if (!kt_pos.isNull()) {
                            if (dmap->getRoomMap() && dmap->getRoomMap()->inSameRoom(mc, kt_pos)) {
                                visible_to_kt = true;
                                break;
                            }
                        }
                    }
                    if (visible_to_kt) continue;
                    
                    // ok we can place it
                    shared_ptr<Item> item(new Item(*item_to_generate.first, item_to_generate.second));
                    dmap->addItem(mc, item);
                    break;
                
                }
            }
        }
    }
    
    // re-schedule after "interval"
    tm.addTask(shared_from_this(), TP_NORMAL, gvt + interval);
}
