/*
 * home_manager.cpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "dungeon_map.hpp"
#include "home_manager.hpp"
#include "mediator.hpp"
#include "player.hpp"
#include "rng.hpp"
#include "special_tiles.hpp"

void HomeManager::addHome(const MapCoord &pos, MapDirection facing)
{
    homes[make_pair(pos,facing)] = 0;
}

void HomeManager::secureHome(const Player &pl, DungeonMap &dmap, const MapCoord &pos,
                             MapDirection facing, shared_ptr<Tile> secured_wall_tile)
{
    HomeMap::iterator it = homes.find(make_pair(pos,facing));
    if (it == homes.end()) return;
    if (it->second == &pl) return; // don't secure homes twice ...
    
    // We have to secure the home
    // First find the home tile
    MapCoord home_mc = DisplaceCoord(pos,facing);
    shared_ptr<Home> home_tile;
    
    vector<shared_ptr<Tile> > tiles;
    dmap.getTiles(home_mc, tiles);
    for (int i=0; i<tiles.size(); ++i) {
        home_tile = dynamic_pointer_cast<Home>(tiles[i]);
        if (home_tile) break;
    }

    // Now do the actual securing
    if (it->second == 0) {
        it->second = &pl; // let the record show that this home is secured by this player only
        if (home_tile) {
            home_tile->secure(dmap, home_mc, pl.getSecuredCC()); // set the colour-change too.
        }
    } else {
        homes.erase(it);  // let the record show that this home is secured by both players
        if (home_tile) {
            dmap.rmTile(home_mc, home_tile);   // remove the home tile
            dmap.addTile(home_mc, secured_wall_tile);  // add a wall tile instead
        }
    }

    // Finally, we have to reassign homes if necessary
    const vector<Player*> &players (Mediator::instance().getPlayers());
    for (int i=0; i<players.size(); ++i) {
        // Check the home
        HomeMap::iterator it = homes.find(make_pair(players[i]->getHomeLocation(),
                                                    players[i]->getHomeFacing()));
        if (it == homes.end() // my home secured by both players
        || (it->second && it->second != players[i])) {   // my home secured by somebody, who is not me
            // My home seems to have been secured. Replace it with a new one
            pair<MapCoord,MapDirection> p = getRandomHomeFor(*players[i]);
            players[i]->resetHome(p.first, p.second);
        }

        // NOTE: We don't bother reassigning exit points. This means that the exit point
        // remain usable by both players until it is secured by both players. (After that,
        // the quest cannot be completed by escaping the dungeon. This never makes the
        // game impossible though, because you can always win by fighting to the death.)

        // ALSO: The above only applies to random or guarded exits. If the exit is set
        // to "own entry" or "other's entry" then the exit will get reassigned automatically
        // when the relevant home is reassigned.
    }
}

pair<MapCoord,MapDirection> HomeManager::getRandomHomeFor(const Player &pl) const
{
    // First of all, work out how many homes are not secured against pl
    int count = 0;
    for (HomeMap::const_iterator it = homes.begin(); it != homes.end(); ++it) {
        if (it->second == 0 || it->second == &pl) {
            ++count;
        }
    }

    if (count == 0) return make_pair(MapCoord(), D_NORTH);
    
    // Generate a random number
    int r = g_rng.getInt(0, count);

    // Now find the rth home
    for (HomeMap::const_iterator it = homes.begin(); it != homes.end(); ++it) {
        if (it->second == 0 || it->second == &pl) {
            --r;
            if (r < 0) {
                // Return this home
                return it->first;
            }
        }
    }

    ASSERT(0);
    return make_pair(MapCoord(), D_NORTH);
}

