/*
 * gore_manager.hpp
 *
 * Blood/gore effects. These are done as special Tiles which overlay the normal map tiles.
 * There are three sorts of gore tiles:
 * (i) Blood splats. These are placed whenever a creature loses hit points.
 * (ii) Monster corpses (dead vampire bats or zombies).
 * (iii) Knight corpses (these are handled slightly differently to monsters).
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef GORE_MANAGER_HPP
#define GORE_MANAGER_HPP

class DungeonMap;
class Graphic;
class MapCoord;
class MonsterType;
class Player;
class Tile;

#include "boost/shared_ptr.hpp"
using namespace boost;

#include <map>
#include <vector>
using namespace std;

class GoreManager {
public:
    GoreManager();
    
    //
    // setup
    //

    // For monster corpses and blood tiles, the "add" routine can be called more than once,
    // and this will create a sequence of tiles to be used one after the other. (Eg this is
    // used for the blood tiles, with increasing amounts of blood on each tile in the
    // sequence.)

    // For knight corpses, tiles with and without blood splats can be given. (The latter is
    // used when a knight is poisoned, the former when he is killed in the normal way.)

    // NB -- if addMonsterCorpse is called with monster == NULL, then we add a 'generic'
    // corpse tile, that is not associated with a particular monster, but still counts
    // as a corpse for purposes of zombie activity. (This is used for the bone tiles.)

    void setKnightCorpse(const Player &, shared_ptr<Tile> with_blood,
                         shared_ptr<Tile> without_blood);
    void addMonsterCorpse(const MonsterType *monster, shared_ptr<Tile> t);
    void addBloodTile(shared_ptr<Tile>);
    void setBloodIcon(const Graphic *);

    
    // 
    // placing blood/gore into the map.
    // These are called from Mediator at appropriate points.
    //

    void placeBlood(DungeonMap &, const MapCoord &);
    void placeKnightCorpse(DungeonMap &, const MapCoord &, const Player &, bool blood);
    void placeMonsterCorpse(DungeonMap &, const MapCoord &, const MonsterType &);

    
private:
    void placeNextTile(DungeonMap &dmap, const MapCoord &mc,
                       const vector<shared_ptr<Tile> > &tiles);
    void placeTile(DungeonMap &dmap, const MapCoord &mc, shared_ptr<Tile> new_tile);

private:
    // blood tiles
    vector<shared_ptr<Tile> > blood_tiles;
    const Graphic *blood_icon;

    // knight corpses
    struct KnightCorpse {
        shared_ptr<Tile> with_blood;
        shared_ptr<Tile> without_blood;
    };
    map<const Player*, KnightCorpse> knight_corpses;

    // monster corpses
    map<const MonsterType *, vector<shared_ptr<Tile> > > monster_corpses;
};

#endif
