/*
 * dungeon_map.hpp
 *
 * DungeonMap: holds Entities, Items and Tiles.
 * The server holds a DungeonMap which represents the definitive
 * current state of the dungeon. Each client also holds its own
 * DungeonMap, which contains only what the client knows about (and
 * will therefore be incomplete and/or out of date with respect to the
 * server).
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef DUNGEON_MAP_HPP
#define DUNGEON_MAP_HPP

#include "map_support.hpp"

#include "boost/multi_index_container.hpp"
#include "boost/multi_index/hashed_index.hpp"
#include "boost/shared_ptr.hpp"
using namespace boost;
using namespace boost::multi_index;

#include <list>
#include <stack>
#include <vector>
using namespace std;

class Creature;
class Entity;
class Item;
class ItemReplacementTask; // used internally
class RoomMap;
class Task;
class TaskManager;
class Tile;

class DungeonMap {
    friend class MapHelper;
public:
    DungeonMap();
    ~DungeonMap();
    void create(int w, int h);  // initialize empty dmap of the given size.

    // room map (NB room map is deleted by DungeonMap dtor.)
    void setRoomMap(RoomMap *r); // deletes the old room map
    RoomMap * getRoomMap() const { return room_map; }
    
    // check access level for a given square & height
    // (if entity present already, returns A_BLOCKED, otherwise looks up MapAccess for
    // underlying tile.)
    MapAccess getAccess(const MapCoord &mc, MapHeight h, Entity *ignore = 0) const;

    // same, but checks only tiles -- not entities.
    MapAccess getAccessTilesOnly(const MapCoord &mc, MapHeight h) const;
    
    // check if we can place a new missile at a given square
    bool canPlaceMissile(const MapCoord &mc, MapDirection dir_of_travel) const;

    // get width, height
    int getWidth() const { return map_width; }
    int getHeight() const { return map_height; }

    // check that a given coord is in the map
    // (which means x between 0 and width-1, and y between 0 and height-1 -- there are never
    // any "holes" in a map.)
    bool valid(const MapCoord &mc) const {
        return (mc.getX() >= 0 && mc.getX() < map_width && mc.getY() >= 0
                && mc.getY() < map_height);
    }
    
    // add/remove items/tiles
    // these return true if successful.
    bool addItem(const MapCoord &mc, shared_ptr<Item> it);
    bool rmItem(const MapCoord &mc);
    bool addTile(const MapCoord &mc, shared_ptr<Tile> ti);
    bool rmTile(const MapCoord &mc, shared_ptr<Tile> ti);

    // this removes all tiles at a given mapcoord.
    void clearTiles(const MapCoord &mc);

    // remove all entities, items and tiles from the map. (Used at the end of a game...)
    void clearAll();
    
    // get entities at a given square (approached entities do not count)
    // existing contents of "results" are cleared.
    // NB if an entity is halfway between squares then it counts as being on both of those
    // squares.
    void getEntities(const MapCoord &mc, vector<shared_ptr<Entity> > &results) const;
    
    // get all entities, including approached ones
    // return them in "results" vector (existing contents of "results" vector are cleared).
    void getAllEntities(const MapCoord &mc, vector<shared_ptr<Entity> > &results) const;

    // get target creature: this is used for combat.
    // allow_both_sqs: if true, can target attacker's sq or the sq one ahead of the attacker.
    // if false, can target the sq. that the attacker is mainly on (either his sq or the
    // sq ahead, depending on whether offset < or > 50%).
    shared_ptr<Creature> getTargetCreature(const Entity & attacker, bool allow_both_sqs) const;
    
    // get item at a given square
    shared_ptr<Item> getItem(const MapCoord &mc) const;

    // get tiles at a given square
    // Returns results in a vector (not the most efficient method, but probably the most
    // flexible). Existing contents of the vector are deleted.
    void getTiles(const MapCoord &mc, vector<shared_ptr<Tile> > &output) const;
    
    // displaced items:
    // Used for items that should have been dropped (eg when a knight died), but there
    // was no space to drop them. They will be automatically re-inserted into the dungeon
    // (onto a random square) as soon as possible.
    void addDisplacedItem(shared_ptr<Item> i);

private:
    // helper function for getAllEntities
    void doEntity(const MapCoord &mc, MapDirection facing, 
                 vector<shared_ptr<Entity> > &results) const;

    // helper for getTargetCreature
    shared_ptr<Creature> getTargetCreatureHelper(const Entity &, const MapCoord &) const;
    
    // get index corresponding to a mapcoord
    int index(const MapCoord &mc) const
        { ASSERT(valid(mc)); return mc.getY() * map_width + mc.getX(); }
    
private:
    // Dungeon contents. Use hash tables for entities and items (since
    // these are likely to be sparse) but a straightforward
    // index-by-square for tiles (tiles are likely to be dense, with
    // at least one on each square). Use boost::multi_index for the
    // hash tables (I don't like to rely on hash_map/hash_set since
    // they seem to be non-standardized between different compilers).

    // Note that any Entity halfway between two squares is given two
    // separate entries in entities index (except for approaching
    // entities which only get one entry, on their "base" square).

    template<class T>
    struct GetPos : unary_function<pair<MapCoord,T>, MapCoord> {
        const MapCoord & operator()(const pair<MapCoord, T> &p) const {
            return p.first;
        }
    };
    template<class T>
    struct GetSecond : unary_function<pair<MapCoord,T>, T> {
        const T & operator()(const pair<MapCoord,T> &p) const {
            return p.second;
        }
    };

    typedef boost::multi_index_container<pair<MapCoord, shared_ptr<Entity> >, 
        indexed_by< hashed_non_unique<GetPos<shared_ptr<Entity> > > > > EntityContainer;
    typedef boost::multi_index_container<pair<MapCoord, shared_ptr<Item> >,
        indexed_by< hashed_unique<GetPos<shared_ptr<Item> > > > > ItemContainer;
    typedef vector<list<shared_ptr<Tile> > > TileContainer;

    EntityContainer entities;
    ItemContainer items;
    TileContainer tiles;

    int map_width, map_height;

    RoomMap *room_map;

    // List of items that need to be re-placed
    stack<shared_ptr<Item> > displaced_items;
    friend class ItemReplacementTask;
};


#endif
