/*
 * dungeon_generator.hpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef DUNGEON_GENERATOR_HPP
#define DUNGEON_GENERATOR_HPP

#include "dungeon_generation_failed.hpp"
#include "kconfig_fwd.hpp"
#include "map_support.hpp"

#include "boost/shared_ptr.hpp"
using namespace boost;

#include <map>
#include <string>
#include <vector>
using namespace std;

class DungeonMap;
class ItemGenerator;
class ItemType;
class KnightsConfigImpl;
class MenuSelections;
class MonsterManager;
class RandomDungeonLayout;
class Segment;
class SegmentSet;
class Tile;

//
// Usage: create a DungeonGenerator, apply your DungeonDirectives to it, then call "generate".
// NB The present algorithm assumes that all Segments are the same size.
//

enum HomeType { H_NONE, H_CLOSE, H_AWAY, H_RANDOM };
enum ExitType { E_NONE, E_SELF, E_OTHER, E_RANDOM, E_SPECIAL };

class DungeonGenerator {
public:
    // hdoor and vdoor are wooden doors (used to knock through between adjacent segments).
    DungeonGenerator(const SegmentSet &rs, shared_ptr<Tile> wall_,
                     shared_ptr<Tile> hdoor1, shared_ptr<Tile> hdoor2,
                     shared_ptr<Tile> vdoor1, shared_ptr<Tile> vdoor2) 
        : segment_set(rs), wall(wall_), horiz_door_1(hdoor1), horiz_door_2(hdoor2),
          vert_door_1(vdoor1), vert_door_2(vdoor2), rlayout(0), exit_type(E_NONE),
          exit_category(-1),
          home_type(H_NONE), premapped(false), pretrapped(false), nkeys(0),
          vampire_bats(0), zombie_activity(0), total_stuff_weight(0),
          respawn_lockpicks(0), respawn_interval(99999), respawn_time_base(99999), respawn_odds(99999) { }

    // dungeon generation parameters
    // these are typically called by DungeonDirectives.
    void setExitType(ExitType e, int ecat) { exit_type = e; exit_category = ecat; }
    void setHomeType(HomeType h) { home_type = h; }
    void setLayouts(const vector<const RandomDungeonLayout *> &lay) { layouts = lay; }
    void setPremapped(bool p) { premapped = p; }
    void setPretrapped(bool p) { pretrapped = p; }
    void setNumKeys(int n) { nkeys = n; }
    void addRequiredItem(int number, const ItemType &itype);
    void addRequiredSegment(int segment_category)
        { required_segments.push_back(segment_category); }
    void setStuff(int tile_category, int chance, const ItemGenerator *generator, int weight);
    void setVampireBats(int v) { vampire_bats = v; }
    void setZombieActivity(int z) { zombie_activity = z; }
    void addStartingGear(const ItemType &it, const vector<int> &nos) { gears.push_back(make_pair(&it, nos)); }
    void setRespawnGenerator(const ItemGenerator *ig) { respawn_generator = ig; }
    void setRespawnLockpicks(const ItemType *it) { respawn_lockpicks = it; }
    void setRespawnTimings(int interval, int time_base, int odds)
        { respawn_interval = interval; respawn_time_base = time_base; respawn_odds = odds; }
    
    // "generate" routine -- generates the dungeon map.
    void generate(DungeonMap &dmap);

    // Add vampire bats to an already-generated dungeon map.
    void addVampireBats(DungeonMap &dmap, MonsterManager &mmgr,
                        int nbats_normal, int nbats_guarded_exit);
    
    // query home/exit locations
    int getNumHomes() const { return 2; }
    void getHome(int i, MapCoord &pos, MapDirection &facing_toward_home) const;
    ExitType getExitType() const { return exit_type; }
    void getExit(int i, MapCoord &pos, MapDirection &facing_toward_exit) const;
    int getNumHomesOverall() const { return unassigned_homes.size() + assigned_homes.size(); }
    void getHomeOverall(int i, MapCoord &pos, MapDirection &facing_toward_home) const;
    
    // query "incidental" data
    // Note: This is getting a bit large now. Should probably create a separate class to hold this stuff...
    int getVampireBats() const { return vampire_bats; }
    int getZombieActivity() const { return zombie_activity; }
    bool isPremapped() const { return premapped; }
    const vector<pair<const ItemType*, vector<int> > > & getStartingGears() const { return gears; }
    const ItemGenerator * getRespawnGenerator() const { return respawn_generator; }
    const ItemType * getRespawnLockpicks() const { return respawn_lockpicks; }
    int getRespawnInterval() const { return respawn_interval; }
    int getRespawnTimeBase() const { return respawn_time_base; }
    int getRespawnOdds() const { return respawn_odds; }
    
private:
    struct BlockInfo {
        int x, y;
        bool special;
    };
    struct IsSpecial {
        bool operator()(const BlockInfo &b) { return b.special; }
    };

    void fetchEdge(int&,int&);
    void fetchEdgeOrBlock(int&,int&);
    void setHomeSegment(int,int,int,int);
    void setSpecialSegment(int,int,int);
    void doLayout();
    static void shiftHomes(vector<pair<MapCoord,MapDirection> > &, int, int);
    void chopTopSide();
    void chopBottomSide();
    void chopLeftSide();
    void chopRightSide();
    void compress();
    void copySegmentsToMap(DungeonMap&);
    void fillWithWalls(DungeonMap&, const MapCoord&, int, int);
    bool placeDoor(DungeonMap&, const MapCoord &, const MapCoord &, const MapCoord &,
                   const MapCoord &, const MapCoord &, shared_ptr<Tile>, shared_ptr<Tile>);
    void knockThroughDoors(DungeonMap&);
    void generateExits();
    void generateRequiredItems(DungeonMap &);
    bool forbidden(int) const;
    void generateStuff(DungeonMap &);

    static int findItemCategory(const DungeonMap &, const MapCoord &,
                                const vector<shared_ptr<Tile> > &);
    static void placeItem(DungeonMap &, const MapCoord &, vector<shared_ptr<Tile> > &,
                          const ItemType &, int no);
    void generateLocksAndTraps(DungeonMap &, int);

    bool getHorizExit(int x, int y) { return horiz_exits[y*(lwidth-1)+x]; }
    bool getVertExit(int x, int y) { return horiz_exits[y*lwidth+x]; }

    void placeRegularBats(DungeonMap &dmap, MonsterManager &mmgr, int nbats);
    void placeGuardedBats(DungeonMap &dmap, MonsterManager &mmgr, int nbats,
                          shared_ptr<Tile> bat_tile, int left, int bottom,
                          int right, int top);
    
private:
    // "fixed" data
    const SegmentSet &segment_set;
    shared_ptr<Tile> wall, horiz_door_1, horiz_door_2, vert_door_1, vert_door_2;
    const RandomDungeonLayout *rlayout; // currently selected layout
    vector<const RandomDungeonLayout*> layouts;
    ExitType exit_type;
    int exit_category;
    HomeType home_type;
    bool premapped; 
    bool pretrapped;
    int nkeys;
    vector<int> required_segments;  // contains segment categories
    vector<const ItemType*> required_items; // contains only non-null pointers
    int vampire_bats, zombie_activity;
    struct StuffInfo {
        int chance;
        int weight;   // actually max(input weight, 0)
        const ItemGenerator *generator;
        bool forbid;  // true if input weight was <0
    };
    map<int,StuffInfo> stuff;  // LHS is the tile category.
    int total_stuff_weight;
    vector<pair<const ItemType *, vector<int> > > gears; // Starting gear.
    const ItemGenerator * respawn_generator;
    const ItemType * respawn_lockpicks;
    int respawn_interval;
    int respawn_time_base;
    int respawn_odds;
    
    // "temporary" data (set during dungeon generation)
    vector<BlockInfo> blocks, edges;  // layout info.
    vector<bool> horiz_exits, vert_exits;  // more layout info
    vector<const Segment *> segments;            // actual segments chosen
    vector<int> segment_categories;              // the category of each chosen segment.
    int lwidth, lheight;       // layout dimensions (in segments)
    int rwidth, rheight;       // segment dimensions (in tiles)
    vector<pair<MapCoord,MapDirection> > unassigned_homes, assigned_homes, exits;   // homes
};


class DungeonDirective {
public:
    virtual ~DungeonDirective() { }

    // this pops an argument list from the kfile.
    static DungeonDirective * create(const string &name, KnightsConfigImpl &kc);

    // "apply" requires the MenuSelections object (since some DungeonDirectives are tied
    // to the value of certain menu options; eg DungeonItem(i_gem, GetMenu("gems_needed")).)
    virtual void apply(DungeonGenerator &dg, const MenuSelections &) const = 0;
};

#endif
