/*
 * dungeon_generator.cpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "dungeon_generator.hpp"
#include "dungeon_map.hpp"
#include "item.hpp"
#include "item_generator.hpp"
#include "knights_config_impl.hpp"
#include "lockable.hpp"
#include "menu_int.hpp"
#include "menu_selections.hpp"
#include "monster_manager.hpp"
#include "rng.hpp"
#include "room_map.hpp"
#include "segment.hpp"
#include "tile.hpp"

#include "kfile.hpp"
using namespace KConfig;

#ifdef lst2
#undef lst2
#endif

//
// DungeonDirectives
//

class DungeonBats : public DungeonDirective {
public:
    explicit DungeonBats(const MenuInt &b) : bat_level(b) { }
    virtual void apply(DungeonGenerator &dg, const MenuSelections &ms) const
        { dg.setVampireBats(bat_level.getValue(ms)); }
private:
    const MenuInt &bat_level;
};

class DungeonExit : public DungeonDirective {
public:
    explicit DungeonExit(ExitType e, int rc) : et(e), rcat(rc) { }
    virtual void apply(DungeonGenerator &dg, const MenuSelections &) const
        { dg.setExitType(et, rcat); }
private:
    ExitType et;
    int rcat;
};

class DungeonGear : public DungeonDirective {
public:
    DungeonGear(const ItemType& it, const vector<int> &nos) : itype(it), numbers(nos) { }
    virtual void apply(DungeonGenerator &dg, const MenuSelections &) const
        { dg.addStartingGear(itype, numbers); }
private:
    const ItemType &itype;
    vector<int> numbers;
};

class DungeonHome : public DungeonDirective {
public:
    explicit DungeonHome(HomeType h) : ht(h) { }
    virtual void apply(DungeonGenerator &dg, const MenuSelections &) const
        { dg.setHomeType(ht); }
private:
    HomeType ht;
};

class DungeonItem : public DungeonDirective {
public:
    DungeonItem(const MenuInt &no, const ItemType &it) : number(no), itype(it) { }
    virtual void apply(DungeonGenerator &dg, const MenuSelections &ms) const
        { dg.addRequiredItem(number.getValue(ms), itype); }
private:
    const MenuInt &number;
    const ItemType &itype;
};

class DungeonKeys : public DungeonDirective {
public:
    DungeonKeys(KnightsConfigImpl &kc, KFile::List &lst);
    virtual void apply(DungeonGenerator &dg, const MenuSelections &) const;
private:
    vector<const ItemType *> keys;
};

DungeonKeys::DungeonKeys(KnightsConfigImpl &kc, KFile::List &lst)
{
    if (lst.getSize() < 2) return;
    for (int i=0; i<lst.getSize(); ++i) {
        lst.push(i);
        keys.push_back(kc.popItemType());
    }
}

void DungeonKeys::apply(DungeonGenerator &dg, const MenuSelections &) const
{
    if (keys.size()<2) {
        dg.setNumKeys(0);
    } else {
        dg.setNumKeys(keys.size()-1);
        for (int i=0; i<keys.size(); ++i) {
            // require one of each key plus one set of lock picks
            if (keys[i]) dg.addRequiredItem(1, *keys[i]);
        }
    }
}

class DungeonLayoutDir : public DungeonDirective { 
public:
    void addLayout(const RandomDungeonLayout *r) { layouts.push_back(r); }
    virtual void apply(DungeonGenerator &dg, const MenuSelections &) const
        { dg.setLayouts(layouts); }
private:
    vector<const RandomDungeonLayout *> layouts;
};

class DungeonPremapped : public DungeonDirective {
public:
    virtual void apply(DungeonGenerator &dg, const MenuSelections &) const
        { dg.setPremapped(true); }
};

class DungeonPretrapped : public DungeonDirective {
public:
    virtual void apply(DungeonGenerator &dg, const MenuSelections &) const
        { dg.setPretrapped(true); }
};

class DungeonRespawn : public DungeonDirective {
public:
    explicit DungeonRespawn(const ItemGenerator *ig_, const ItemType * lock_) : ig(ig_), lockpicks(lock_) { }
    virtual void apply(DungeonGenerator &dg, const MenuSelections &) const
        { dg.setRespawnGenerator(ig); dg.setRespawnLockpicks(lockpicks); }
private:
    const ItemGenerator *ig;
    const ItemType * lockpicks;
};

class DungeonRespawnTimings : public DungeonDirective {
public:
    DungeonRespawnTimings(int interval_, int time_base_, int odds_)
        : interval(interval_), time_base(time_base_), odds(odds_) { }
    virtual void apply(DungeonGenerator &dg, const MenuSelections &) const
        { dg.setRespawnTimings(interval, time_base, odds); }
private:
    int interval, time_base, odds;
};

class DungeonSegment : public DungeonDirective {
public:
    explicit DungeonSegment(int r) : segment_cat(r) { }
    virtual void apply(DungeonGenerator &dg, const MenuSelections &) const
        { dg.addRequiredSegment(segment_cat); }
private:
    int segment_cat;
};

class DungeonStuff : public DungeonDirective {
public:
    DungeonStuff(int tc, int ch, const ItemGenerator *ig_, int wt)
        : tile_cat(tc), chance(ch), ig(ig_), weight(wt) { }
    virtual void apply(DungeonGenerator &dg, const MenuSelections &) const
        { dg.setStuff(tile_cat, chance, ig, weight); }
private:
    int tile_cat;
    int chance;
    const ItemGenerator *ig;
    int weight;
};

class DungeonZombies : public DungeonDirective { 
public:
    explicit DungeonZombies(const MenuInt &a) : activity(a) { }
    virtual void apply(DungeonGenerator &dg, const MenuSelections &ms) const
        { dg.setZombieActivity(activity.getValue(ms)); }
private:
    const MenuInt &activity;
};



//
// DungeonDirective::create
//

DungeonDirective * DungeonDirective::create(const string &name, KnightsConfigImpl &kc)
{
    if (!kc.getKFile()) return 0;

    if (name.substr(0,7) != "Dungeon") {
        kc.getKFile()->errExpected("MenuDirective");
        return 0;
    }
    string n = name.substr(7);

    if (n == "Bats" || n == "Zombies") {
        KFile::List lst(*kc.getKFile(), "", 1);
        lst.push(0);
        const MenuInt *number = kc.popMenuInt();
        if (!number) kc.getKFile()->errExpected("integer or GetMenu directive");
        if (n=="Bats") {
            return new DungeonBats(*number);
        } else {
            return new DungeonZombies(*number);
        }

    } else if (n == "Exit") {
        KFile::List lst(*kc.getKFile(), "", 1);
        lst.push(0);
        string x = kc.getKFile()->popString();
        if (x == "same_as_entry") return new DungeonExit(E_SELF,0);
        else if (x == "others_entry") return new DungeonExit(E_OTHER,0);
        else if (x == "total_random") return new DungeonExit(E_RANDOM,0);
        else {
            int rcat = kc.getSegmentCategory(x);
            kc.useSegmentCategory(rcat);
            if (rcat >= 0) {
                return new DungeonExit(E_SPECIAL, rcat);
            } else {
                kc.getKFile()->errExpected("exit type");
            }
        }

    } else if (n == "Gear") {
        KFile::List lst(*kc.getKFile(), "", 2);

        lst.push(0);
        const ItemType *const itype = kc.popItemType();
        if (!itype) kc.getKFile()->errExpected("item type");

        lst.push(1);
        KFile::List lst2(*kc.getKFile(), "item quantities");
        if (lst2.getSize() < 1) kc.getKFile()->error("empty quantity list for DungeonGear");

        vector<int> nos(lst2.getSize());
        for (int i=0; i<lst2.getSize(); ++i) {
            lst2.push(i);
            nos[i] = kc.getKFile()->popInt();
        }

        return new DungeonGear(*itype, nos);

    } else if (n == "Home") {
        KFile::List lst(*kc.getKFile(), "", 1);
        lst.push(0);
        string x = kc.getKFile()->popString();
        if (x == "total_random") return new DungeonHome(H_RANDOM);
        else if (x == "close_to_other") return new DungeonHome(H_CLOSE);
        else if (x == "away_from_other") return new DungeonHome(H_AWAY);
        else kc.getKFile()->errExpected("entry type");

    } else if (n == "Item") {
        KFile::List lst(*kc.getKFile(), "", 2);
        lst.push(0);
        const MenuInt *number = kc.popMenuInt();
        if (!number) kc.getKFile()->errExpected("integer or GetMenu directive");
        lst.push(1);
        const ItemType *const itype = kc.popItemType();
        if (!itype) kc.getKFile()->errExpected("item type");
        return new DungeonItem(*number, *itype);

    } else if (n == "Keys") {
        KFile::List lst(*kc.getKFile(), "");
        return new DungeonKeys(kc, lst);

    } else if (n == "Layout") {
        KFile::List lst(*kc.getKFile(), "");
        DungeonLayoutDir * dir = new DungeonLayoutDir;
        for (int i=0; i<lst.getSize(); ++i) {
            lst.push(i);
            dir->addLayout( kc.popRandomDungeonLayout() );
        }
        return dir;

    } else if (n == "Premapped") {
        KFile::List lst(*kc.getKFile(), "", 0);
        return new DungeonPremapped;
        
    } else if (n == "Pretrapped") {
        KFile::List lst(*kc.getKFile(), "", 0);
        return new DungeonPretrapped;

    } else if (n == "Respawn") {
        KFile::List lst(*kc.getKFile(), "", 1);
        lst.push(0);
        const ItemGenerator *ig = kc.popItemGenerator();
        lst.push(1);
        const ItemType *itype = kc.popItemType();
        return new DungeonRespawn(ig, itype);

    } else if (n == "RespawnTimings") {
        KFile::List lst(*kc.getKFile(), "", 3);
        lst.push(0);
        const int interval = kc.getKFile()->popInt();
        lst.push(1);
        const int time_base = kc.getKFile()->popInt();
        lst.push(2);
        const int odds = kc.getKFile()->popInt();
        return new DungeonRespawnTimings(interval, time_base, odds);
        
    } else if (n == "Segment") {
        KFile::List lst(*kc.getKFile(), "", 1);
        lst.push(0);
        string rname = kc.getKFile()->popString();
        int rcat = kc.getSegmentCategory(rname);
        kc.useSegmentCategory(rcat);
        if (rcat >= 0) return new DungeonSegment(rcat);
    
    } else if (n == "Stuff") {
        KFile::List lst(*kc.getKFile(), "", 3, 4);
        lst.push(0);
        string tname = kc.getKFile()->popString();
        int tcat = kc.getTileCategory(tname);
        kc.useTileCategory(tcat);
        lst.push(1);
        int prob = kc.popProbability();
        lst.push(2);
        ItemGenerator *ig = kc.popItemGenerator();
        lst.push(3);
        int wt = kc.getKFile()->popInt(0);
        return new DungeonStuff(tcat, prob, ig, wt);

    } else {
        kc.getKFile()->errExpected("MenuDirective");
    }

    return 0;
}


//
// The dungeon generator itself
//

void DungeonGenerator::addRequiredItem(int number, const ItemType &itype)
{
    for (int i=0; i<number; ++i) {
        required_items.push_back(&itype);
    }
}

void DungeonGenerator::setStuff(int tile_category, int chance, const ItemGenerator *generator,
                                int weight)
{
    if (stuff.find(tile_category) != stuff.end()) {
        total_stuff_weight -= stuff[tile_category].weight;
    }
    StuffInfo si;
    si.chance = chance;
    si.generator = generator;
    si.weight = std::max(0, weight);
    si.forbid = weight < 0;
    total_stuff_weight += si.weight;
    stuff[tile_category] = si;
}

void DungeonGenerator::generate(DungeonMap &dmap)
{
    // We try each possible layout three times. If there is an exception of any kind
    // (e.g. due to the dungeon being too small for the quest) we just try the next
    // possible layout. If we run out of layouts the exception is re-thrown.

    for (vector<const RandomDungeonLayout*>::const_iterator i = layouts.begin();
    i != layouts.end(); ++i) {
        for (int j=0; j<3; ++j) {
            rlayout = *i;

            try {
                // clear everything...
                blocks.clear();
                edges.clear();
                horiz_exits.clear();
                vert_exits.clear();
                segments.clear();
                segment_categories.clear();
                lwidth = lheight = rwidth = rheight = 0;
                unassigned_homes.clear();
                assigned_homes.clear();
                exits.clear();

                // create the layout
                doLayout();        // lays out the segments, and 'assigns' homes and special exit point.
                compress();        // deletes any unused space around the edges of the map.

                // create the DungeonMap itself
                dmap.create(lwidth*(rwidth+1)+1, lheight*(rheight+1)+1);
                copySegmentsToMap(dmap);         // copies the segments into the map.
                knockThroughDoors(dmap);      // creates doorways between the different segments.

                // fill in exits
                generateExits();
    
                // lock doors and chests
                generateLocksAndTraps(dmap, nkeys);
    
                // generate items
                generateRequiredItems(dmap);
                generateStuff(dmap);

                // If we get here then we have had a successful dungeon generation.
                return;

            } catch (const DungeonGenerationFailed &) {
                vector<const RandomDungeonLayout*>::const_iterator i2 = i;
                ++i2;
                if (i2 == layouts.end() && j == 2) {
                    // Give up.
                    throw;
                }
                // Otherwise we go on to the next attempt
            }
            // (Other exceptions are allowed to propagate upwards)
        }
    }
}


void DungeonGenerator::fetchEdge(int &x, int &y)
{
    if (edges.empty()) throw DungeonGenerationFailed();
    x = edges.back().x;
    y = edges.back().y;
    edges.pop_back();
}

void DungeonGenerator::fetchEdgeOrBlock(int &x, int &y)
{
    // This fetches an edge (preferably), or else a block.
    if (!edges.empty()) {
        x = edges.back().x;
        y = edges.back().y;
        edges.pop_back();
    } else if (!blocks.empty()) {
        x = blocks.back().x;
        y = blocks.back().y;
        blocks.pop_back();
    } else {
        throw DungeonGenerationFailed();
    }
}

void DungeonGenerator::setHomeSegment(int x, int y, int minhomes, int assign)
{
    const int max_attempts = 50;

    if (y < 0 || y >= lheight || x < 0 || x >= lwidth) return;

    // generate a segment and store it into the "segments" vector
    const Segment *r = 0;
    for (int i=0; i<max_attempts; ++i) {
        r = segment_set.getHomeSegment(minhomes);
        if (find(segments.begin(), segments.end(), r) == segments.end()) break;
    }
    if (!r) throw DungeonGenerationFailed();
    segments[y*lwidth+x] = r;

    // copy homes to appropriate lists (either assigned or unassigned).
    if (r) {
        rwidth = r->getWidth();
        rheight = r->getHeight();
        const int xbase = x*(rwidth+1)+1;
        const int ybase = y*(rheight+1)+1;
        vector<HomeInfo> h(r->getHomes());
        RNG_Wrapper myrng(g_rng);
        random_shuffle(h.begin(), h.end(), myrng);
        for (int i=0; i<h.size(); ++i) {
            MapCoord mc(h[i].x + xbase, h[i].y + ybase);
            MapDirection facing(h[i].facing);
            if (i < assign) {
                assigned_homes.push_back(make_pair(mc,facing));
            } else {
                unassigned_homes.push_back(make_pair(mc,facing));
            }
        }
    }
}

void DungeonGenerator::setSpecialSegment(int x, int y, int category)
{
    // homes are ignored this time. we just generate the segment.
    const int max_attempts = 50;
    if (y < 0 || y >= lheight || x < 0 || x >= lwidth) return;
    const Segment *r = 0;
    for (int i=0; i<max_attempts; ++i) {
        r = segment_set.getSpecialSegment(category);
        if (find(segments.begin(), segments.end(), r) == segments.end()) break;
    }
    if (!r) throw DungeonGenerationFailed();
    segments[y*lwidth + x] = r;
    segment_categories[y*lwidth+x] = category;
}

void DungeonGenerator::doLayout()
{
    RNG_Wrapper myrng(g_rng);

    int x, y;
    
    const int nplayers = 2;
    const int homes_required = nplayers;

    // randomize the layout first:
    const DungeonLayout * layout = rlayout->choose();
    
    // get width and height
    lwidth = layout->getWidth();
    lheight = layout->getHeight();

    // work out where the edges and blocks are.
    // flip and/or rotate the layout if necessary.
    const bool flipx = g_rng.getBool();
    const bool flipy = g_rng.getBool();
    const bool rotate = g_rng.getBool();
    for (int i=0; i<lwidth; ++i) {
        for (int j=0; j<lheight; ++j) {
            x=i;
            y=j;
            if (flipx) x = lwidth-1-x;
            if (flipy) y = lheight-1-y;
            if (rotate) {
                const int tmp = y;
                y = x;
                x = lheight-1-tmp;
            }

            BlockInfo bi;
            bi.x = x;
            bi.y = y;
            bi.special = false;
            switch (layout->getBlockType(i,j)) {
            case BT_NONE:
                // Don't add anything here :)
                break;
            case BT_BLOCK:
                blocks.push_back(bi);
                break;
            case BT_SPECIAL:
                bi.special = true;
                // fall through
            case BT_EDGE:
                edges.push_back(bi);
                break;
            }
        }
    }

    const int new_lwidth = rotate? lheight : lwidth;
    const int new_lheight = rotate? lwidth : lheight;

    // exits information also needs to be flipped and/or rotated
    vert_exits.resize(new_lwidth*(new_lheight-1));
    horiz_exits.resize((new_lwidth-1)*new_lheight);
    for (int i=0; i<lwidth-1; ++i) {
        for (int j=0; j<lheight; ++j) {
            x=i;
            y=j;
            if (flipx) x = lwidth-2-x;
            if (flipy) y = lheight-1-y;
            if (rotate) {
                // ynew = x
                // xnew = lheight-1-y
                vert_exits[x*new_lwidth + (lheight-1-y)] = layout->hasHorizExit(i,j);
            } else {
                horiz_exits[y*(new_lwidth-1) + x] = layout->hasHorizExit(i,j);
            }
        }
    }
    for (int i=0; i<lwidth; ++i) {
        for (int j=0; j<lheight-1; ++j) {
            x=i;
            y=j;
            if (flipx) x = lwidth-1-x;
            if (flipy) y = lheight-2-y;
            if (rotate) {
                // ynew = x
                // xnew = lheight-2-y
                horiz_exits[x*(new_lwidth-1) + (lheight-2-y)] = layout->hasVertExit(i,j);
            } else {
                vert_exits[y*new_lwidth + x] = layout->hasVertExit(i,j);
            }
        }
    }

    // randomize blocks and edges
    // set new width and height
    // resize segments array.
    random_shuffle(blocks.begin(), blocks.end(), myrng);
    random_shuffle(edges.begin(), edges.end(), myrng);
    lwidth = new_lwidth;
    lheight = new_lheight;
    segments.resize(lwidth*lheight);
    segment_categories.resize(lwidth*lheight);
    fill(segment_categories.begin(), segment_categories.end(), -1);

    // "Away From Other" homes, on Edges
    if (home_type == H_AWAY) {
        for (int i=0; i<nplayers; ++i) {
            fetchEdge(x, y);
            setHomeSegment(x, y, 1, 1);
        }
    }

    // "DungeonSegment()" segments -- on Edges if possible, Blocks otherwise.
    // NOTE that homes in these segments will not be added to the home-lists. This affects
    // guarded exit points only (at time of writing)... and these are dealt with separately
    // (see generateExits.)
    for (vector<int>::const_iterator it = required_segments.begin();
    it != required_segments.end(); ++it) {
        fetchEdgeOrBlock(x, y);
        setSpecialSegment(x, y, *it);
    }

    // Eliminate "special" tiles if they have not been assigned.
    // Also no further distinction between Edges and Blocks beyond this point.
    edges.erase(remove_if(edges.begin(), edges.end(), IsSpecial()), edges.end());
    copy(edges.begin(), edges.end(), back_inserter(blocks));
    edges.clear();

    // "Close To Other" homes
    if (home_type == H_CLOSE) {
        fetchEdgeOrBlock(x, y);
        setHomeSegment(x, y, 2, 2);
    }

    // Fill in all remaining blocks.
    while (!blocks.empty()) {
        const int h = homes_required - assigned_homes.size() - unassigned_homes.size();  // minimum number of homes still required
        const int nblocks = int(blocks.size());
        int n;  // *minimum* number of homes to generate in the new block
        if (h > 2*nblocks) throw DungeonGenerationFailed();
        else if (h == 2*nblocks) n = 2;  // need 2 in every block
        else if (h == 2*nblocks-1) n = 1;  // need at least one here plus two in all others
        else n = 0;  // can get away with 0 here if we want to
        fetchEdgeOrBlock(x, y);
        setHomeSegment(x, y, n, 0);
    }

    // Make sure there are enough homes in the "assigned" list.
    random_shuffle(assigned_homes.begin(), assigned_homes.end(), myrng);
    random_shuffle(unassigned_homes.begin(), unassigned_homes.end(), myrng);
    while (assigned_homes.size() < homes_required) {
        if (unassigned_homes.empty()) {
            // this can happen eg if you ask for gnome room on a 1x1 map... there won't be
            // any space left for the homes!!
            throw DungeonGenerationFailed();
        }
        assigned_homes.push_back(unassigned_homes.back());
        unassigned_homes.pop_back();
    }
}

void DungeonGenerator::shiftHomes(vector<pair<MapCoord,MapDirection> > &homes, int dx, int dy)
{
    for (int i=0; i<homes.size(); ++i) {
        homes[i].first.setX(homes[i].first.getX() + dx);
        homes[i].first.setY(homes[i].first.getY() + dy);
    }
}

void DungeonGenerator::chopLeftSide()
{
    vector<const Segment *> new_segments((lwidth-1)*lheight);
    vector<int> new_cats((lwidth-1)*lheight);
    for (int x=0; x<lwidth-1; ++x) {
        for (int y=0; y<lheight; ++y) {
            new_segments[y*(lwidth-1)+x] = segments[y*lwidth+x+1];
            new_cats[y*(lwidth-1)+x] = segment_categories[y*lwidth+x+1];
        }
    }
    segments.swap(new_segments);
    segment_categories.swap(new_cats);
    --lwidth;
}

void DungeonGenerator::chopRightSide()
{
    vector<const Segment *> new_segments((lwidth-1)*lheight);
    vector<int> new_cats((lwidth-1)*lheight);
    for (int x=0; x<lwidth-1; ++x) {
        for (int y=0; y<lheight; ++y) {
            new_segments[y*(lwidth-1)+x] = segments[y*lwidth+x];
            new_cats[y*(lwidth-1)+x] = segment_categories[y*lwidth+x];
        }
    }
    segments.swap(new_segments);
    --lwidth;
}

void DungeonGenerator::chopBottomSide()
{
    vector<const Segment *> new_segments(lwidth*(lheight-1));
    vector<int> new_cats(lwidth*(lheight-1));
    for (int x=0; x<lwidth; ++x) {
        for (int y=0; y<lheight-1; ++y) {
            new_segments[y*lwidth+x] = segments[(y+1)*lwidth+x];
            new_cats[y*lwidth+x] = segment_categories[(y+1)*lwidth+x];
        }
    }
    segments.swap(new_segments);
    --lheight;
}

void DungeonGenerator::chopTopSide()
{
    segments.resize(segments.size()-lwidth);
    segment_categories.resize(segment_categories.size()-lwidth);
    --lheight;
}

void DungeonGenerator::compress()
{
    // crop left side
    while (1) {
        if (lwidth<=0 || lheight<=0) throw DungeonGenerationFailed();
        bool empty = true;
        for (int y=0; y<lheight; ++y) {
            if (segments[y*lwidth+0] != 0) {
                empty = false;
                break;
            }
        }
        if (empty) {
            chopLeftSide();
            shiftHomes(assigned_homes, -rwidth+1, 0);
            shiftHomes(unassigned_homes, -rwidth+1, 0);
        } else {
            break;
        }
    }

    // crop right side
    while (1) {
        if (lwidth <= 0 || lheight <= 0) throw DungeonGenerationFailed();
        bool empty = true;
        for (int y=0; y<lheight; ++y) {
            if (segments[y*lwidth+(lwidth-1)] != 0) {
                empty = false;
                break;
            }
        }
        if (empty) {
            chopRightSide();
        } else {
            break;
        }
    }

    // crop bottom side
    while (1) {
        if (lwidth <= 0 || lheight <= 0) throw DungeonGenerationFailed();
        bool empty = true;
        for (int x=0; x<lwidth; ++x) {
            if (segments[0*lwidth+x] != 0) {
                empty = false;
                break;
            }
        }
        if (empty) {
            chopBottomSide();
            shiftHomes(assigned_homes, 0, -rheight+1);
            shiftHomes(unassigned_homes, 0, -rheight+1);
        } else {
            break;
        }
    }

    // crop top side
    while (1) {
        if (lwidth <= 0 || lheight <= 0) throw DungeonGenerationFailed();
        bool empty = true;
        for (int x=0; x<lwidth; ++x) {
            if (segments[(lheight-1)*lwidth + x] != 0) {
                empty = false;
                break;
            }
        }
        if (empty) {
            chopTopSide();
        } else {
            break;
        }
    }
}

void DungeonGenerator::copySegmentsToMap(DungeonMap & dmap)
{
    if (!dmap.getRoomMap()) {
        dmap.setRoomMap(new RoomMap);
    }

    // copy segments to map (this also adds rooms)
    for (int x=0; x<lwidth; ++x) {
        for (int y=0; y<lheight; ++y) {
            MapCoord corner( x*(rwidth+1)+1, y*(rheight+1)+1 );
            if (segments[y*lwidth+x]) {
                segments[y*lwidth+x]->copyToMap(dmap, corner);
            } else {
                fillWithWalls(dmap, corner, rwidth, rheight);
            }
        }
    }

    // tell RoomMap that all rooms have been added
    dmap.getRoomMap()->doneAddingRooms();
    
    // fill in walls (around the edge of each segment)
    if (wall) {

        for (int xs=0; xs<lwidth; ++xs) {
            for (int ys=0; ys<lheight; ++ys) {
                // horizontal walls
                for (int x=0; x<rwidth+2; ++x) {
                    MapCoord mc(xs*(rwidth+1)+x, ys*(rheight+1));
                    dmap.clearTiles(mc);
                    dmap.addTile(mc, wall->clone(false));
                    mc.setY((ys+1)*(rheight+1));
                    dmap.clearTiles(mc);
                    dmap.addTile(mc, wall->clone(false));
                }
                
                // vertical walls
                for (int y=0; y<rheight+2; ++y) {
                    MapCoord mc(xs*(rwidth+1), ys*(rheight+1)+y);
                    dmap.clearTiles(mc);
                    dmap.addTile(mc, wall->clone(false));
                    mc.setX((xs+1)*(rwidth+1));
                    dmap.clearTiles(mc);
                    dmap.addTile(mc, wall->clone(false));
                }
            }
        }
    }
}

void DungeonGenerator::fillWithWalls(DungeonMap & dmap, const MapCoord &corner,
                                     int width, int height)
{
    if (wall) {
        for (int y=0; y<height; ++y) {
            for (int x=0; x<width; ++x) {
                const MapCoord mc(corner.getX() + x, corner.getY() + y);
                dmap.clearTiles(mc);
                dmap.addTile(mc, wall->clone(false));
            }
        }
    }
}

bool DungeonGenerator::placeDoor(DungeonMap & dmap, const MapCoord &mc,
                                 const MapCoord &side1, const MapCoord &side2,
                                 const MapCoord &front, const MapCoord &back,
                                 shared_ptr<Tile> door_tile_1, shared_ptr<Tile> door_tile_2)
{
    if (!door_tile_1) return false;
    vector<shared_ptr<Tile> > tiles;

    // front and back must have A_CLEAR at H_WALKING
    if (dmap.getAccess(front, H_WALKING) != A_CLEAR) return false;
    if (dmap.getAccess(back, H_WALKING) != A_CLEAR) return false;

    // front and back must have at least one tile assigned
    // also: front and back must not be stair tiles.
    // also: front and back must allow items (this stops doors being generated in front of
    // pits).
    dmap.getTiles(front, tiles);
    if (tiles.empty()) return false;
    for (int i=0; i<tiles.size(); ++i) {
        if (tiles[i]->isStairOrTop() || !tiles[i]->itemsAllowed()) {
            return false;
        }
    }
    dmap.getTiles(back, tiles);
    if (tiles.empty()) return false;
    for (int i=0; i<tiles.size(); ++i) {
        if (tiles[i]->isStairOrTop() || !tiles[i]->itemsAllowed()) {
            return false;
        }
    }

    // sides must not already have doors on them -- check this using access
    // (should be A_BLOCKED).
    if (dmap.getAccess(side1, H_WALKING) != A_BLOCKED) return false;
    if (dmap.getAccess(side2, H_WALKING) != A_BLOCKED) return false;

    // The proposed door tile should not be a corner of a room.
    if (dmap.getRoomMap()->isCorner(mc)) return false;
    
    // OK.
    // Remove all existing tiles
    dmap.getTiles(mc, tiles);
    for (int i=0; i<tiles.size(); ++i) dmap.rmTile(mc, tiles[i]);

    // Add the new door tile.
    dmap.addTile(mc, door_tile_1->clone(false));
    if (door_tile_2) {
        dmap.addTile(mc, door_tile_2->clone(false));
    }

    return true;
}

void DungeonGenerator::knockThroughDoors(DungeonMap & dmap)
{
    const int max_attempts = 30;

    // NB there are two copies of the door code in this routine, one
    // for horiz doors and one for vert doors ....
    
    // Horizontal doors (vertical exits)
    // Doorway between (x,y) and (x,y+1).
    for (int x=0; x<lwidth; ++x) {
        for (int y=0; y<lheight-1; ++y) {
            if (segments[y*lwidth+x] && segments[(y+1)*lwidth+x] && vert_exits[y*lwidth+x]) {

                // Try to place 3 doors (but only up to max_attempts attempts)
                int ndoors_placed = 0;
                for (int i=0; i<max_attempts; ++i) {
                    MapCoord mc(g_rng.getInt(0, rwidth) + x*(rwidth+1) + 1,
                                (y+1)*(rheight+1));
                    MapCoord side1 = DisplaceCoord(mc, D_WEST);
                    MapCoord side2 = DisplaceCoord(mc, D_EAST);
                    MapCoord front = DisplaceCoord(mc, D_NORTH);
                    MapCoord back = DisplaceCoord(mc, D_SOUTH);
                    if ( placeDoor(dmap, mc, side1, side2, front, back, horiz_door_1,
                    horiz_door_2) ) {
                        ++ndoors_placed;
                    }
                    if (ndoors_placed == 3) break;  // success -- all three doors were placed.
                }

                // We might not have placed all 3 doors within the
                // time limit. If only 1 or 2 doors were placed, we
                // can accept that, but we can't accept 0 doors.
                if (ndoors_placed == 0) throw DungeonGenerationFailed();
            }
        }
    }

    // Vertical doors (horizontal exits)
    // Doorway between (x,y) and (x+1,y)
    // This is similar to the above (but with x and y swapped over basically).
    for (int x=0; x<lwidth-1; ++x) {
        for (int y=0; y<lheight; ++y) {
            if (segments[y*lwidth+x] && segments[y*lwidth+(x+1)]
            && horiz_exits[y*(lwidth-1)+x]) {
                int ndoors_placed = 0;
                for (int i=0; i<max_attempts; ++i) {
                    MapCoord mc((x+1)*(rwidth+1),
                                g_rng.getInt(0, rheight) + y*(rheight+1) + 1);
                    MapCoord side1 = DisplaceCoord(mc, D_NORTH);
                    MapCoord side2 = DisplaceCoord(mc, D_SOUTH);
                    MapCoord front = DisplaceCoord(mc, D_WEST);
                    MapCoord back = DisplaceCoord(mc, D_EAST);
                    if ( placeDoor(dmap, mc, side1, side2, front, back, vert_door_1,
                    vert_door_2) ) {
                        ++ndoors_placed;
                    }
                    if (ndoors_placed == 3) break;
                }
                if (ndoors_placed == 0) throw DungeonGenerationFailed();
            }
        }
    }
}


void DungeonGenerator::generateExits()
{
    // This generates the "standard" player exit points (not the special exit points).
    switch (exit_type) {
    case E_SELF:
        {
            for (int i=0; i<assigned_homes.size(); ++i) {
                exits.push_back(assigned_homes[i]);
            }
        }
        break;      
    case E_OTHER:
        {
            // This only works with two players at the moment
            if (assigned_homes.size()!=2) throw DungeonGenerationFailed();
            for (int i=1; i>=0; --i) {
                exits.push_back(assigned_homes[i]);
            }
        }
        break;
    case E_SPECIAL:
        {
            // This is a little unsophisticated, but fine for what we want to use it for:
            // Look for the first segment of the given category.
            // Then look for the first home within that segment.
            // Then add it to unassigned_homes, as well as setting it as the exit point.
            bool found = false;
            for (int x=0; x<lwidth; ++x) {
                for (int y=0; y<lheight; ++y) {
                    if (segment_categories[y*lwidth+x] == exit_category) {
                        const Segment *seg = segments[y*lwidth+x];
                        ASSERT(seg); // if seg_category is set, then so must seg be
                        if (!seg->getHomes().empty()) {
                            for (int i=0; i<assigned_homes.size(); ++i) {
                                const HomeInfo &hi(seg->getHomes().front());
                                exits.push_back(make_pair(MapCoord(hi.x + x*(rwidth+1)+1,
                                                                   hi.y + y*(rheight+1)+1),
                                                          hi.facing));
                                // add it to unassigned_homes, too (this is so wand of
                                // securing can work on it):
                                unassigned_homes.push_back(exits.back());
                            }
                            found = true;
                            break;
                        }
                    }
                }
                if (found) break;
            }
        }
        break;
    case E_RANDOM:
        {
            const int k = g_rng.getInt(0, unassigned_homes.size() + assigned_homes.size());
            for (int i=0; i<assigned_homes.size(); ++i) {
                if (k < assigned_homes.size()) exits.push_back(assigned_homes[k]);
                else exits.push_back(unassigned_homes[k-assigned_homes.size()]);
            }
        }
        break;
    default:
        {
            for (int i=0; i<assigned_homes.size(); ++i) {
                exits.push_back(make_pair(MapCoord(), D_NORTH));
            }
        }
        break;
    }
}


void DungeonGenerator::getHome(int i, MapCoord &pos, MapDirection &facing_toward_home) const
{
    // "assigned_homes" is what we want here
    if (i < 0 || i >= assigned_homes.size()) return;
    pos = assigned_homes[i].first;
    facing_toward_home = assigned_homes[i].second;
    pos = DisplaceCoord(pos, Opposite(facing_toward_home)); // we want the square 1 in front of the home.
}

void DungeonGenerator::getHomeOverall(int i, MapCoord &pos, MapDirection &facing) const
{
    // This needs to use either unassigned homes or assigned homes, as appropriate
    if (i < 0 || i >= getNumHomesOverall()) return;
    if (i < assigned_homes.size()) {
        pos = assigned_homes[i].first;
        facing = assigned_homes[i].second;
    } else {
        pos = unassigned_homes[i - assigned_homes.size()].first;
        facing = unassigned_homes[i - assigned_homes.size()].second;
    }
    pos = DisplaceCoord(pos, Opposite(facing));
}
     

void DungeonGenerator::getExit(int i, MapCoord &pos, MapDirection &facing) const
{
    if (i < 0 || i >= exits.size()) return;
    pos = exits[i].first;
    facing = exits[i].second;
    pos = DisplaceCoord(pos, Opposite(facing));
}

void DungeonGenerator::generateRequiredItems(DungeonMap &dmap)
{
    const int maxtries = 5;
    const int w = dmap.getWidth(), h = dmap.getHeight();

    vector<shared_ptr<Tile> > tiles;
    
    for (vector<const ItemType *>::iterator it = required_items.begin();
    it != required_items.end(); ++it) {
        bool found = false;
        for (int tries = 0; tries < maxtries; ++tries) {
            // Select a tile category
            if (total_stuff_weight == 0) throw DungeonGenerationFailed(); // can't have required items w/o stuff categories!!
            int t = g_rng.getInt(0, total_stuff_weight);
            int chosen_cat = -999;
            for (map<int,StuffInfo>::iterator si = stuff.begin(); si != stuff.end(); ++si) {
                t -= si->second.weight;
                if (t < 0) {
                    chosen_cat = si->first;
                    break;
                }
            }
            ASSERT(t<0);

            // Now randomly pick squares until we find one of the required category
            // (Or, if this is the last try, we accept any tile category that doesn't have 'forbid' set.)
            MapCoord mc;
            for (int q=0; q<w*h; ++q) {
                mc.setX(g_rng.getInt(0,w));
                mc.setY(g_rng.getInt(0,h));

                // work out the tile category
                dmap.getTiles(mc, tiles);
                const int cat = findItemCategory(dmap, mc, tiles);

                if (cat == chosen_cat || (tries==maxtries-1 && cat >= 0 && !forbidden(cat))) {
                    // OK, put the item here
                    placeItem(dmap, mc, tiles, **it, 1);
                    found = true;
                    break;
                }
            }
            if (found) break;
        }
        if (!found) {
            // Failed to place the item. This is fatal.
            throw DungeonGenerationFailed();
        }
    }   
}

bool DungeonGenerator::forbidden(int cat) const
{
    const map<int,StuffInfo>::const_iterator it = stuff.find(cat);
    if (it == stuff.end()) return false;
    return it->second.forbid;
}

int DungeonGenerator::findItemCategory(const DungeonMap &dmap, const MapCoord &mc,
                                        const vector<shared_ptr<Tile> > &tiles)
{
    int chosen_cat = -1;

    for (vector<shared_ptr<Tile> >::const_iterator it = tiles.begin();
    it != tiles.end(); ++it) {
        int cat = (*it)->getItemCategory();
        if (cat >=0) {
            chosen_cat = cat;
        }
        if ((*it)->canPlaceItem()) {
            // Don't generate an item if one has already been 'placed'
            if ((*it)->itemPlacedAlready()) return -1;
        } else {
            // For tiles that aren't explicitly accepting items, we go by the itemsAllowed()
            // flag, and generate items only where itemsAllowed()==true.
            // (Note that this doesn't work the same for 'placed' items, e.g. barrels have
            // itemsAllowed()==false but can still accept a placed item.)
            if ((*it)->itemsAllowed()==false) {
                return -1;
            }
        }
    }

    // Don't generate an item if one is already present
    if (dmap.getItem(mc)) return -1;

    return chosen_cat;
}

void DungeonGenerator::placeItem(DungeonMap &dmap, const MapCoord &mc,
                                 vector<shared_ptr<Tile> > &tiles, const ItemType &itype,
                                 int no)
{
    shared_ptr<Item> item(new Item(itype, no));
    bool placed = false;
    for (vector<shared_ptr<Tile> >::iterator it = tiles.begin(); it != tiles.end(); ++it) {
        if ((*it)->canPlaceItem() && !(*it)->itemPlacedAlready()) {
            (*it)->placeItem(item);
            placed = true;
            break;
        }
    }
    if (!placed) dmap.addItem(mc, item);
}

void DungeonGenerator::generateStuff(DungeonMap &dmap) 
{
    vector<shared_ptr<Tile> > tiles;
    for (int i=0; i<dmap.getWidth(); ++i) {
        for (int j=0; j<dmap.getHeight(); ++j) {
            MapCoord mc(i,j);

            // Find item-category associated with this tile. Non-negative value means we
            // should try to generate an item.
            dmap.getTiles(mc, tiles);
            const int chosen_cat = findItemCategory(dmap, mc, tiles);
            
            // Look for the chosen category in "stuff" (our container of ItemGenerators).
            if (chosen_cat >= 0) {
                map<int,StuffInfo>::const_iterator it = stuff.find(chosen_cat);
                if (it != stuff.end() && g_rng.getBool(it->second.chance/100.0f)) {
                    const ItemGenerator *generator = it->second.generator;
                    if (generator) {
                        // We are to generate an item
                        pair<const ItemType *, int> result = generator->get();
                        ASSERT(result.first);
                        placeItem(dmap, mc, tiles, *result.first, result.second);
                    }
                }
            }
        }
    }
}

void DungeonGenerator::generateLocksAndTraps(DungeonMap &dmap, int nkeys)
{
    vector<shared_ptr<Tile> > tiles;
    for (int i=0; i<dmap.getWidth(); ++i) {
        for (int j=0; j<dmap.getHeight(); ++j) {
            MapCoord mc(i,j);
            dmap.getTiles(mc, tiles);
            for (vector<shared_ptr<Tile> >::iterator it = tiles.begin(); it != tiles.end();
            ++it) {
                shared_ptr<Lockable> lockable = dynamic_pointer_cast<Lockable>(*it);
                if (lockable) {
                    // try to generate a trap. (only if pretrapped is on)
                    bool trap_placed = pretrapped ? lockable->generateTrap(dmap, mc) : false;
                    if (!trap_placed) {
                        // if that fails, then try to generate a lock instead.
                        lockable->generateLock(nkeys);
                    }
                }
            }
        }
    }
}


void DungeonGenerator::addVampireBats(DungeonMap &dmap, MonsterManager &mmgr,
                                      int nbats_normal, int nbats_guarded_exit)
{
    if (segments.empty()) return;  // dungeon was not generated yet!    
    ASSERT(segments.size() == lwidth*lheight);

    // First, place "regular" bats
    placeRegularBats(dmap, mmgr, nbats_normal);

    // Now place "guarded exit" bats.
    // First, we have to find the guarded exit segment
    for (int x=0; x<lwidth; ++x) {
        for (int y=0; y<lheight; ++y) {
            const Segment *seg = segments[y*lwidth+x];
            if (seg) {
                shared_ptr<Tile> bat_tile = seg->getBatPlacementTile();
                if (bat_tile) {
                    // Place "guarded" bats on this segment.
                    placeGuardedBats(dmap, mmgr, nbats_guarded_exit, bat_tile,
                                     x*(rwidth+1), y*(rwidth+1),
                                     (x+1)*(rwidth+1), (y+1)*(rwidth+1));
                }
            }
        }
    }
}

void DungeonGenerator::placeRegularBats(DungeonMap &dmap, MonsterManager &mmgr,
                                        int nbats_normal)
{
    vector<shared_ptr<Tile> > tiles;
    for (int i=0; i<nbats_normal; ++i) {
        for (int tries=0; tries<10; ++tries) {
            const int x = g_rng.getInt(0,dmap.getWidth());
            const int y = g_rng.getInt(0,dmap.getHeight());
            const MapCoord mc(x,y);

            // To place a regular bat, need a non-stair tile with clear access at H_FLYING.
            if (dmap.getAccess(mc, H_FLYING) != A_CLEAR) continue;
            dmap.getTiles(mc, tiles);
            bool ok = true;
            for (int i=0; i<tiles.size(); ++i) {
                if (tiles[i]->isStairOrTop()) {
                    ok = false;
                    break;
                }
            }
            if (!ok) continue;

            // Place the bat
            mmgr.placeVampireBat(dmap, mc);
            break;
        }
    }
}

void DungeonGenerator::placeGuardedBats(DungeonMap &dmap, MonsterManager &mmgr, int nbats,
                                        shared_ptr<Tile> bat_tile, int left, int bottom,
                                        int right, int top)
{
    vector<shared_ptr<Tile> > tiles;
    for (int i=0; i<nbats; ++i) {
        // Generate a random starting point for the search
        int x = g_rng.getInt(left, right);
        int y = g_rng.getInt(bottom, top);
        const int startx = x;
        const int starty = y;
        while (1) {
            // We look for a square that's clear at H_FLYING and has the bat_tile on it.
            const MapCoord mc(x,y);
            if (dmap.getAccess(mc, H_FLYING) == A_CLEAR) {
                dmap.getTiles(mc, tiles);
                if (find(tiles.begin(), tiles.end(), bat_tile) != tiles.end()) {
                    // Found it
                    // Place bat here.
                    mmgr.placeVampireBat(dmap, mc);
                    break;
                }
            }
            
            // Hmm. Have to go on to next square
            ++x;
            if (x == right) {
                // Hit end of row; go to start of next row
                x = left;
                ++y;
                if (y == top) {
                    // Went off the top. Start again from the bottom
                    y = bottom;
                }
            }

            if (x == startx && y == starty) {
                // We have come full circle without placing a bat. This means there are
                // no more bat placement squares available; abort.
                return;
            }
        }
    }
}
