/*
 * control_actions.hpp
 *
 * Actions that make a knight do something. These are typically used
 * for Controls.
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef CONTROL_ACTIONS_HPP
#define CONTROL_ACTIONS_HPP

#include "action.hpp"

//
// Append pointers to standard controls to the given vector.
// Creates the standard controls if necessary.
//
void GetStandardControls(std::vector<const Control*> &ctrls);


//
// Actions used by the standard controls
//

class A_Attack : public Action {
public:
    explicit A_Attack(MapDirection d) : dir(d) { }
    virtual bool canExecuteWhileMoving() const { return true; }
    virtual void execute(const ActionData &) const;
private:
    MapDirection dir;
};

class A_Move : public Action {
public:
    explicit A_Move(MapDirection d) : dir(d) { }
    virtual bool canExecuteWhileMoving() const { return true; }
    virtual void execute(const ActionData &) const;
private:
    MapDirection dir;
};

class A_Withdraw : public Action {
public:
    virtual void execute(const ActionData &) const;
};

class A_Suicide : public Action {
public:
    virtual bool canExecuteWhileStunned() const { return true; }
    virtual bool canExecuteWhileMoving() const { return true; }
    virtual void execute(const ActionData &) const;
};


//
// Other control-related actions.
//

class A_Drop : public Action {
public:
    explicit A_Drop(const ItemType &it_) : it(it_) { }
    virtual bool possible(const ActionData &) const;
    virtual void execute(const ActionData &) const;
private:
    const ItemType &it;
    ACTION_MAKER("Drop");
};

class A_DropHeld : public Action {
public:
    virtual bool possible(const ActionData &) const;
    virtual void execute(const ActionData &) const;
private:
    ACTION_MAKER("DropHeld");
};

class A_PickLock : public Action {
public:
    A_PickLock(int p, int wt) : prob(p), waiting_time(wt) { }
    virtual bool possible(const ActionData &) const;
    virtual void execute(const ActionData &) const;
private:
    int prob, waiting_time; // prob in %, waiting_time in ticks
    ACTION_MAKER("PickLock");
};

class A_PickUp : public Action {
public:
    virtual bool possible(const ActionData &) const;
    virtual void execute(const ActionData &) const;
private:
    ACTION_MAKER("PickUp");
};

class A_SetBearTrap : public Action {
public:
    explicit A_SetBearTrap(const ItemType &trap_) : trap_itype(trap_) { }
    virtual bool possible(const ActionData &) const;
    virtual void execute(const ActionData &) const;
private:
    const ItemType &trap_itype;
    ACTION_MAKER("SetBearTrap");
};

class A_SetBladeTrap : public Action {
public:
    explicit A_SetBladeTrap(const ItemType &missile) : missile_type(missile) { }
    virtual bool possible(const ActionData &) const;
    virtual void execute(const ActionData &) const;
private:
    const ItemType &missile_type;
    ACTION_MAKER("SetBladeTrap");
};

class A_SetPoisonTrap : public Action {
public:
    virtual bool possible(const ActionData &) const;
    virtual void execute(const ActionData &) const;
private:
    ACTION_MAKER("SetPoisonTrap");
};

class A_Swing : public Action {
public:
    virtual bool possible(const ActionData &) const;
    virtual bool canExecuteWhileMoving() const { return true; }
    virtual void execute(const ActionData &) const;
private:
    ACTION_MAKER("Swing");
};

// A_Throw is usually used for throwing daggers. It looks at the
// ActionData Item parameter to determine which item to throw.
class A_Throw : public Action {
public:
    virtual bool possible(const ActionData &) const;
    virtual bool canExecuteWhileMoving() const { return true; }
    virtual void execute(const ActionData &) const;
private:
    ACTION_MAKER("Throw");
};

#endif
