/*
 * control_actions.cpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */


// NOTE -- When writing new actions, don't forget to set stun_time if
// required (see DoStun(creature)).


#include "misc.hpp"

#include "concrete_traps.hpp"
#include "control.hpp"
#include "control_actions.hpp"
#include "creature.hpp"
#include "dungeon_map.hpp"
#include "item.hpp"
#include "knight.hpp"
#include "lockable.hpp"
#include "mediator.hpp"
#include "missile.hpp"
#include "rng.hpp"
#include "tile.hpp"

#include "boost/scoped_ptr.hpp"
#include "boost/thread/mutex.hpp"

// Undefine annoying windows macro that conflicts with my SC_MOVE constant
#ifdef SC_MOVE
#undef SC_MOVE
#endif

// Make sure the global ctors get linked (needed for MSVC)
#ifdef _MSC_VER
extern "C" void InitControls() { }
#endif

namespace {
    void DoStun(shared_ptr<Creature> actor, int gvt) 
    {
        actor->stunUntil(gvt + Mediator::instance().cfgInt("action_delay"));
    }

    shared_ptr<Knight> ExtractKnight(const ActionData &ad)
    {
        // Return actor (as knight), or NULL if no knight found or the knight is
        // not in a map.
        shared_ptr<Knight> actor = dynamic_pointer_cast<Knight>(ad.getActor());
        if (!actor || !actor->getMap()) return shared_ptr<Knight>();
        else return actor;
    }

    shared_ptr<Creature> ExtractCreature(const ActionData &ad)
    {
        // Same for Creature
        shared_ptr<Creature> actor = ad.getActor();
        if (!actor || !actor->getMap()) return shared_ptr<Creature>();
        else return actor;
    }

    const ItemType * ExtractItemType(const ActionData &ad)
    {
        DungeonMap *dummy;
        MapCoord dummy2;
        const ItemType *result;
        ad.getItem(dummy, dummy2, result);
        return result;
    }

    MapDirection GetPreferredDropDirection(const Entity &kt)
    {
        // If the kt is approaching, then we want to drop behind,
        // otherwise, drop in front.
        if (kt.isApproaching()) return Opposite(kt.getFacing());
        else return kt.getFacing();
    }

    void DoSwing(shared_ptr<Creature> actor)
    {
        if (!actor) return;
        if (actor->canSwing()) {
            actor->swing(); // no need to stun
        }
    }

    void DoThrow(shared_ptr<Creature> actor)
    {
        if (!actor) return;
        if (actor->canThrowItemInHand(true)) {
            actor->throwItemInHand();
        }
    }

    void DoShoot(shared_ptr<Creature> actor, int gvt)
    {
        // Shooting of crossbows is (somewhat anomalously) handled here rather than
        // as a Knight function.
        if (!actor) return;
        if (actor->isApproaching()) return;
        const ItemType *itype = actor->getItemInHand();
        if (itype && itype->canShoot() && itype->getAmmo()) {
            // create the missile
            DungeonMap *dmap = actor->getMap();
            if (!dmap) return;

            // note: we don't care if CreateMissile was successful or not...
            CreateMissile(*dmap, actor->getPos(), actor->getFacing(),
                          *itype->getAmmo(), false, actor->hasStrength());

            // unload the crossbow, and stun
            actor->setItemInHand(itype->getUnloaded());
            actor->stunUntil(gvt + Mediator::instance().cfgInt("crossbow_delay"));
            
            // run event hook
            Mediator::instance().runHook("HOOK_SHOOT", actor);
        }
    }
}

//
// GetStandardControls
//

namespace {
    // The standard controls are created once and shared between different games.
    boost::mutex g_std_ctrl_mutex;
    std::vector<boost::shared_ptr<Control> > g_standard_controls;
    boost::scoped_ptr<Action> g_attack[4], g_move[4], g_withdraw, g_suicide;
}

void GetStandardControls(std::vector<const Control*> &controls)
{
    boost::lock_guard<boost::mutex> lock(g_std_ctrl_mutex);
    
    if (g_standard_controls.empty())
    {
        // create the actions for the standard controls
        for (int i = 0; i < 4; ++i) {
            g_attack[i].reset(new A_Attack(MapDirection(i)));
            g_move[i].reset(new A_Move(MapDirection(i)));
        }
        g_withdraw.reset(new A_Withdraw);
        g_suicide.reset(new A_Suicide);
        
        // create the standard controls
        g_standard_controls.resize(NUM_STANDARD_CONTROLS);
        boost::shared_ptr<Control> ctrl;
        for (int i=0; i<4; ++i) {
            ctrl.reset(new Control(SC_ATTACK+i+1, 0, D_NORTH, 0, true, 0, g_attack[i].get()));
            g_standard_controls[SC_ATTACK+i] = ctrl;
            ctrl.reset(new Control(SC_MOVE+i+1, 0, D_NORTH, 0, true, 0, g_move[i].get()));
            g_standard_controls[SC_MOVE+i] = ctrl;
        }
        
        ctrl.reset(new Control(SC_WITHDRAW+1, 0, D_NORTH, 0, true, 0, g_withdraw.get()));
        g_standard_controls[SC_WITHDRAW] = ctrl;

        // Suicide is set up as a continuous control -- really it should be discontinuous, but
        // doing it this way prevents repeat suicide commands being sent by the client.
        
        ctrl.reset(new Control(SC_SUICIDE+1, 0, D_NORTH, 0, true, 0, g_suicide.get()));
        g_standard_controls[SC_SUICIDE] = ctrl;
    }

    controls.clear();
    for (std::vector<boost::shared_ptr<Control> >::const_iterator it = g_standard_controls.begin();
         it != g_standard_controls.end(); ++it) {
        controls.push_back(it->get());
    }
}


//
// A_Attack
//

void A_Attack::execute(const ActionData &ad) const
{
    shared_ptr<Creature> actor = ExtractCreature(ad);
    const int gvt = Mediator::instance().getGVT();

    if (!actor) return;
    if (actor->isApproaching()) return;

    if (actor->getFacing() != dir) {
        actor->setFacing(dir);
        DoStun(actor, gvt);
    } else {
        bool can_swing = actor->canSwing();
        bool can_throw = actor->canThrowItemInHand(true);
        bool can_shoot = actor->getItemInHand() && actor->getItemInHand()->canShoot();
        bool missile_attack = can_throw || can_shoot;

        if (can_swing && missile_attack) {
            // Both melee and missile attacks are possible.
            // If a creature is on my square or the square ahead,
            // or a targettable tile is on square ahead,
            // then do former, else do latter.

            // Find creature on my square or square_ahead:
            shared_ptr<Creature> target_cr = actor->getMap()->getTargetCreature(*actor, true);
            if (target_cr) {
                // Melee attack possible -- Cancel the missile option
                can_throw = can_shoot = missile_attack = false;
            }
            if (missile_attack) {
                // Now check for targettable tiles on square ahead:
                // (use getDestinationPos because we can do this while moving & we want sq at the end of the move)
                const MapCoord sq_ahead = DisplaceCoord(actor->getDestinationPos(), actor->getFacing());
                vector<shared_ptr<Tile> > tiles;
                actor->getMap()->getTiles(sq_ahead, tiles);
                for (vector<shared_ptr<Tile> >::iterator it = tiles.begin();
                it != tiles.end(); ++it) {
                    if ((*it)->targettable()) {
                        // Melee attack possible -- Cancel the missile option
                        can_throw = can_shoot = missile_attack = false;
                        break;
                    }
                }

                // If we would not be able to place a missile on the sq ahead either, then we
                // shouldn't allow the missile option. (This has the effect that if you are
                // two squares away from a wall - ie there is one open square between you and
                // the wall - then you will swing rather than throw.)
                if (!actor->getMap()->canPlaceMissile(sq_ahead, actor->getFacing())) {
                    can_throw = can_shoot = missile_attack = false;
                }

                if (missile_attack) {
                    // If we get here, then melee attack is not possible.
                    // Cancel the melee option.
                    can_swing = false;
                }
            }
        }

        if (can_swing) {
            DoSwing(actor); // no need to stun
        } else if (can_throw) {
            DoThrow(actor); // no need to stun
        } else if (can_shoot) {
            DoShoot(actor, gvt); // no need to stun
        } else if (actor->getMap() && actor->getItemInHand()
                   && !actor->getItemInHand()->canSwing()      // .. don't drop melee weapons
                   && !actor->getItemInHand()->canShoot()) {   // .. don't drop loaded xbows (but drop unloaded ones)
            // The reason we can't swing is because we are holding a non-weapon item.
            // Try dropping it. (E.g. if a player holding a book tries to attack, then we 
            // want to automatically drop the book.)
            ActionData data;
            data.setActor(actor, true);
            A_DropHeld action;
            action.execute(data);
        }
    }
}



//
// A_Move
//

void A_Move::execute(const ActionData &ad) const
{
    shared_ptr<Creature> actor = ExtractCreature(ad);
    Mediator &mediator = Mediator::instance();
    const int gvt = mediator.getGVT();
    
    if (!actor) return;
    if (actor->isApproaching()) return;  // can't move if already approached (or approaching)

    if (!actor->isMoving()) {
        // Normal Motion Cmd
        if (actor->getFacing() != dir) {
            actor->setFacing(dir);
            int turn_delay = mediator.cfgInt("turn_delay");
            if (actor->hasQuickness()) turn_delay = (turn_delay * 2) / 3;
            actor->stunUntil(gvt + turn_delay);
        } else {
            MapCoord dest = DisplaceCoord(actor->getPos(), dir, 1);
            MapAccess dest_access = actor->getMap()->getAccess(dest, actor->getHeight());
            if (dest_access == A_CLEAR) {
                actor->move(MT_MOVE);
            } else if (dest_access == A_APPROACH) {
                actor->move(MT_APPROACH);
            }
        }
    } else if (actor->getOffset() <= mediator.cfgInt("walk_limit")) {
        // In this case we are in the middle of moving, and we are allowed to turn back,
        // but no other motion actions are allowed.
        if (dir == Opposite(actor->getFacing())) {
            actor->flipMotion();
        }
    }
}



//
// A_Withdraw
//

void A_Withdraw::execute(const ActionData &ad) const
{
    shared_ptr<Creature> actor = ExtractCreature(ad);
    if (!actor) return;
    if (!actor->isApproaching()) return;  // can only withdraw if approached!
    actor->move(MT_WITHDRAW);
    DoStun(actor, Mediator::instance().getGVT());
}



//
// A_Suicide
//

void A_Suicide::execute(const ActionData &ad) const
{
    shared_ptr<Creature> actor = ExtractCreature(ad);
    if (!actor) return;
    Mediator::instance().runHook("HOOK_KNIGHT_DAMAGE", actor);
    actor->onDeath(Creature::NORMAL_MODE);
    actor->rmFromMap();
}



//
// Drop
//

bool A_Drop::possible(const ActionData &ad) const
{
    shared_ptr<Knight> actor = ExtractKnight(ad);
    if (!actor) return false;

    if (actor->getNumCarried(it) == 0) return false;

    const MapCoord mc = actor->getTargettedPos();
    return CanDropItem(it, *actor->getMap(), mc,
                       GetPreferredDropDirection(*actor));
}

void A_Drop::execute(const ActionData &ad) const
{
    const int gvt = Mediator::instance().getGVT();
    shared_ptr<Knight> actor = ExtractKnight(ad);
    if (!actor) return;
    
    int no_to_drop = actor->getNumCarried(this->it);
    if (no_to_drop == 0) return;
    if (no_to_drop > this->it.getMaxStack()) no_to_drop = this->it.getMaxStack();
    
    const MapCoord mc = actor->getTargettedPos();

    shared_ptr<Item> item(new Item(this->it, no_to_drop));
    bool added_to_map = DropItem(item, *actor->getMap(), mc, false,
                                 GetPreferredDropDirection(*actor),
                                 actor);

    if (added_to_map) {
        actor->rmFromBackpack(this->it, no_to_drop);
    } else {
        int no_dropped = no_to_drop - item->getNumber();
        if (no_dropped > 0) {
            actor->rmFromBackpack(this->it, no_dropped);
            added_to_map = true;
        }
    }

    if (added_to_map) {
        // drop was successful
        DoStun(actor, gvt);
    }
}

A_Drop::Maker A_Drop::Maker::register_me;

Action * A_Drop::Maker::make(ActionPars &pars) const
{
    pars.require(1);
    const ItemType *it = pars.getItemType(0);
    if (it) return new A_Drop(*it);
    else return 0;
}


//
// DropHeld
//

bool A_DropHeld::possible(const ActionData &ad) const
{
    shared_ptr<Knight> actor = ExtractKnight(ad);
    if (!actor) return false;
    if (!actor->canDropHeld()) return false;

    const MapCoord mc = actor->getTargettedPos();

    // Rule: If actor is approaching then he can only drop the item into the
    // approached square. Otherwise, he can drop it onto any adjacent square.
    // (This is to prevent "accidental" drops when when e.g. approaching a
    // closed chest from behind.)
    if (actor->isApproaching()) {
        shared_ptr<Item> dummy;
        return CheckDropSquare(*actor->getMap(), mc, *actor->getItemInHand(), dummy);
    } else {
        return CanDropItem(*actor->getItemInHand(), *actor->getMap(), mc,
                           GetPreferredDropDirection(*actor));
    }
}

void A_DropHeld::execute(const ActionData &ad) const
{
    const int gvt = Mediator::instance().getGVT();
    shared_ptr<Knight> actor = ExtractKnight(ad);
    if (!actor) return;
    if (!actor->canDropHeld()) return;

    const MapCoord mc = actor->getTargettedPos();
    shared_ptr<Item> item(new Item(*actor->getItemInHand()));
    bool added_to_map = DropItem(item, *actor->getMap(), mc, false,
                                 GetPreferredDropDirection(*actor),
                                 actor);
    if (added_to_map) {
        actor->setItemInHand(0);
        DoStun(actor, gvt);
    }
}

A_DropHeld::Maker A_DropHeld::Maker::register_me;

Action * A_DropHeld::Maker::make(ActionPars &pars) const
{
    pars.require(0);
    return new A_DropHeld;
}


//
// PickLock
//

namespace {
    shared_ptr<Lockable> GetLockPickTarget(const ActionData &ad)
    {
        shared_ptr<Creature> actor = ExtractCreature(ad);
        if (!actor || !actor->getMap()) return shared_ptr<Lockable>();
        MapCoord mc = DisplaceCoord(actor->getPos(), actor->getFacing());
        vector<shared_ptr<Tile> > tiles;
        actor->getMap()->getTiles(mc, tiles);
        for (vector<shared_ptr<Tile> >::iterator it = tiles.begin(); it != tiles.end();
        ++it) {
            shared_ptr<Lockable> t = dynamic_pointer_cast<Lockable>(*it);
            if (t && t->isClosed() && t->isLocked() && !t->isSpecialLocked()) {
                return t;
            }
        }
        return shared_ptr<Lockable>();
    }
}

bool A_PickLock::possible(const ActionData &ad) const
{
    return GetLockPickTarget(ad);
}

void A_PickLock::execute(const ActionData &ad) const
{
    Mediator &mediator = Mediator::instance();
    const int gvt = mediator.getGVT();
    shared_ptr<Creature> cr = ExtractCreature(ad);
    if (!cr) return;
    shared_ptr<Lockable> t = GetLockPickTarget(ad);
    if (!t) return;
    if (g_rng.getBool(prob/100.0f)) {
        // lock pick was successful
        t->onOpen(*cr->getMap(), DisplaceCoord(cr->getPos(), cr->getFacing()), cr);
    }
    
    // compute stun time (lock picking is faster with quickness)
    int time = waiting_time;
    if (cr->hasQuickness()) time = time * 100 / mediator.cfgInt("quickness_factor"); 
    cr->stunUntil(gvt + time);
}

A_PickLock::Maker A_PickLock::Maker::register_me;

Action * A_PickLock::Maker::make(ActionPars &pars) const
{
    pars.require(2);
    int p = pars.getProbability(0);
    int wt = pars.getInt(1);
    return new A_PickLock(p, wt);
}

//
// PickUp
//

bool A_PickUp::possible(const ActionData &ad) const
{
    shared_ptr<Knight> actor = ExtractKnight(ad);
    if (!actor) return false;

    // find mc. We need a sort of combination of getTargettedPos and getDestinationPos
    MapCoord mc;
    if (actor->isApproaching()) mc = DisplaceCoord(actor->getPos(), actor->getFacing());
    else mc = actor->getDestinationPos();

    shared_ptr<Item> item = actor->getMap()->getItem(mc);

    if (!item) return false;
    if (!item->getType().canPickUp()) return false;
    
    if (item->getType().isBig()) return true;  // can always pick up a "big" item (by
                                               // dropping existing one if necessary)
    
    return actor->canAddToBackpack(item->getType());

    // TODO: "Magic" items are not handled ...
}

void A_PickUp::execute(const ActionData &ad) const
{
    const int gvt = Mediator::instance().getGVT();
    
    // find knight
    shared_ptr<Knight> actor = ExtractKnight(ad);
    if (!actor) return;

    // find item
    const MapCoord mc = actor->getTargettedPos();
    shared_ptr<Item> item = actor->getMap()->getItem(mc);

    if (!item) return;
    if (!item->getType().canPickUp()) return;

    bool picked_up = false;
    
    if (item->getType().isBig()) {

        // Big item -- may need to drop held item first
        shared_ptr<Item> drop;
        if (actor->canDropHeld()) {
            drop.reset(new Item( *actor->getItemInHand(), 1 ));
        }

        // Pick up item from the map
        actor->getMap()->rmItem(mc);
        actor->setItemInHand(&item->getType());

        // Drop replacement item into the map
        if (drop) {
            actor->getMap()->addItem(mc, drop);
            // Run the "onDrop" event manually (since we're not using DropItem):
            drop->getType().onDrop(*actor->getMap(), mc, actor);
        }

        picked_up = true;

    } else {
        // Small item -- add to backpack
        int amt;
        if (item->getType().isMagic()) {
            // Magic items should be "fully picked up" and not added to backpack
            amt = item->getNumber();
        } else {
            // All other items should be added to the backpack.
            amt = actor->addToBackpack(item->getType(), item->getNumber());
        }
        if (amt > 0) {
            // Remove item from the ground
            if (amt < item->getNumber()) {
                item->setNumber(item->getNumber() - amt);
            } else {
                actor->getMap()->rmItem(mc);
            }
            picked_up = true;
        }
    }

    if (picked_up) {
        // Pickup event and Stun
        item->getType().onPickUp(*actor->getMap(), mc, actor);
        DoStun(actor, gvt);
    }
}

A_PickUp::Maker A_PickUp::Maker::register_me;

Action * A_PickUp::Maker::make(ActionPars &pars) const
{
    pars.require(0);
    return new A_PickUp;
}

//
// SetBearTrap
//

bool A_SetBearTrap::possible(const ActionData &ad) const
{
    shared_ptr<Knight> actor = ExtractKnight(ad);
    if (!actor) return false;

    // Knight must not be approaching.
    if (actor->isApproaching()) return false;

    // We must be able to drop the trap-item in the square ahead.
    MapCoord mc = DisplaceCoord(actor->getPos(), actor->getFacing(), 1);    
    shared_ptr<Item> dummy;
    if (!CheckDropSquare(*actor->getMap(), mc, trap_itype, dummy)) return false;

    return true;
}

void A_SetBearTrap::execute(const ActionData &ad) const
{
    const int gvt = Mediator::instance().getGVT();
    
    shared_ptr<Knight> actor = ExtractKnight(ad);
    if (!actor) return;
    if (actor->isApproaching()) return;
    MapCoord mc = DisplaceCoord(actor->getPos(), actor->getFacing(), 1);

    shared_ptr<Item> curr_item;
    if (!CheckDropSquare(*actor->getMap(), mc, trap_itype, curr_item)) return;
    if (curr_item) return; // bear traps shouldn't be able to stack.

    shared_ptr<Item> trap_item(new Item(trap_itype, 1));
    actor->getMap()->addItem(mc, trap_item);
    actor->setItemInHand(0);
    DoStun(actor, gvt);
}

A_SetBearTrap::Maker A_SetBearTrap::Maker::register_me;

Action * A_SetBearTrap::Maker::make(ActionPars &pars) const
{
    pars.require(1);
    const ItemType *it = pars.getItemType(0);
    if (it) return new A_SetBearTrap(*it);
    else return 0;
}


// NB there is a little bit of shared code between SetPoisonTrap and SetBladeTrap.
// If more trap types are added it may be worth factoring this out.

namespace {
    shared_ptr<Lockable> CheckSetTrap(const ActionData &ad)
    {
        shared_ptr<Creature> actor = ExtractCreature(ad);

        // Normally we require the actor to be approaching the door or chest,
        // but if FLAG is set we skip this check. The flag is only ever set
        // by the pretrapped chests code. This is an evil hack and should probably
        // be removed somehow...
        bool approach_check = ad.getFlag() || actor->isApproaching();
        
        if (!actor || !approach_check) return shared_ptr<Lockable>();
        MapCoord mc = DisplaceCoord(actor->getPos(), actor->getFacing());
        vector<shared_ptr<Tile> > tiles;
        actor->getMap()->getTiles(mc, tiles);
        for (vector<shared_ptr<Tile> >::iterator it = tiles.begin(); it != tiles.end();
        ++it) {
            shared_ptr<Lockable> lck = dynamic_pointer_cast<Lockable>(*it);
            if (lck && lck->isClosed() && !lck->isLocked() && lck->canTrap()) {
                return lck;
            }
        }
        return shared_ptr<Lockable>();
    }

    void RemoveItem(shared_ptr<Creature> cr, const ItemType *it)
    {
        shared_ptr<Knight> kt = dynamic_pointer_cast<Knight>(cr);
        if (kt && it) kt->rmFromBackpack(*it, 1);
    }
}

bool A_SetBladeTrap::possible(const ActionData &ad) const
{
    return bool(CheckSetTrap(ad));
}

void A_SetBladeTrap::execute(const ActionData &ad) const
{
    shared_ptr<Lockable> target = CheckSetTrap(ad);
    shared_ptr<Creature> actor = ExtractCreature(ad);
    if (target && actor) {
        const ItemType *it = ExtractItemType(ad);
        shared_ptr<Trap> trap(new BladeTrap(it, missile_type, Opposite(actor->getFacing())));
        target->setTrap(*actor->getMap(), actor->getPos(), actor, trap);
        RemoveItem(actor, it);
    }
}

A_SetBladeTrap::Maker A_SetBladeTrap::Maker::register_me;

Action * A_SetBladeTrap::Maker::make(ActionPars &pars) const
{
    pars.require(1);
    const ItemType *it = pars.getItemType(0);
    if (it) return new A_SetBladeTrap(*it);
    else return 0;
}


bool A_SetPoisonTrap::possible(const ActionData &ad) const
{
    return bool(CheckSetTrap(ad));
}

void A_SetPoisonTrap::execute(const ActionData &ad) const
{
    shared_ptr<Lockable> target = CheckSetTrap(ad);
    shared_ptr<Creature> actor = ExtractCreature(ad);
    if (target && actor) {
        const ItemType *it = ExtractItemType(ad);
        shared_ptr<Trap> trap(new PoisonTrap(it));
        target->setTrap(*actor->getMap(), actor->getPos(), actor, trap);
        RemoveItem(actor, it);
    }
}

A_SetPoisonTrap::Maker A_SetPoisonTrap::Maker::register_me;

Action * A_SetPoisonTrap::Maker::make(ActionPars &pars) const
{
    pars.require(0);
    return new A_SetPoisonTrap;
}


//
// Swing
//

bool A_Swing::possible(const ActionData &ad) const
{
    // This action is only possible while not approaching
    shared_ptr<Creature> actor = ExtractCreature(ad);
    if (!actor || !actor->getMap()) return false;
    if (actor->isApproaching()) return false;
    if (!actor->canSwing()) return false;
    return true;
}

void A_Swing::execute(const ActionData &ad) const
{
    shared_ptr<Creature> actor = ExtractCreature(ad);
    if (!actor || !actor->canSwing()) return;
    DoSwing(actor);  // no need to stun
}

A_Swing::Maker A_Swing::Maker::register_me;

Action * A_Swing::Maker::make(ActionPars &pars) const
{
    pars.require(0);
    return new A_Swing;
}


//
// Throw
//

bool A_Throw::possible(const ActionData &ad) const
{
    // This can only be used on Knights because it requires throwItem()
    // instead of just throww().
    shared_ptr<Knight> actor = ExtractKnight(ad);
    if (!actor || !actor->getMap()) return false;

    DungeonMap *dum;
    MapCoord dum2;
    const ItemType *itype;
    ad.getItem(dum, dum2, itype);
    if (!itype) return false;

    // Strict set to false - see comments in creature.hpp
    return actor->canThrowItem(*itype, false);
}

void A_Throw::execute(const ActionData &ad) const
{
    shared_ptr<Knight> actor = ExtractKnight(ad);
    if (!actor) return;
    
    DungeonMap *dum;
    MapCoord dum2;
    const ItemType *itype;
    ad.getItem(dum, dum2, itype);
    if (!itype) return;

    actor->throwItem(*itype);
}

A_Throw::Maker A_Throw::Maker::register_me;

Action * A_Throw::Maker::make(ActionPars &pars) const
{
    pars.require(0);
    return new A_Throw;
}
