/*
 * action.cpp
 *
 * Copyright (c) Stephen Thompson, 2008.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "action.hpp"
#include "creature.hpp"
#include "rng.hpp"

#include "boost/thread/mutex.hpp"

void ActionData::setItem(DungeonMap *dmap, const MapCoord &mc, const ItemType * it)
{
    // Can't set the item location (dmap & mc) without setting an item
    // (although an item without a location IS allowed). Also, can't
    // set mc without setting dmap & vice versa.
    if (it && dmap && !mc.isNull()) {
        item_dmap = dmap;
        item_coord = mc;
    } else {
        item_dmap = 0;
        item_coord = MapCoord();
    }
    item = it;
}

void ActionData::setTile(DungeonMap *dmap, const MapCoord &mc, shared_ptr<Tile> t)
{
    // Can't set only one of dmap & mc, they must be set together or
    // not at all. Also, while we can set a dmap & mc with no Tile, we
    // must not set a Tile with no dmap & mc.
    if (dmap && !mc.isNull()) {
        tile_dmap = dmap;
        tile_coord = mc;
        tile = t;
    } else {
        tile_dmap = 0;
        tile_coord = MapCoord();
        tile = shared_ptr<Tile>();
    }
}

pair<DungeonMap *, MapCoord> ActionData::getPos() const
{
    DungeonMap *dmap;
    MapCoord mc;
    
    if (actor && actor->getMap()) dmap = actor->getMap();
    else if (victim && victim->getMap()) dmap = victim->getMap();
    else if (item_dmap) dmap = item_dmap;
    else if (tile_dmap) dmap = tile_dmap;
    else dmap = 0;

    if (dmap) {
        if (actor && actor->getMap()) mc = actor->getPos();
        else if (victim && victim->getMap()) mc = victim->getPos();
        else if (!item_coord.isNull()) mc = item_coord;
        else if (!tile_coord.isNull()) mc = tile_coord;
    }

    return make_pair(dmap, mc);
}

void RandomAction::add(const Action *ac, int wt)
{
    if (wt > 0) {
        data.push_back(make_pair(ac, wt));
        total_weight += wt;
    }
}

void RandomAction::execute(const ActionData &ad) const
{
    if (total_weight > 0) {
        int r = g_rng.getInt(0, total_weight);
        for (int i=0; i<data.size(); ++i) {
            r -= data[i].second;
            if (r < 0) {
                if (data[i].first) {
                    data[i].first->execute(ad);
                }
                return;
            }
        }
        ASSERT(0);
    }
}

bool ListAction::possible(const ActionData &ad) const
{
    for (int i=0; i<data.size(); ++i) {
        if (!data[i]->possible(ad)) return false;
    }
    return true;
}

void ListAction::execute(const ActionData &a) const
{
    ActionData ad(a);
    for (int i=0; i<data.size(); ++i) {
        // Note: we update the "success" flag after each action in the sequence.
        // This is done based on the possible() flag.
        const bool successful = data[i]->possible(ad);
        data[i]->execute(ad);
        ad.setSuccess(successful);
    }
}


//
// ActionPars
//

MapDirection ActionPars::getMapDirection(int index)
{
    string d = getString(index);
    for (string::iterator it = d.begin(); it != d.end(); ++it) { 
        *it = toupper(*it);
    }
    if (d == "NORTH") return D_NORTH;
    else if (d == "EAST") return D_EAST;
    else if (d == "SOUTH") return D_SOUTH;
    else if (d == "WEST") return D_WEST;
    error();
    return D_NORTH;
}


//
// ActionMaker
//

boost::mutex g_makers_mutex;

map<string, const ActionMaker*> & MakersMap()
{
    static boost::shared_ptr<map<string, const ActionMaker*> > p(new map<string, const ActionMaker*>);
    return *p;
}

ActionMaker::ActionMaker(const string &name)
{
    // this is called before main() starts (in a single threaded way)
    // so don't need to lock the mutex. Indeed 'g_makers_mutex' may not
    // have been constructed yet so it would not be safe to do so.
    MakersMap()[name] = this;
}

Action * ActionMaker::createAction(const string &name, ActionPars &pars)
{
    const ActionMaker * maker = 0;
    {
        // We lock 'g_makers_mutex' in case std::map does not support
        // multiple concurrent readers.
        boost::lock_guard<boost::mutex> lock(g_makers_mutex);
        map<string, const ActionMaker *> & makers = MakersMap();
        map<string, const ActionMaker *>::iterator it = makers.find(name);
        if (it != makers.end()) maker = it->second;
    }
    if (maker) return maker->make(pars);
    else return 0;
}
