/*
 * FILE:
 *   sdl_sound_driver.cpp
 *
 * AUTHOR:
 *   Stephen Thompson, 2008
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *
 */

#include "sdl_sound_driver.hpp"
#include "../core/sdl_error.hpp"
#include "../core/istream_rwops.hpp"
#include "../../sound/sound.hpp"

#include "boost/noncopyable.hpp"

namespace Coercri {

    namespace {
        bool g_sound_open;   // Can only open the audio device once!
        
        void FlipEndian(Uint8 *data, int nbytes)
        {
            while (nbytes > 0) {
                std::swap(*data, *(data+1));
                nbytes -= 2;
            }
        }
    }

    struct SDLSound : Sound {
        Uint16 *data;
        Uint32 len;
        SDLSound(Uint16 *d, Uint32 l) : data(d), len(l) { }
        ~SDLSound() { SDL_FreeWAV(reinterpret_cast<Uint8*>(data)); }
    };
    
    struct Voice {
        float pos;
        boost::shared_ptr<SDLSound> sound;
        int freq;
    };

    SDLSoundDriver::SDLSoundDriver(float vf)
        : audio_subsystem_handle(SDL_INIT_AUDIO), volume_factor(vf)
    {
        if (g_sound_open) {
            throw CoercriError("Sound already open");
        }

        SDL_AudioSpec desired;
        memset(&desired, 0, sizeof(desired));
        memset(&spec, 0, sizeof(spec));
        desired.freq = 22050;
        desired.format = AUDIO_S16SYS;
        desired.channels = 1;  // We don't support stereo at the moment
        desired.samples = 1024;
        desired.callback = audioCallback;
        desired.userdata = this;

        const int err = SDL_OpenAudio(&desired, &spec);
        if (err < 0) {
            throw SDLError("Couldn't open audio");
        }

        SDL_PauseAudio(0);
        g_sound_open = true;
    }

    SDLSoundDriver::~SDLSoundDriver()
    {
        SDL_PauseAudio(1);
        SDL_CloseAudio();
        g_sound_open = false;
    }
    
    boost::shared_ptr<Sound> SDLSoundDriver::loadSound(boost::shared_ptr<std::istream> str)
    {
        SDL_AudioSpec spec;
        memset(&spec, 0, sizeof(spec));

        Uint8 *data = 0;
        Uint32 len = 0;

        SDL_RWops * rwops = CreateRWOpsForIstream(str);
        SDL_LoadWAV_RW(rwops, 1, &spec, &data, &len);

        if (!data) {
            throw SDLError("Failed to load sound file");
        }
        if (spec.format != AUDIO_S16LSB && spec.format != AUDIO_S16MSB) {
            throw CoercriError("Unsupported sound format");
        }
        if (spec.format != AUDIO_S16SYS) {
            FlipEndian(data, len);
        }

        boost::shared_ptr<Sound> sound(new SDLSound(reinterpret_cast<Uint16*>(data), len/2));
        return sound;
    }

    void SDLSoundDriver::playSound(boost::shared_ptr<Sound> sound, int frequency)
    {
        boost::shared_ptr<SDLSound> sdl_sound = boost::dynamic_pointer_cast<SDLSound>(sound);
        if (sdl_sound) {
            // Add a new voice playing this sample
            Voice v;
            v.pos = 0;
            v.sound = sdl_sound;
            v.freq = frequency;
            SDL_LockAudio();
            voices.push_front(v);
            SDL_UnlockAudio();
        }
    }

    void SDLSoundDriver::audioCallback(void *userdata, Uint8 *stream, int len)
    {
        SDLSoundDriver* snd_drv = static_cast<SDLSoundDriver*>(userdata);
        Sint16 *buf = reinterpret_cast<Sint16*>(stream);
        
        // Zero out the buffer
        memset(buf, 0, len/2);
        
        // For each voice currently playing
        for (std::list<Voice>::iterator it = snd_drv->voices.begin(); it != snd_drv->voices.end(); ) {
            bool erase = false;
            // For each data point
            for (int n = 0; n < len/2; ++n) {
                if (it->pos >= (it->sound->len - 2)) {
                    erase = true;
                    break;
                } else {
                    // Linearly interpolate
                    const int p1 = int(it->pos);
                    const float lambda = it->pos - p1;
                    const Sint16 x1 = it->sound->data[p1];
                    const Sint16 x2 = it->sound->data[p1 + 1];
                    const float interp = (1-lambda)*x1 + lambda*x2;

                    const Sint32 new_val = buf[n] + Sint16(snd_drv->volume_factor * interp);
                    if (new_val >= 32767) {
                        buf[n] = 32767;
                    } else if (new_val <= -32768) {
                        buf[n] = -32768;
                    } else {
                        buf[n] = new_val;
                    }
                        
                    it->pos += (float(it->freq) / snd_drv->spec.freq);
                }
            }
            if (erase) {
                std::list<Voice>::iterator it2 = it++;
                snd_drv->voices.erase(it2);
            } else {
                ++it;
            }
        }
    }
}
