/*
 * FILE:
 *   sdl_ttf_font.cpp
 *
 * AUTHOR:
 *   Stephen Thompson, 2008
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *
 *   Note that to use this file in a working program you will need to
 *   link it to the SDL_TTF library which is LGPL.
 *
 */

#include "../../core/coercri_error.hpp"
#include "../core/istream_rwops.hpp"
#include "delete_sdl_surface.hpp"
#include "sdl_gfx_context.hpp"
#include "sdl_ttf_font.hpp"

namespace Coercri {

    namespace {
        // Deleter for TTF_Font
        struct DeleteTTFFont {
            void operator()(TTF_Font *font) {
                TTF_CloseFont(font);
            }
        };
    }

    SDLTTFFont::SDLTTFFont(boost::shared_ptr<std::istream> str, int size)
    {
        // Initialize SDL_TTF if we haven't already
        if (!TTF_WasInit() && TTF_Init()==-1) {
            throw CoercriError("Failed to initialize SDL_TTF");
        }

        // Open the font
        SDL_RWops * rwops = CreateRWOpsForIstream(str);
        font.reset(TTF_OpenFontRW(rwops, true, size), DeleteTTFFont());
        if (!font) {
            throw CoercriError(std::string("Failed to open font. SDL error was: ") + TTF_GetError());
        }
    }

    void SDLTTFFont::drawText(GfxContext &cxt, int x, int y, const std::string &text, Color col, bool antialias) const
    {
        // TODO: transparency support
        if (col.a != 255) {
            throw CoercriError("Coercri::SDLTTFFont::drawText: transparency not yet implemented");
        }
        
        // Get the surface from the gfx context (must be an SDL gfx context)
        SDLGfxContext *sdl_gc = dynamic_cast<SDLGfxContext*>(&cxt);
        if (!sdl_gc) return;
        sdl_gc->unlock();
        SDL_Surface *dest = sdl_gc->getSurface();

        // Don't try to draw an empty string (TTF_RenderText would give an error if we did).
        if (text.empty()) {
            return;
        }

        // Render the text
        SDL_Color sdl_col = {col.r, col.g, col.b};
        boost::shared_ptr<SDL_Surface> text_surface;
        if (antialias) {
            text_surface.reset(TTF_RenderText_Blended(font.get(), text.c_str(), sdl_col), DeleteSDLSurface());
        } else {
            text_surface.reset(TTF_RenderText_Solid(font.get(), text.c_str(), sdl_col), DeleteSDLSurface());
        }

        if (text_surface) {
            SDL_Rect dest_rect;
            dest_rect.x = x;
            dest_rect.y = y;
            SDL_BlitSurface(text_surface.get(), 0, dest, &dest_rect);
        }
    }

    void SDLTTFFont::getTextSize(const std::string &text, int &w, int &h) const
    {
        const int err = TTF_SizeText(font.get(), text.c_str(), &w, &h);
        if (err) w = h = 0;
    }

    int SDLTTFFont::getTextHeight() const
    {
        return std::max(TTF_FontLineSkip(font.get()), TTF_FontHeight(font.get()));
    }
}
