/*
 * FILE:
 *   sdl_gfx_driver.cpp
 *
 * AUTHOR:
 *   Stephen Thompson, 2008
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *
 */

#include "sdl_gfx_context.hpp"
#include "sdl_gfx_driver.hpp"
#include "sdl_graphic.hpp"
#include "sdl_window.hpp"
#include "../../core/coercri_error.hpp"
#include "../../gfx/mouse_button.hpp"
#include "../../gfx/rectangle.hpp"
#include "../../gfx/region.hpp"
#include "../../gfx/window_listener.hpp"

#include "SDL.h"
#include "boost/weak_ptr.hpp"
#include <map>

#ifdef WIN32
#include "SDL_syswm.h"
#include <windows.h>
#ifdef MB_RIGHT
#undef MB_RIGHT
#endif
#endif

namespace Coercri {

    // SDL only supports one window at a time - we store a global
    // pointer to this Window here.
    boost::weak_ptr<SDLWindow> g_sdl_window;
    unsigned int g_sdl_required_flags;
    bool g_sdl_has_focus;

    namespace {

        // Global key tables
        std::map<SDLKey, KeyCode> g_keytable;
        std::map<SDLKey, bool> g_keystate;  // true if key down, false if up.

        void InitKeyTable()
        {
            if (!g_keytable.empty()) return;  // already initialized

            g_keytable[SDLK_BACKSPACE] = KC_BACKSPACE;
            g_keytable[SDLK_TAB] = KC_TAB;
            g_keytable[SDLK_CLEAR] = KC_CLEAR;
            g_keytable[SDLK_RETURN] = KC_RETURN;
            g_keytable[SDLK_PAUSE] = KC_PAUSE;
            g_keytable[SDLK_ESCAPE] = KC_ESCAPE;
            g_keytable[SDLK_SPACE] = KC_SPACE;
            g_keytable[SDLK_EXCLAIM] = KC_EXCLAIM;
            g_keytable[SDLK_QUOTEDBL] = KC_DOUBLE_QUOTE;
            g_keytable[SDLK_HASH] = KC_HASH;
            g_keytable[SDLK_DOLLAR] = KC_DOLLAR;
            g_keytable[SDLK_AMPERSAND] = KC_AMPERSAND;
            g_keytable[SDLK_QUOTE] = KC_SINGLE_QUOTE;
            g_keytable[SDLK_LEFTPAREN] = KC_LEFT_PAREN;
            g_keytable[SDLK_RIGHTPAREN] = KC_RIGHT_PAREN;
            g_keytable[SDLK_ASTERISK] = KC_ASTERISK;
            g_keytable[SDLK_PLUS] = KC_PLUS;
            g_keytable[SDLK_COMMA] = KC_COMMA;
            g_keytable[SDLK_MINUS] = KC_MINUS;
            g_keytable[SDLK_PERIOD] = KC_PERIOD;
            g_keytable[SDLK_SLASH] = KC_SLASH;
            g_keytable[SDLK_0] = KC_0;
            g_keytable[SDLK_1] = KC_1;
            g_keytable[SDLK_2] = KC_2;
            g_keytable[SDLK_3] = KC_3;
            g_keytable[SDLK_4] = KC_4;
            g_keytable[SDLK_5] = KC_5;
            g_keytable[SDLK_6] = KC_6;
            g_keytable[SDLK_7] = KC_7;
            g_keytable[SDLK_8] = KC_8;
            g_keytable[SDLK_9] = KC_9;
            g_keytable[SDLK_COLON] = KC_COLON;
            g_keytable[SDLK_SEMICOLON] = KC_SEMICOLON;
            g_keytable[SDLK_LESS] = KC_LESS;
            g_keytable[SDLK_EQUALS] = KC_EQUALS;
            g_keytable[SDLK_GREATER] = KC_GREATER;
            g_keytable[SDLK_QUESTION] = KC_QUESTION;
            g_keytable[SDLK_AT] = KC_AT;
            g_keytable[SDLK_LEFTBRACKET] = KC_LEFT_BRACKET;
            g_keytable[SDLK_BACKSLASH] = KC_BACKSLASH;
            g_keytable[SDLK_RIGHTBRACKET] = KC_RIGHT_BRACKET;
            g_keytable[SDLK_CARET] = KC_CARET;
            g_keytable[SDLK_UNDERSCORE] = KC_UNDERSCORE;
            g_keytable[SDLK_BACKQUOTE] = KC_BACKQUOTE;
            g_keytable[SDLK_a] = KC_A;
            g_keytable[SDLK_b] = KC_B;
            g_keytable[SDLK_c] = KC_C;
            g_keytable[SDLK_d] = KC_D;
            g_keytable[SDLK_e] = KC_E;
            g_keytable[SDLK_f] = KC_F;
            g_keytable[SDLK_g] = KC_G;
            g_keytable[SDLK_h] = KC_H;
            g_keytable[SDLK_i] = KC_I;
            g_keytable[SDLK_j] = KC_J;
            g_keytable[SDLK_k] = KC_K;
            g_keytable[SDLK_l] = KC_L;
            g_keytable[SDLK_m] = KC_M;
            g_keytable[SDLK_n] = KC_N;
            g_keytable[SDLK_o] = KC_O;
            g_keytable[SDLK_p] = KC_P;
            g_keytable[SDLK_q] = KC_Q;
            g_keytable[SDLK_r] = KC_R;
            g_keytable[SDLK_s] = KC_S;
            g_keytable[SDLK_t] = KC_T;
            g_keytable[SDLK_u] = KC_U;
            g_keytable[SDLK_v] = KC_V;
            g_keytable[SDLK_w] = KC_W;
            g_keytable[SDLK_x] = KC_X;
            g_keytable[SDLK_y] = KC_Y;
            g_keytable[SDLK_z] = KC_Z;
            g_keytable[SDLK_DELETE] = KC_DELETE;
            g_keytable[SDLK_KP0] = KC_KP_0;
            g_keytable[SDLK_KP1] = KC_KP_1;
            g_keytable[SDLK_KP2] = KC_KP_2;
            g_keytable[SDLK_KP3] = KC_KP_3;
            g_keytable[SDLK_KP4] = KC_KP_4;
            g_keytable[SDLK_KP5] = KC_KP_5;
            g_keytable[SDLK_KP6] = KC_KP_6;
            g_keytable[SDLK_KP7] = KC_KP_7;
            g_keytable[SDLK_KP8] = KC_KP_8;
            g_keytable[SDLK_KP9] = KC_KP_9;
            g_keytable[SDLK_KP_PERIOD] = KC_KP_PERIOD;
            g_keytable[SDLK_KP_DIVIDE] = KC_KP_DIVIDE;
            g_keytable[SDLK_KP_MULTIPLY] = KC_KP_MULTIPLY;
            g_keytable[SDLK_KP_MINUS] = KC_KP_MINUS;
            g_keytable[SDLK_KP_PLUS] = KC_KP_PLUS;
            g_keytable[SDLK_KP_ENTER] = KC_KP_ENTER;
            g_keytable[SDLK_KP_EQUALS] = KC_KP_EQUALS;
            g_keytable[SDLK_UP] = KC_UP;
            g_keytable[SDLK_DOWN] = KC_DOWN;
            g_keytable[SDLK_RIGHT] = KC_RIGHT;
            g_keytable[SDLK_LEFT] = KC_LEFT;
            g_keytable[SDLK_INSERT] = KC_INSERT;
            g_keytable[SDLK_HOME] = KC_HOME;
            g_keytable[SDLK_END] = KC_END;
            g_keytable[SDLK_PAGEUP] = KC_PAGE_UP;
            g_keytable[SDLK_PAGEDOWN] = KC_PAGE_DOWN;
            g_keytable[SDLK_F1] = KC_F1;
            g_keytable[SDLK_F2] = KC_F2;
            g_keytable[SDLK_F3] = KC_F3;
            g_keytable[SDLK_F4] = KC_F4;
            g_keytable[SDLK_F5] = KC_F5;
            g_keytable[SDLK_F6] = KC_F6;
            g_keytable[SDLK_F7] = KC_F7;
            g_keytable[SDLK_F8] = KC_F8;
            g_keytable[SDLK_F9] = KC_F9;
            g_keytable[SDLK_F10] = KC_F10;
            g_keytable[SDLK_F11] = KC_F11;
            g_keytable[SDLK_F12] = KC_F12;
            g_keytable[SDLK_F13] = KC_F13;
            g_keytable[SDLK_F14] = KC_F14;
            g_keytable[SDLK_F15] = KC_F15;
            g_keytable[SDLK_NUMLOCK] = KC_NUM_LOCK;
            g_keytable[SDLK_CAPSLOCK] = KC_CAPS_LOCK;
            g_keytable[SDLK_SCROLLOCK] = KC_SCROLL_LOCK;
            g_keytable[SDLK_RSHIFT] = KC_RIGHT_SHIFT;
            g_keytable[SDLK_LSHIFT] = KC_LEFT_SHIFT;
            g_keytable[SDLK_RCTRL] = KC_RIGHT_CONTROL;
            g_keytable[SDLK_LCTRL] = KC_LEFT_CONTROL;
            g_keytable[SDLK_RALT] = KC_RIGHT_ALT;            
            g_keytable[SDLK_LALT] = KC_LEFT_ALT;
            g_keytable[SDLK_RMETA] = KC_RIGHT_META;
            g_keytable[SDLK_LMETA] = KC_LEFT_META;
            g_keytable[SDLK_LSUPER] = KC_LEFT_SUPER;
            g_keytable[SDLK_RSUPER] = KC_RIGHT_SUPER;
            g_keytable[SDLK_MODE] = KC_MODE;
            g_keytable[SDLK_COMPOSE] = KC_COMPOSE;
            g_keytable[SDLK_HELP] = KC_HELP;
            g_keytable[SDLK_PRINT] = KC_PRINT;
            g_keytable[SDLK_SYSREQ] = KC_SYSREQ;
            g_keytable[SDLK_BREAK] = KC_BREAK;
            g_keytable[SDLK_MENU] = KC_MENU;
            g_keytable[SDLK_POWER] = KC_POWER;
            g_keytable[SDLK_EURO] = KC_EURO;
            g_keytable[SDLK_UNDO] = KC_UNDO;
        }

        int GetModifiers(unsigned int sdl_mod)
        {
            int result = 0;
            if (sdl_mod & KMOD_ALT) result += KM_ALT;
            if (sdl_mod & KMOD_CTRL) result += KM_CONTROL;
            if (sdl_mod & KMOD_META) result += KM_META;
            if (sdl_mod & KMOD_SHIFT) result += KM_SHIFT;
            // TODO: Super not supported by SDL (we could catch it by monitoring the raw keyup/keydown events though).
            return result;
        }
        
        MouseButton MouseButtonFromSDL(int m, bool &error)
        {
            switch (m) {
            case SDL_BUTTON_LEFT: return MB_LEFT;
            case SDL_BUTTON_RIGHT: return MB_RIGHT;
            case SDL_BUTTON_MIDDLE: return MB_MIDDLE;
            case SDL_BUTTON_WHEELUP: return MB_WHEEL_UP;
            case SDL_BUTTON_WHEELDOWN: return MB_WHEEL_DOWN;
            default: error = true; return MB_LEFT;
            }
        }

        //
        // Handle SDL events.
        //

        void DoEvent(const SDL_Event &event)
        {
            typedef std::vector<WindowListener*> wl_vec;
            typedef wl_vec::const_iterator iter;
            
            boost::shared_ptr<SDLWindow> window(g_sdl_window.lock());
            if (!window) return;
            const wl_vec & wls = window->getListeners();

            switch (event.type) {
            case SDL_VIDEOEXPOSE:
                {
                    // Invalidate the entire screen
                    // TODO: Better tracking of the invalid region (does SDL tell us an invalid region?)
                    SDL_Surface * vid_surf = SDL_GetVideoSurface();
                    Rectangle rectangle(0, 0, vid_surf->w, vid_surf->h);
                    window->invalidateRectangle(rectangle);
                }
                break;

            case SDL_QUIT:
                for (iter it = wls.begin(); it != wls.end(); ++it) {
                    (*it)->onClose();
                }
                break;

            case SDL_ACTIVEEVENT:
                if ((event.active.state & SDL_APPINPUTFOCUS) != 0) {
                    // Window focus gained or lost.
                    if (event.active.gain) {
                        g_sdl_has_focus = true;
                        for (iter it = wls.begin(); it != wls.end(); ++it) {
                            (*it)->onGainFocus();
                        }
                    } else {
                        g_sdl_has_focus = false;
                        for (iter it = wls.begin(); it != wls.end(); ++it) {
                            (*it)->onLoseFocus();
                        }
                    }
                }
                if ((event.active.state & SDL_APPACTIVE) != 0) {
                    // Activation/deactivation (i.e. minimization or un-minimization).
                    if (event.active.gain) {
                        for (iter it = wls.begin(); it != wls.end(); ++it) {
                            (*it)->onActivate();
                        }
                    } else {
                        for (iter it = wls.begin(); it != wls.end(); ++it) {
                            (*it)->onDeactivate();
                        }
                    }
                }
                break;

            case SDL_VIDEORESIZE:
                {
                    // 1) Must reset the video mode. (This will also trigger a repaint.)
                    SDL_SetVideoMode(event.resize.w, event.resize.h, 0, g_sdl_required_flags);
                    window->invalidateAll();
                    
                    // 2) Inform the WindowListeners of the resize.
                    for (iter it = wls.begin(); it != wls.end(); ++it) {
                        (*it)->onResize(event.resize.w, event.resize.h);
                    }
                }
                break;

            case SDL_KEYDOWN:
            case SDL_KEYUP:
                {
                    // Find the corresponding character (if any)
                    int unicode = event.key.keysym.unicode;

                    // filter out control characters
                    if (unicode < 32 || unicode == 127) unicode = 0;

                    // Find the corresponding KeyCode.
                    const SDLKey keysym = event.key.keysym.sym;
                    KeyCode kc = KC_UNKNOWN;
                    std::map<SDLKey, KeyCode>::const_iterator key_it = g_keytable.find(keysym);
                    if (key_it != g_keytable.end()) kc = key_it->second;

                    // Work out what type of event this is
                    KeyEvent ke;
                    if (event.type == SDL_KEYDOWN) {
                        if (g_keystate[keysym]) {
                            ke = KE_AUTO_REPEAT;
                        } else {
                            ke = KE_PRESSED;
                        }
                        g_keystate[keysym] = true;
                    } else {
                        ke = KE_RELEASED;
                        g_keystate[keysym] = false;
                    }

                    // Get modifier state
                    const int modifiers = GetModifiers(event.key.keysym.mod);
                    
                    // Send the event.
                    for (iter it = wls.begin(); it != wls.end(); ++it) {
                        (*it)->onKey(ke, kc, unicode, modifiers);
                    }
                }
                break;

            case SDL_MOUSEBUTTONDOWN:
                {
                    bool err = false;
                    const MouseButton mb = MouseButtonFromSDL(event.button.button, err);
                    if (!err) {
                        for (iter it = wls.begin(); it != wls.end(); ++it) {
                            (*it)->onMouseDown(event.button.x, event.button.y, mb);
                        }
                    }
                }
                break;

            case SDL_MOUSEBUTTONUP:
                {
                    bool err = false;
                    const MouseButton mb = MouseButtonFromSDL(event.button.button, err);
                    if (!err) {
                        for (iter it = wls.begin(); it != wls.end(); ++it) {
                            (*it)->onMouseUp(event.button.x, event.button.y, mb);
                        }
                    }
                }
                break;

            case SDL_MOUSEMOTION:
                for (iter it = wls.begin(); it != wls.end(); ++it) {
                    (*it)->onMouseMove(event.motion.x, event.motion.y);
                }
                break;
            }
        }

    } // namespace


    //
    // SDLGfxDriver implementation
    //
    
    SDLGfxDriver::SDLGfxDriver()
        : video_subsystem(SDL_INIT_VIDEO), icon_id(-1)
    {
        const SDL_VideoInfo* info = SDL_GetVideoInfo();
        desktop_mode.width = info->current_w;
        desktop_mode.height = info->current_h;
        
        InitKeyTable();
        SDL_EnableUNICODE(1);
    }

    SDLGfxDriver::DisplayModeVector SDLGfxDriver::getFullScreenModes()
    {
        SDL_Rect ** modes = SDL_ListModes(0, SDL_ANYFORMAT | SDL_HWSURFACE | SDL_DOUBLEBUF | SDL_FULLSCREEN);

        if (modes == 0 || modes == reinterpret_cast<SDL_Rect**>(-1)) {
            return DisplayModeVector();
        } else {
            DisplayModeVector result;
            for (int i = 0; modes[i]; ++i) {
                DisplayMode m;
                m.width = modes[i]->w;
                m.height = modes[i]->h;
                result.push_back(m);
            }
            return result;
        }
    }

    SDLGfxDriver::DisplayMode SDLGfxDriver::getDesktopMode()
    {
        return desktop_mode;
    }
    
    boost::shared_ptr<Window> SDLGfxDriver::createWindow(int width, int height,
                                                         bool resizable, bool fullscreen,
                                                         const std::string &title)
    {
        if (g_sdl_window.lock()) {
            throw CoercriError("SDL only supports one window at a time");
        }

#ifdef WIN32
        // Set window icon from the given resource
        // This has to be done after SDL_Init, but before SDL_SetVideoMode.
        if (icon_id != -1) {
            const HINSTANCE handle = ::GetModuleHandle(NULL);
            const HICON icon = ::LoadIcon(handle, MAKEINTRESOURCE(icon_id));

            SDL_SysWMinfo wminfo;
            SDL_VERSION(&wminfo.version);
            if (SDL_GetWMInfo(&wminfo) != 1) {
                // error: wrong SDL version
            } else {
                const HWND hwnd = wminfo.window;
                ::SetClassLong(hwnd, GCL_HICON, (LONG) icon);
            }
        }
#endif
        
        g_sdl_required_flags = SDL_ANYFORMAT | SDL_HWSURFACE | SDL_DOUBLEBUF;
        if (fullscreen) g_sdl_required_flags |= SDL_FULLSCREEN;
        if (resizable) g_sdl_required_flags |= SDL_RESIZABLE;

        // In windowed mode we call SetVideoMode twice. This is a workaround for a bug when SDL is used with xmonad.
        SDL_SetVideoMode(width, height, 0, g_sdl_required_flags);
        if (!fullscreen) SDL_SetVideoMode(width, height, 0, g_sdl_required_flags);

        SDL_WM_SetCaption(title.c_str(), title.c_str());

        SDL_EnableUNICODE(1);
        
        boost::shared_ptr<SDLWindow> result(new SDLWindow);
        g_sdl_window = result;
        return result;
    }

    boost::shared_ptr<Graphic> SDLGfxDriver::createGraphic(boost::shared_ptr<const PixelArray> pixels, int hx, int hy)
    {
        return boost::shared_ptr<Graphic>(new SDLGraphic(pixels, hx, hy));
    }

    bool SDLGfxDriver::pollEvents()
    {
        boost::shared_ptr<SDLWindow> window(g_sdl_window.lock());
        if (window && window->need_window_resize) {
            // process this first.
            const std::vector<WindowListener*> & wls = window->getListeners();
            int w,h;
            window->getSize(w,h);
            for (std::vector<WindowListener*>::const_iterator it = wls.begin(); it != wls.end(); ++it) {
                (*it)->onResize(w,h);
            }
            window->need_window_resize = false;
            return true;
        }
        
        SDL_Event event;
        if (SDL_PollEvent(&event)) {
            DoEvent(event);
            return true;
        } else {
            return false;
        }
    }

    void SDLGfxDriver::setWindowsIcon(int resource_id)
    {
        icon_id = resource_id;
    }

    void SDLGfxDriver::setKeyRepeat(bool enabled)
    {
        if (enabled) {
            SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL);
        } else {
            SDL_EnableKeyRepeat(0, 0);
        }
    }
}
