/*
 * FILE:
 *   sdl_gfx_context.cpp
 *
 * AUTHOR:
 *   Stephen Thompson, 2008
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *
 */

#include "sdl_gfx_context.hpp"
#include "sdl_graphic.hpp"
#include "sdl_ttf_font.hpp"

namespace {
    // Low level SDL pixel routines
    // Parts of these are taken from the SDL introduction,
    // http://www.libsdl.org/intro.en/usingvideo.html

    void ReadPixel(SDL_Surface *surface, int x, int y, Uint8 &r, Uint8 &g, Uint8 &b)
    {
        switch (surface->format->BytesPerPixel) {
        case 1:
            {
                const Uint8 *bufp = (Uint8*)surface->pixels + y * surface->pitch + x;
                SDL_GetRGB(*bufp, surface->format, &r, &g, &b);
            }
            break;
            
        case 2:
            {
                const Uint16 *bufp = (Uint16*)surface->pixels + y * surface->pitch/2 + x;
                SDL_GetRGB(*bufp, surface->format, &r, &g, &b);
            }
            break;
            
        case 3:
            {
                const Uint8 *bufp = (Uint8*)surface->pixels + y * surface->pitch + x*3;
                r = *(bufp + surface->format->Rshift/8);
                g = *(bufp + surface->format->Gshift/8);
                b = *(bufp + surface->format->Bshift/8);
            }
            break;
            
        case 4:
            {
                const Uint32 *bufp = (Uint32*)surface->pixels + y * surface->pitch/4 + x;
                SDL_GetRGB(*bufp, surface->format, &r, &g, &b);
            }
            break;
        }
    }
    
    void PlotPixel(SDL_Surface *surface, int x, int y, Uint8 r, Uint8 g, Uint8 b)
    {
        const Uint32 color = SDL_MapRGB(surface->format, r, g, b);

        switch (surface->format->BytesPerPixel) {
        case 1:
            {
                Uint8 *bufp = (Uint8 *)surface->pixels + y * surface->pitch + x;
                *bufp = color;
            }
            break;
            
        case 2:
            {
                Uint16 *bufp = (Uint16 *)surface->pixels + y * surface->pitch/2 + x;
                *bufp = color;
            }
            break;

        case 3:
            {
                Uint8 *bufp = (Uint8 *)surface->pixels + y * surface->pitch + x*3;
                *(bufp+surface->format->Rshift/8) = r;
                *(bufp+surface->format->Gshift/8) = g;
                *(bufp+surface->format->Bshift/8) = b;
            }
            break;

        case 4:
            {
                Uint32 *bufp = (Uint32 *)surface->pixels + y*surface->pitch/4 + x;
                *bufp = color;
            }
            break;
        }
    }

    // Alpha Blending
    inline Uint8 AlphaBlend(Uint8 input, Uint8 screen, Uint8 alpha)
    {
        // alpha*input + (1-alpha)*screen
        // = alpha*(input - screen) + screen
        const int blend = int(alpha) * (int(input) - int(screen));
        const int result = (blend >> 8) + int(screen);
        return Uint8(result);
    }
}
    

namespace Coercri {

    SDLGfxContext::SDLGfxContext(SDL_Surface &surf)
        : surface(&surf), locked(false)
    {
        clearClipRectangle();
    }

    SDLGfxContext::~SDLGfxContext()
    {
        unlock();
        SDL_Flip(surface);
    }

    void SDLGfxContext::setClipRectangle(const Rectangle &rect)
    {
        clip_rectangle = IntersectRects(rect, Rectangle(0, 0, surface->w, surface->h));
        loadClipRectangle();
    }

    void SDLGfxContext::clearClipRectangle()
    {
        clip_rectangle = Rectangle(0, 0, surface->w, surface->h);
        loadClipRectangle();
    }

    void SDLGfxContext::loadClipRectangle()
    {
        SDL_Rect sdl_rect;
        sdl_rect.x = clip_rectangle.getLeft();
        sdl_rect.y = clip_rectangle.getTop();
        sdl_rect.w = clip_rectangle.getWidth();
        sdl_rect.h = clip_rectangle.getHeight();
        SDL_SetClipRect(surface, &sdl_rect);
    }
        
    Rectangle SDLGfxContext::getClipRectangle() const
    {
        return clip_rectangle;
    }

    int SDLGfxContext::getWidth() const
    {
        return surface->w;
    }

    int SDLGfxContext::getHeight() const
    {
        return surface->h;
    }
    
    void SDLGfxContext::clearScreen(Color colour)
    {
        unlock();
        SDL_FillRect(surface, 0, SDL_MapRGB(surface->format, colour.r, colour.g, colour.b));
    }

    void SDLGfxContext::plotPixel(int x, int y, Color input_col)
    {
        lock();

        if (!pointInClipRectangle(x, y)) return;

        Uint8 r = input_col.r, g = input_col.g, b = input_col.b;
        
        if (input_col.a < 255) {
            // Read the current contents of the pixel and do the alpha blending.
            Uint8 screen_r, screen_g, screen_b;
            ReadPixel(surface, x, y, screen_r, screen_g, screen_b);
            r = AlphaBlend(input_col.r, screen_r, input_col.a);
            g = AlphaBlend(input_col.g, screen_g, input_col.a);
            b = AlphaBlend(input_col.b, screen_b, input_col.a);
        }

        PlotPixel(surface, x, y, r, g, b);
    }

    void SDLGfxContext::drawGraphic(int x, int y, const Graphic &graphic)
    {
        unlock();
        const SDLGraphic *sdl_graphic = dynamic_cast<const SDLGraphic*>(&graphic);
        if (sdl_graphic) {
            sdl_graphic->blit(*surface, x, y);
        }
    }

    void SDLGfxContext::drawText(int x, int y, const Font &font, const std::string &text, Color col, bool antialias)
    {
        font.drawText(*this, x, y, text, col, antialias);
    }

    void SDLGfxContext::fillRectangle(const Rectangle &rect, Color col)
    {
        unlock();
        
        SDL_Rect sdl_rect;
        sdl_rect.x = rect.getLeft();
        sdl_rect.y = rect.getTop();
        sdl_rect.w = rect.getWidth();
        sdl_rect.h = rect.getHeight();

        if (col.a == 255) {
            // Non-blended case -- we can just use SDL_FillRect
            SDL_FillRect(surface, &sdl_rect, SDL_MapRGB(surface->format, col.r, col.g, col.b));
        } else {
            // Blended case -- create a temporary surface (filled with the requested colour) and blit it.
            SDL_Color sdl_col = {col.r, col.g, col.b};
            SDL_Surface * temp_surface = SDL_CreateRGBSurface(SDL_SRCALPHA, rect.getWidth(), rect.getHeight(), 8, 0, 0, 0, 0);
            SDL_SetColors(temp_surface, &sdl_col, 0, 1);
            SDL_FillRect(temp_surface, 0, 0); // clear the surface to zero.
            SDL_SetAlpha(temp_surface, SDL_SRCALPHA, col.a);
            SDL_BlitSurface(temp_surface, 0, surface, &sdl_rect);
            SDL_FreeSurface(temp_surface);
        }
    }
    
    bool SDLGfxContext::pointInClipRectangle(int x, int y) const
    {
        return PointInRect(clip_rectangle, x, y);
    }
    
    void SDLGfxContext::lock()
    {
        if (!locked) {
            SDL_LockSurface(surface);
            locked = true;
        }
    }

    void SDLGfxContext::unlock()
    {
        if (locked) {
            SDL_UnlockSurface(surface);
            locked = false;
        }
    }
}
