/*
 * FILE:
 *   sdl_subsystem_handle.cpp
 *
 * AUTHOR:
 *   Stephen Thompson, 2008
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *
 */

#include "sdl_subsystem_handle.hpp"
#include "sdl_error.hpp"

#include "SDL.h"
#include <map>

namespace Coercri {
    namespace {
        std::map<unsigned int, int> g_subsys_counts;
        int g_sdl_count = 0;
    }


    SDLSubSystemHandle::SDLSubSystemHandle(unsigned int s)
        : subsys(s), ref_count(0)
    {        
        // Initialize SDL if necessary
        if (g_sdl_count == 0) {
            int err = SDL_Init(SDL_INIT_NOPARACHUTE);
            if (err == -1) {
                throw SDLError("SDL_Init failed");
            }
            ++g_sdl_count;
        }

        // If no subsystem was requested then we are done
        if (s == 0) return;
        
        // Error check
        Uint32 mask = 1;
        int count = 0;
        while (1) {
            if (s & mask) ++count;
            if (mask >= 0x80000000) break;
            mask <<= 1;
        }
        if (count != 1) throw CoercriError("initSubSystem: invalid parameter");

        // Initialize the subsystem if necessary, and setup ref_count ptr.
        ref_count = &(g_subsys_counts[subsys]);
        if (*ref_count == 0) {
            int err = SDL_InitSubSystem(subsys | SDL_INIT_NOPARACHUTE);
            if (err == -1) {
                throw SDLError("SDL_InitSubSystem failed");
            }
        }

        // Add a new reference to it.
        ++ (*ref_count);
    }        

    SDLSubSystemHandle::SDLSubSystemHandle(const SDLSubSystemHandle &rhs)
        : subsys(0), ref_count(0)
    {
        ++g_sdl_count;
        if (rhs.subsys != 0) {
            subsys = rhs.subsys;
            ref_count = rhs.ref_count;
            ++(*ref_count);
        }
    }

    SDLSubSystemHandle & SDLSubSystemHandle::operator=(const SDLSubSystemHandle &rhs)
    {
        if (this == &rhs) return *this;

        if (subsys != 0) {
            --(*ref_count);
            if (*ref_count == 0) SDL_QuitSubSystem(subsys);
        }

        subsys = rhs.subsys;

        if (subsys != 0) {
            ref_count = rhs.ref_count;
            ++(*ref_count);
        }

        return *this;
    }

    SDLSubSystemHandle::~SDLSubSystemHandle()
    {
        if (subsys != 0) {
            --(*ref_count);
            if (*ref_count == 0) SDL_QuitSubSystem(subsys);
        }

        --g_sdl_count;
        if (g_sdl_count == 0) SDL_Quit();
    }
}
