/*
 * FILE:
 *   istream_rwops.cpp
 *
 * AUTHOR:
 *   Stephen Thompson, 2008
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *
 */

#include "istream_rwops.hpp"

#include <istream>

namespace {

    typedef boost::shared_ptr<std::istream> *stream_ptr;
    
    int Rseek(SDL_RWops *context, int offset, int whence)
    {
        stream_ptr str = static_cast<stream_ptr>(context->hidden.unknown.data1);
        switch (whence) {
        case SEEK_SET:
            (*str)->seekg(offset, std::ios_base::beg);
            break;
        case SEEK_CUR:
            (*str)->seekg(offset, std::ios_base::cur);
            break;
        case SEEK_END:
            (*str)->seekg(offset, std::ios_base::end);
            break;
        default:
            return -1; // error
        }
        if (!(*str)) return -1;
        else return (*str)->tellg();
    }
    
    int Rread(SDL_RWops *context, void *ptr, int size, int maxnum)
    {
        stream_ptr str = static_cast<stream_ptr>(context->hidden.unknown.data1);
        (*str)->read(static_cast<char*>(ptr), size*maxnum);
        if (!(*str)) return -1;
        else return (*str)->gcount() / size;
    }
    
    int Rwrite(SDL_RWops *context, const void *ptr, int size, int maxnum)
    {
        return -1; // read only!
    }
    
    int Rclose(SDL_RWops *context)
    {
        stream_ptr str_ptr = static_cast<stream_ptr>(context->hidden.unknown.data1);
        delete str_ptr;
        delete context;
        return 0;
    }
    
}  // namespace

SDL_RWops * Coercri::CreateRWOpsForIstream(boost::shared_ptr<std::istream> str)
{
    SDL_RWops * rwops = new SDL_RWops;
    try {
        rwops->seek = &Rseek;
        rwops->read = &Rread;
        rwops->write = &Rwrite;
        rwops->close = &Rclose;
        
        boost::shared_ptr<std::istream> * str_ptr = new boost::shared_ptr<std::istream>(str);
        rwops->hidden.unknown.data1 = static_cast<void*>(str_ptr); // never throws

        return rwops;    // never throws

    } catch (...) {
        delete rwops;
        throw;
    }
}
