/*
 * FILE:
 *   udp_socket.hpp
 *
 * PURPOSE:
 *   Wrapper over UDP sockets. Supports both unicast and broadcast
 *   sending, and receiving.
 *
 * AUTHOR:
 *   Stephen Thompson
 *
 * CREATED:
 *   14-Nov-2008
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *
 */

#ifndef COERCRI_UDP_SOCKET_HPP
#define COERCRI_UDP_SOCKET_HPP

#include "boost/noncopyable.hpp"

#include <string>

namespace Coercri {

    class UDPSocket : boost::noncopyable {
    public:
        // a UDPSocket can be bound (in which case it can both send
        // packets, and receive packets on the port it is bound to) or
        // unbound (in which case it can only send). See
        // NetworkDriver::createUDPSocket.
        
        virtual ~UDPSocket() { }

        // NOTE: Msgs are sent/received as strings. It might be better
        // to use vector<unsigned char> instead (for consistency with
        // NetworkConnection), but currently I think it's more
        // convenient to keep them as strings, because I am mostly
        // using the UDPSocket/TCPSocket classes for ASCII rather than
        // binary data at the moment...
        
        // Send a message to a specific address and port.
        virtual void send(const std::string &address, int port, const std::string &msg) = 0;
        
        // Broadcast a message on a given port.
        virtual void broadcast(int port, const std::string &msg) = 0;

        // Receive a message. For bound sockets, listens for incoming
        // packets on the port that was set in ctor; returns false if
        // no msg waiting, otherwise returns true and copies the msg
        // and address of sender into the two parameters. For unbound
        // sockets, always returns false.
        virtual bool receive(std::string &address, std::string &msg) = 0;
    };

}

#endif
