/*
 * FILE:
 *   network_driver.hpp
 *
 * PURPOSE:
 *   Networking interface.
 *
 *   Currently there are two basic interfaces provided -- a low level
 *   interface providing direct access to UDP sockets, and a
 *   higher level interface which provides TCP like connections
 *   optimized for games (currently this is built on top of ENet but
 *   other implementations are certainly possible).
 *
 * AUTHOR:
 *   Stephen Thompson
 *
 * CREATED:
 *   29-Jun-2008
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *   
 */

#ifndef COERCRI_NETWORK_DRIVER_HPP
#define COERCRI_NETWORK_DRIVER_HPP

#include "boost/noncopyable.hpp"
#include "boost/shared_ptr.hpp"

#include <string>
#include <vector>

namespace Coercri {

    class NetworkConnection;
    class UDPSocket;

    // This class is NOT thread safe in general, i.e. only the main
    // thread should make calls to NetworkDriver functions. However
    // see exception for resolveAddress below.
    
    
    class NetworkDriver : boost::noncopyable {
    public:
        virtual ~NetworkDriver() { }

        
        //
        // "High level" interface -- NetworkConnection class. 
        //
        
        // Opening outgoing connections:
        virtual boost::shared_ptr<NetworkConnection> openConnection(const std::string &host, int port) = 0;

        // Listening for incoming connections:

        // set the server port
        // (can only listen on one port at a time for now.)
        // (Can only set server port while server is disabled. Throws CoercriError if called when server enabled.)
        virtual void setServerPort(int port) = 0;

        // enable/disable the server
        // (initially disabled.)
        // (disabling the server will disconnect any existing connections.)
        virtual void enableServer(bool enabled) = 0;

        // poll for incoming connections
        // (this returns a list of new incoming connections that have appeared since we were last called.)
        typedef std::vector<boost::shared_ptr<NetworkConnection> > Connections;
        virtual Connections pollIncomingConnections() = 0;
        
        // Process any pending network events. This should be called
        // regularly from the program's main loop, to ensure that
        // packets get sent/received promptly. Returns TRUE if it "did
        // anything".
        virtual bool doEvents() = 0;

        // Are there still outstanding connections (this includes
        // connections that have been put into CLOSED state but are
        // not fully cleaned up yet).
        virtual bool outstandingConnections() = 0;
        

        
        //
        // Lower level interface -- direct access to UDP sockets.
        //
        // Note these don't use doEvents, instead they let the OS
        // handle the processing.
        //
        // port should be set to -1 to make an unbound socket or a
        // valid UDP port number to make a bound socket.
        //
        // reuseaddr controls whether we use the SO_REUSEADDR option 
        // when creating the socket. This allows multiple processes to
        // listen on the same port; broadcasts sent to the port will
        // then be received by each of the processes (although this 
        // doesn't apply to unicast messages apparently).
        //

        virtual boost::shared_ptr<UDPSocket> createUDPSocket(int port, bool reuseaddr) = 0;


        //
        // Utility functions
        //

        // Resolve an IP address to a hostname. (If not possible then
        // just returns the IP address back again.) May block.
        //
        // Thread safety: resolveAddress can be called from outside
        // the main thread (even if the main thread is executing some
        // other coercri function), BUT only one thread should be
        // inside resolveAddress at any given time.
        //
        virtual std::string resolveAddress(const std::string &ip_address) = 0;
    };

}

#endif
