/*
 * FILE:
 *   network_connection.hpp
 *
 * PURPOSE:
 *   High level networking interface. Implements a reliable, TCP-like
 *   protocol for games.
 * 
 *   The NetworkConnection object represents a single connection to a
 *   foreign host. Can send and receive packets. Note: data may not be
 *   actually sent/received until NetworkDriver::pollEvents is called.
 *
 * AUTHOR:
 *   Stephen Thompson, 2008
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *
 */

#ifndef COERCRI_NETWORK_CONNECTION_HPP
#define COERCRI_NETWORK_CONNECTION_HPP

#include <string>
#include <vector>

namespace Coercri {

    class NetworkConnection {
    public:
        virtual ~NetworkConnection() { }

        // Possible connection states
        enum State {
            PENDING,    // (Client side only) Awaiting connection to server
            CONNECTED,  // Ready to send/receive data
            CLOSED,     // Connection closed (either by local or remote close() call, or by a timeout).
            FAILED      // The connection request timed out or was refused.
        };

        
        // Query current connection state
        // NOTE: a CLOSED or FAILED connection may still be able to read data (if there has been buffering).
        // In particular it is always possible to read to the end of the current packet.
        virtual State getState() const = 0;

        
        // Close the connection. Note that we wait for any pending
        // data to be sent before actually closing.
        virtual void close() = 0;

        
        // Read data.
        //
        // Will return all received packets concatenated together into
        // a single vector (in the order they were sent). Packets will
        // never be broken up though (i.e. each packet is always
        // received either whole or not at all).
        //
        // If no data is available to be read, an empty vector is
        // returned.
        //
        // Existing contents of the vector are replaced.
        //
        virtual void receive(std::vector<unsigned char> &) = 0;

        
        // Send data.
        // The contents of the vector are treated as a single packet.
        virtual void send(const std::vector<unsigned char> &) = 0;


        // Find out the address of the other end of this connection.
        virtual std::string getAddress() = 0;
    };
}

#endif
