/*
 * FILE:
 *   byte_buf.hpp
 *
 * PURPOSE:
 *   Classes for marshalling various types (ints, strings etc) to/from
 *   a vector<unsigned char>. This makes it easier to send/receive
 *   binary data over a NetworkConnection.
 *   
 * AUTHOR:
 *   Stephen Thompson
 *
 * CREATED:
 *   20-Mar-2009
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *
 */

#ifndef COERCRI_BYTE_BUF_HPP
#define COERCRI_BYTE_BUF_HPP

#include <string>
#include <vector>

namespace Coercri {

    // Thread Safety: Multiple threads can access DIFFERENT ByteBufs
    // concurrently, but only one thread should access a given ByteBuf
    // at any one time.
    
    class InputByteBuf {
    public:
        typedef unsigned char ubyte;
    
        // NOTE: vector must NOT be reallocated while this class is used,
        // since we keep an iterator into it.
        InputByteBuf(const std::vector<ubyte> &buf_) : buf(buf_), iter(buf_.begin()) { }

        bool eof() const { return iter == buf.end(); }

        int readUbyte();
        int readByte();
        int readUshort();
        int readShort();
        int readVarInt();  // variable length int (1,2,3 or 4 bytes)
        void readNibbles(int &x, int &y);   // two numbers in range [0,15], encoded in one byte
        std::string readString();  // encoded as length (as VarInt) + data.

    private:
        const std::vector<ubyte> &buf;
        std::vector<ubyte>::const_iterator iter;
    };

    class OutputByteBuf {
    public:
        typedef unsigned char ubyte;

        // This appends to the given vector.
        OutputByteBuf(std::vector<ubyte> &buf_) : buf(buf_) { }

        void writeUbyte(int x);
        void writeUshort(int x);
        void writeShort(int x);
        void writeVarInt(int x);
        void writeNibbles(int x, int y);
        void writeString(const std::string &x);

    private:
        std::vector<ubyte> &buf;
    };

}

#endif
