/*
 * FILE:
 *   byte_buf.cpp
 *
 * AUTHOR:
 *   Stephen Thompson
 *
 * CREATED:
 *   20-Mar-2009
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *
 */

#include "byte_buf.hpp"

#include "../core/coercri_error.hpp"

namespace Coercri {

    int InputByteBuf::readUbyte()
    {
        if (eof()) {
            throw CoercriError("readUbyte failed");
        } else {
            return *iter++;
        }
    }
    
    int InputByteBuf::readByte()
    {
        const int x = readUbyte();
        if (x < 128) {
            return x;
        } else {
            return x - 256;
        }
    }
    
    int InputByteBuf::readUshort()
    {
        const int high = readUbyte();
        const int low = readUbyte();
        return (high * 256) + low;
    }
    
    int InputByteBuf::readShort()
    {
        const int high = readByte();
        const int low = readUbyte();
        return (high * 256) + low;
    }
    
    int InputByteBuf::readVarInt()
    {
        unsigned int val;
        bool sign;
        int c = readUbyte();
        // bit 7 = extend bit
        // bit 6 = sign bit
        // bits 5-0 = least significant bits
        val = (c & 0x3f);
        sign = (c & 0x40) != 0;
        if (c & 0x80) {
            // 2nd byte
            c = readUbyte();
            val |= ((c & 0x7f) << 6);
            if (c & 0x80) {
                // 3rd byte
                c = readUbyte();
                val |= ((c & 0x7f) << 13);
                if (c & 0x80) {
                    // 4th byte
                    c = readUbyte();
                    val |= (c << 20);
                } 
            }
        }
        if (sign) {
            return -int(val) - 1;
        } else {
            return val;
        }
    }

    void InputByteBuf::readNibbles(int &x, int &y)
    {
        int ub = readUbyte();
        x = (ub >> 4);
        y = (ub & 15);
    }
    
    std::string InputByteBuf::readString()
    {
        std::string s;
        const ptrdiff_t size = readVarInt();
        s.resize(std::min(size, buf.end() - iter));
        for (std::string::iterator it = s.begin(); it != s.end(); ++it) {
            *it = static_cast<char>(readUbyte());
        }
        return s;
    }


    void OutputByteBuf::writeUbyte(int x)
    {
        if (x < 0 || x > 255) throw CoercriError("writeUbyte: out of range");
        buf.push_back(x);
    }
    
    void OutputByteBuf::writeUshort(int x)
    {
        if (x < 0 || x > 65535) throw CoercriError("writeUshort: out of range");
        buf.push_back(x >> 8);
        buf.push_back(x & 0xff);
    }

    void OutputByteBuf::writeShort(int x)
    {
        if (x < -32768 || x > 32767) throw CoercriError("writeShort: out of range");
        // converting signed to unsigned is safe
        writeUshort(x);
    }

    void OutputByteBuf::writeVarInt(int x)
    {
        if (x < -268435456 || x > 268435455) throw CoercriError("writeVarInt: out of range");
        
        unsigned int sign;
        unsigned int val;
        if (x >= 0) {
            val = x;
            sign = 0;
        } else {
            val = -x-1;
            sign = 0x40;
        }
        
        if (val <= 63) {
            writeUbyte(sign + val);
        } else {
            writeUbyte(0x80 + sign + (val & 0x3f));
            if (val <= 8191) {
                writeUbyte(val >> 6);
            } else {
                writeUbyte(0x80 + ((val >> 6) & 0x7f));
                if (val <= 1048575) {
                    writeUbyte(val >> 13);
                } else {
                    writeUbyte(0x80 + ((val >> 13) & 0x7f));
                    writeUbyte(val >> 20);
                }
            }
        }
    }
    
    void OutputByteBuf::writeNibbles(int x, int y)
    {
        if (x < 0 || x > 15 || y < 0 || y > 15) throw CoercriError("writeNibbles: value out of range");
        writeUbyte((x << 4) + y);
    }
    
    void OutputByteBuf::writeString(const std::string &x)
    {
        writeVarInt(x.size());
        for (std::string::const_iterator it = x.begin(); it != x.end(); ++it) {
            writeUbyte(static_cast<unsigned char>(*it));
        }
    }

}
