/*
 * FILE:
 *   window.hpp
 *
 * PURPOSE:
 *   Class representing a Window. Windows can display graphics and
 *   receive keyboard and mouse input.
 *
 * AUTHOR:
 *   Stephen Thompson, 2008.
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *
 */

#ifndef COERCRI_WINDOW_HPP
#define COERCRI_WINDOW_HPP

#include "rectangle.hpp"
#include "region.hpp"

#include "boost/shared_ptr.hpp"
#include <vector>

namespace Coercri {

    class GfxContext;
    class WindowListener;
    
    class Window {
    public:
        Window() { }
        virtual ~Window() { }

        //
        // Virtual functions
        //
        
        // get current window properties
        virtual void getSize(int &w, int &h) const = 0;
        
        // Query whether the window currently has focus
        virtual bool hasFocus() const = 0;

        // Show or hide the mouse pointer.
        virtual void showMousePointer(bool shown) = 0;
        
        // Switch between windowed and full screen
        virtual void switchToWindowed(int w, int h) = 0;
        virtual void switchToFullScreen(int w, int h) = 0;

        // Get a GfxContext for drawing.
        // NB only one GfxContext should exist at a time (for each window),
        // otherwise strange things might happen...
        virtual std::auto_ptr<GfxContext> createGfxContext() = 0;
        
        //
        // Non-virtual functions
        //
        
        // WindowListeners for this window
        void addWindowListener(WindowListener *);
        void rmWindowListener(WindowListener *);

        // Invalid region handling
        void invalidateRectangle(const Rectangle &rect) { invalid_region.addRectangle(rect); }
        void invalidateRegion(const Region &region) { invalid_region.addRegion(region); }
        void invalidateAll();
        void cancelInvalidRegion() { invalid_region.clear(); }
        const Region & getInvalidRegion() const { return invalid_region; }
        bool needsRepaint() const { return !invalid_region.isEmpty(); }
        
    protected:
        std::vector<WindowListener*> listeners;
        Region invalid_region;
        
    private:
        // prevent copying
        Window(const Window &);
        void operator=(const Window &);
    };

}

#endif
