/*
 * FILE: 
 *   region.cpp
 *
 * AUTHOR:
 *   Stephen Thompson, 2008
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *
 */

#include "rectangle.hpp"
#include "region.hpp"

#include <list>

using Coercri::Rectangle;
using std::list;
using std::vector;

namespace {
    void AddRect(int l, int t, int r, int b, list<Rectangle> &lst, list<Rectangle>::iterator it)
    {
        Rectangle rect = Rectangle(l, t, r-l, b-t);
        if (!rect.isDegenerate()) {
            lst.insert(it, rect);
        }
    }
    
    void RectMinus(const Rectangle &lhs, const Rectangle &rhs, list<Rectangle> &lst, list<Rectangle>::iterator it)
    {
        const Rectangle intsct = IntersectRects(lhs, rhs);
        AddRect(lhs.getLeft(), lhs.getTop(), lhs.getRight(), intsct.getTop(), lst, it);
        AddRect(lhs.getLeft(), intsct.getTop(), intsct.getLeft(), intsct.getBottom(), lst, it);
        AddRect(intsct.getRight(), intsct.getTop(), lhs.getRight(), intsct.getBottom(), lst, it);
        AddRect(lhs.getLeft(), intsct.getBottom(), lhs.getRight(), lhs.getBottom(), lst, it);
    }
}

namespace Coercri {

    void Region::addRectangle(const Rectangle &rect)
    {
        if (rect.isDegenerate()) return;
        
        // Break into several rectangles if necessary.
        // Note: we don't bother trying to optimize by merging together adjacent
        // rectangles that share a side.
        list<Rectangle> new_rects;
        new_rects.push_back(rect);
        
        for (vector<Rectangle>::const_iterator exist = rectangles.begin(); exist != rectangles.end(); ++exist) {
            for (list<Rectangle>::iterator add = new_rects.begin(); add != new_rects.end(); ) {
                RectMinus(*add, *exist, new_rects, add);  // inserts Add\Exist into new_rects.
                add = new_rects.erase(add);               // removes Add from new_rects
            }
        }
        rectangles.reserve(rectangles.size() + new_rects.size());
        for (list<Rectangle>::const_iterator it = new_rects.begin(); it != new_rects.end(); ++it) {
            rectangles.push_back(*it);
        }
    }

    void Region::addRegion(const Region &other)
    {
        for (const_iterator it = other.begin(); it != other.end(); ++it) {
            addRectangle(*it);
        }
    }

    Rectangle Region::getBoundingBox() const
    {
        Rectangle bb;
        for (vector<Rectangle>::const_iterator it = rectangles.begin(); it != rectangles.end(); ++it) {
            if (bb.isDegenerate()) {
                bb = *it;
            } else {
                const int new_left = std::min(bb.getLeft(), it->getLeft());
                const int new_right = std::max(bb.getRight(), it->getRight());
                const int new_top = std::min(bb.getTop(), it->getTop());
                const int new_bottom = std::max(bb.getBottom(), it->getBottom());
                bb = Rectangle(new_left, new_top, new_right-new_left, new_bottom-new_top);
            }
        }
        return bb;
    }
}
