/*
 * FILE:
 *   rectangle.hpp
 *
 * PURPOSE:
 *   Rectangle class
 *
 * AUTHOR:
 *   Stephen Thompson, 2008
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *
 */

#ifndef COERCRI_RECTANGLE_HPP
#define COERCRI_RECTANGLE_HPP

#include <algorithm>

namespace Coercri {

    class Rectangle {
    public:
        // Note invariant: width >= 0 && height >= 0

        // Constructors
        Rectangle()
            : left(0), top(0), width(0), height(0) { }
        Rectangle(int l, int t, int w, int h)
        : left(l), top(t), width(w), height(h) {
            if (width < 0) width = 0; if (height < 0) height = 0;
        }

        // query positions
        int getLeft() const { return left; }
        int getRight() const { return left + width; }
        int getTop() const { return top; }
        int getBottom() const { return top + height; }

        // query size
        int getWidth() const { return width; }
        int getHeight() const { return height; }

        // "degenerate" means zero area.
        bool isDegenerate() const { return width == 0 || height == 0; }

        // Move/Resize
        void moveTo(int l, int t) {
            left = l; top = t;
        }
        void translate(int x, int y) {
            left += x; top += y;
        }
        void setSize(int w, int h) {  // fixes top left corner
            width = w; if (width < 0) width = 0;
            height = h; if (height < 0) height = 0;
        }
        void addToSize(int w, int h) {
            width += w; if (width < 0) width = 0;
            height += h; if (height < 0) height = 0;
        }
        
    private:
        int left, top;
        int width, height;
    };

    // Intersect two rectangles
    inline Rectangle IntersectRects(const Rectangle &r1, const Rectangle &r2)
    {
        const int new_left = std::max(r1.getLeft(), r2.getLeft());
        const int new_right = std::min(r1.getRight(), r2.getRight());
        const int new_top = std::max(r1.getTop(), r2.getTop());
        const int new_bottom = std::min(r1.getBottom(), r2.getBottom());
        return Rectangle(new_left, new_top, new_right - new_left, new_bottom - new_top);
    }

    // Check if a point is in a rectangle
    inline bool PointInRect(const Rectangle &r, int x, int y)
    {
        return x >= r.getLeft() && x < r.getRight() && y >= r.getTop() && y < r.getBottom();
    }
}

#endif
