/*
 * FILE:
 *   load_system_ttf.cpp
 *
 * AUTHOR:
 *   Stephen Thompson, 2008
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *
 */

#include "font.hpp"
#include "load_system_ttf.hpp"
#include "ttf_loader.hpp"
#include "../core/coercri_error.hpp"

#ifdef USE_FONTCONFIG
#include <fontconfig/fontconfig.h>
#endif

#include <fstream>

namespace Coercri {

#ifdef USE_FONTCONFIG
    namespace {
        struct DeleteFcPattern {
            void operator()(FcPattern *p) { FcPatternDestroy(p); }
        };
        struct DeleteFcFontSet {
            void operator()(FcFontSet *f) { FcFontSetDestroy(f); }
        };
    }
#endif    
    
    boost::shared_ptr<Font> LoadSystemTTF(TTFLoader &loader, const std::vector<std::string> &font_names, int size)
    {
        std::vector<std::string> filenames;  // actual filename(s) found for this font.
        filenames.reserve(font_names.size());
        
#ifdef USE_FONTCONFIG

        FcInit();

        boost::shared_ptr<FcPattern> pattern(FcPatternCreate(), DeleteFcPattern());
        FcPatternAddInteger(pattern.get(), FC_SLANT, FC_SLANT_ROMAN);
        FcPatternAddInteger(pattern.get(), FC_WEIGHT, FC_WEIGHT_NORMAL);
        FcPatternAddInteger(pattern.get(), FC_WIDTH, FC_WIDTH_NORMAL);

        for (std::vector<std::string>::const_iterator it = font_names.begin(); it != font_names.end(); ++it) {
            FcPatternAddString(pattern.get(), FC_FAMILY, (const FcChar8*)(it->c_str()));
        }

        FcDefaultSubstitute(pattern.get());
        FcConfigSubstitute(0, pattern.get(), FcMatchPattern);

        FcResult dummy;
        boost::shared_ptr<FcFontSet> matching_set(FcFontSort(0, pattern.get(), false, 0, &dummy), DeleteFcFontSet());
        
        if (matching_set) {
            for (int i = 0; i < matching_set->nfont; ++i) {
                FcChar8 * filename = 0;
                FcPatternGetString(matching_set->fonts[i], FC_FILE, 0, &filename);
                if (filename) {
                    filenames.push_back((char*)(filename));
                }
            }
        }

#elif defined(WIN32)

        const std::string font_dir = std::string(getenv("WINDIR")) + std::string("\\Fonts\\");
        for (std::vector<std::string>::const_iterator it = font_names.begin(); it != font_names.end(); ++it) {
            filenames.push_back(font_dir + (*it) + ".ttf");
        }

#else
#error TrueType font loading not supported on this platform.
#endif

        // Now try to open each file in turn.
        for (std::vector<std::string>::const_iterator it = filenames.begin(); it != filenames.end(); ++it) {
            boost::shared_ptr<std::ifstream> str(new std::ifstream(it->c_str(), std::ios_base::in | std::ios_base::binary));
            if (*str) {
                boost::shared_ptr<Font> result;
                try {
                    result = loader.loadFont(str, size);
                } catch (...) {
                    // disregard exceptions, as we want to try the
                    // next font if there was an error with this one.
                }
                if (result) {
                    // yay! loaded the font successfully.
                    return result;
                }
            }
        }

        // Oh dear, it seems we couldn't load the font
        throw CoercriError("Could not load font");        
    }
}        
