/*
 * FILE:
 *   load_bmp.cpp
 *
 * AUTHOR:
 *   Stephen Thompson, 2008
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *
 * NOTES:
 *   Source: http://en.wikipedia.org/wiki/BMP_file_format
 * 
 */

#include "load_bmp.hpp"
#include "pixel_array.hpp"
#include "../core/coercri_error.hpp"

#include "boost/shared_array.hpp"
#include <istream>

namespace Coercri {

    // BMPs are stored little-endian so we need some endian-independent read functions:

    unsigned char ReadByte(std::istream &str)
    {
        unsigned char c = 0;
        str.read((char*)&c, 1);
        return c;
    }
    
    unsigned short int ReadShort(std::istream &str)
    {
        unsigned char c[2] = {0,0};
        str.read((char*)c, 2);
        return (c[0] + (c[1]<<8));
    }

    unsigned int ReadLong(std::istream &str)
    {
        unsigned char c[4] = {0,0,0,0};
        str.read((char*)c, 4);
        return (c[0] + (c[1]<<8) + (c[2]<<16) + (c[3]<<24));
    }

    boost::shared_ptr<PixelArray> LoadBMP(std::istream &str)
    {
        boost::shared_ptr<PixelArray> result;

        // Check the magic number
        char magic[2] = {0,0};
        str.read(magic, 2);
        if (!str || magic[0] != 'B' || magic[1] != 'M') {
            throw CoercriError("Failed to load BMP file: incorrect file format");
        }

        ReadLong(str);   // file size
        ReadLong(str);   // reserved
        ReadLong(str);   // offset to bitmap data
        const unsigned int header_size = ReadLong(str);

        // Header size 40 is the Windows V3 format, which is the most common
        // Header size 12 is the OS/2 V1 format which is apparently also popular
        if (header_size != 40 && header_size != 12) throw CoercriError("Failed to load BMP file: unsupported header size");

        const unsigned int bitmap_width = (header_size == 40 ? ReadLong(str) : ReadShort(str));
        const unsigned int bitmap_height = (header_size == 40 ? ReadLong(str) : ReadShort(str));
        ReadShort(str);   // number of color planes (always 1)
        const unsigned int bits_per_pixel = ReadShort(str);

        // I only support either 8-bpp or 24-bpp bitmaps for now.
        // (other values are more complicated to work with and also
        // uncommon in practice.)
        if (bits_per_pixel != 8 && bits_per_pixel != 24) {
            throw CoercriError("Failed to load BMP file: unsupported bits per pixel");
        }
        
        unsigned int compression_method = 0;
        unsigned int number_of_colors = 0;
        if (header_size == 40) {
            compression_method = ReadLong(str);
            ReadLong(str);   // image size
            ReadLong(str);   // horizontal resolution
            ReadLong(str);   // vertical resolution
            number_of_colors = ReadLong(str);
            ReadLong(str);   // number of important colors
        }

        if (number_of_colors == 0 && bits_per_pixel <= 8) {
            // Number_of_colors == 0 means use the default of 2^n colours
            number_of_colors = (1 << bits_per_pixel);
        }
        if (bits_per_pixel > 8) {
            // The palette isn't used for greater than 8 bits per pixel.
            number_of_colors = 0;
        }
        if (number_of_colors > 256) {
            throw CoercriError("Failed to load BMP file: Incorrect number of colors");
        }
        
        if (compression_method != 0) {
            // We don't support compressed bitmaps yet
            throw CoercriError("Failed load BMP file: unsupported compression method");
        }

        boost::shared_array<Color> palette(new Color[number_of_colors]);
        for (unsigned int i = 0; i < number_of_colors; ++i) {
            // palette entries are stored in BGR order
            const unsigned char b = ReadByte(str);
            const unsigned char g = ReadByte(str);
            const unsigned char r = ReadByte(str);
            if (header_size==40) ReadByte(str);  // skip padding byte
            palette[i] = Color(r,g,b);
        }

        result.reset(new PixelArray(bitmap_width, bitmap_height));

        // calculate the number of padding bytes at the end of each row.
        const int bytes_per_row = (bits_per_pixel/8) * bitmap_width;
        const int left_over = bytes_per_row % 4;
        const int padding = (left_over==0) ? 0 : 4 - left_over;
        
        // Read the data. (BMPs are stored bottom-to-top so the y loop runs backwards.)
        for (int y = bitmap_height - 1; y >= 0; --y) {
            for (int x = 0; x < bitmap_width; ++x) {
                Color& col = (*result)(x,y);
                if (bits_per_pixel == 24) {
                    // don't forget: BGR order!
                    col.b = ReadByte(str);
                    col.g = ReadByte(str);
                    col.r = ReadByte(str);
                } else {
                    unsigned char index = ReadByte(str);
                    col = palette[index];
                }
                // BMPs don't support transparency so make sure A=255
                col.a = 255;
            }
            for (int p = 0; p < padding; ++p) ReadByte(str);
        }

        return result;
    }
}
