/*
 * FILE:
 *   gfx_driver.hpp
 *
 * PURPOSE:
 *   The central "graphics driver" object -- responsible for creating
 *   Windows, Graphics, Fonts, etc., and also for dispatching events
 *   to windows.
 *
 * AUTHOR:
 *   Stephen Thompson, 2008
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *
 */

#ifndef COERCRI_GFX_DRIVER_HPP
#define COERCRI_GFX_DRIVER_HPP

#include "boost/noncopyable.hpp"
#include "boost/shared_ptr.hpp"

#include <vector>

namespace Coercri {

    class Font;
    class GfxListener;
    class Graphic;
    class PixelArray;
    class Window;
    
    class GfxDriver : boost::noncopyable {
    public:
        virtual ~GfxDriver() { }

        // Find available display sizes for full screen
        struct DisplayMode {
            int width, height;
        };
        typedef std::vector<DisplayMode> DisplayModeVector;
        virtual DisplayModeVector getFullScreenModes() = 0;
        virtual DisplayMode getDesktopMode() = 0;
        
        // Create a new Window
        virtual boost::shared_ptr<Window> createWindow(int width, int height,
                                                       bool resizable, bool fullscreen,
                                                       const std::string &title) = 0;

        // Create a Graphic from a PixelArray.
        virtual boost::shared_ptr<Graphic> createGraphic(boost::shared_ptr<const PixelArray> pixels,
                                                         int hx = 0,
                                                         int hy = 0) = 0;

        // Poll for events, runs any WindowListeners that have been attached to windows
        // Returns TRUE if any events were processed.
        virtual bool pollEvents() = 0;

        // MS Windows only -- sets icon for creation of new windows
        virtual void setWindowsIcon(int resource_id) { }

        // Enable/disable key repeat. Default is disabled.
        virtual void setKeyRepeat(bool enabled) = 0;
    };

}

#endif
