/*
 * FILE:
 *   gfx_context.cpp
 *
 * AUTHOR:
 *   Stephen Thompson, 2008
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *
 */

#include "gfx_context.hpp"
#include "rectangle.hpp"

namespace Coercri {

    void GfxContext::drawLine(int x0, int y0, int x1, int y1, Color col)
    {
        // Use Bresenham's algorithm
        const bool steep = std::abs(y1 - y0) > std::abs(x1 - x0);
        if (steep) {
            std::swap(x0, y0);
            std::swap(x1, y1);
        }
        if (x0 > x1) {
            std::swap(x0, x1);
            std::swap(y0, y1);
        }
        const int deltax = x1 - x0;
        const int deltay = std::abs(y1 - y0);
        int y = y0;
        int error = 0;
        const int ystep = y0 < y1 ? 1 : -1;
        const int errstep = (deltay + deltay);
        const int errmax = (deltax + deltax);
        for (int x = x0; x <= x1; ++x) {
            if (steep) {
                plotPixel(y, x, col);
            } else {
                plotPixel(x, y, col);
            }
            error += errstep;
            if (error >= deltax) {
                y += ystep;
                error -= errmax;
            }
        }
    }

    void GfxContext::drawRectangle(const Rectangle &rect, Color col)
    {
        if (rect.isDegenerate()) return;

        const int left = rect.getLeft();
        const int right = rect.getRight() - 1;
        const int top = rect.getTop();
        const int bottom = rect.getBottom() - 1;

        if (top == bottom) {
            drawLine(left, top, right, top, col);
        } else if (left == right) {
            drawLine(left, top, left, bottom, col);
        } else {
            drawLine(left, top, right, top, col);
            drawLine(left, bottom, right, bottom, col);
            if (bottom > top+1) {
                drawLine(left, top+1, left, bottom-1, col);
                drawLine(right, top+1, right, bottom-1, col);
            }
        }
    }

    void GfxContext::fillRectangle(const Rectangle &rect, Color col)
    {
        if (rect.isDegenerate()) return;

        const int left = rect.getLeft();
        const int right = rect.getRight() - 1;

        for (int y = rect.getTop(); y < rect.getBottom(); ++y) {
            drawLine(left, y, right, y, col);
        }
    }
}
