/*
 * FILE:
 *   bitmap_font.cpp
 *
 * AUTHOR:
 *   Stephen Thompson, 2008
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *
 */

#include "bitmap_font.hpp"
#include "gfx_context.hpp"
#include "pixel_array.hpp"

#include <cstring>

namespace Coercri {

    BitmapFont::BitmapFont(boost::shared_ptr<PixelArray> pix)
        : pixels(pix)
    {
        const int w = pixels->getWidth();

        std::memset(ofs, 0, sizeof(ofs));
        std::memset(width, 0, sizeof(width));

        int ch = 32;
        ofs[ch] = 0;
        for (int i = 0; i < w; ++i) {
            const Color& pixel = (*pixels)(i, 0);
            if (pixel.r==255 && pixel.g==0 && pixel.b==255) {
                width[ch] = i - ofs[ch];
                ++ch;
                for (; i < w; ++i) {
                    const Color& pix2 = (*pixels)(i, 0);
                    if (pix2.r == 0 && pix2.g == 0 && pix2.b == 0) {
                        ofs[ch] = i;
                        break;
                    }
                }
            }
        }

        // spaces are a special case
        width[32] = ofs[33];
    }

    void BitmapFont::drawText(GfxContext &dest, int x, int y, const std::string &text, Color col, bool antialias) const
    {
        for (std::string::const_iterator it = text.begin(); it != text.end(); ++it) {
            if (*it > 0 && *it < 256) {
                if (*it != 32) {  // space
                    for (int j = 1; j < pixels->getHeight(); ++j) {
                        for (int i = 0; i < width[*it]; ++i) {
                            const Color &rgb = (*pixels)(i + ofs[*it], j);
                            if (rgb.r == 255 && rgb.g == 255 && rgb.b == 255) {
                                dest.plotPixel(x+i, y+j-1, col);
                            }
                        }
                    }
                }
                x += width[*it];
            }
        }
    }

    void BitmapFont::getTextSize(const std::string &text, int &w, int &h) const
    {
        w = 0;
        for (std::string::const_iterator it = text.begin(); it != text.end(); ++it) {
            if (*it > 0 && *it < 256) {
                w += width[*it];
            }
        }
        // assume all chars same height for now
        h = pixels->getHeight() - 1;
    }

    int BitmapFont::getTextHeight() const
    {
        return pixels->getHeight() - 1;
    }

}
