/*
 * FILE:
 *   cg_listener.cpp
 *
 * AUTHOR:
 *   Stephen Thompson, 2008
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *
 */

#include "cg_graphics.hpp"
#include "cg_input.hpp"
#include "cg_listener.hpp"
#include "../gfx/window.hpp"
#include "../timer/timer.hpp"

#include "guichan.hpp"

namespace Coercri {

    class CGListenerImpl {
    public:
        CGListenerImpl(boost::shared_ptr<Window> win, boost::shared_ptr<gcn::Gui> g, boost::shared_ptr<Timer> tmr)
            : window(win), gui(g), timer(tmr), input(tmr), gui_enabled(false),
              last_mouse_x(0), last_mouse_y(0)
        { }
        
        boost::shared_ptr<Window> window;
        boost::shared_ptr<gcn::Gui> gui;
        boost::shared_ptr<Timer> timer;
        
        CGInput input;
        CGGraphics graphics;
        
        bool gui_enabled;
        int last_mouse_x, last_mouse_y;
    };

    CGListener::CGListener(boost::shared_ptr<Window> window, boost::shared_ptr<gcn::Gui> gui, boost::shared_ptr<Timer> timer)
        : pimpl(new CGListenerImpl(window, gui, timer))
    {
        pimpl->gui->setInput(&pimpl->input);
        pimpl->gui->setGraphics(&pimpl->graphics);
    }

    void CGListener::enableGui()
    {
        if (pimpl->gui_enabled) return;
        pimpl->gui_enabled = true;

        // Window may have been resized while the GUI was disabled, so do a resize now.
        int w, h;
        pimpl->window->getSize(w, h);
        onResize(w, h);
    }

    void CGListener::disableGui()
    {
        pimpl->gui_enabled = false;
    }

    bool CGListener::isGuiEnabled() const
    {
        return pimpl->gui_enabled;
    }

    bool CGListener::processInput()
    {
        if (!pimpl->gui_enabled) return false;
        
        if (pimpl->input.inputWaiting()) {
            pimpl->gui->logic();
            pimpl->window->invalidateAll();
            return true;
        } else {
            return false;
        }
    }
    
    void CGListener::draw(GfxContext &gc)
    {
        if (!pimpl->gui_enabled) return;

        pimpl->graphics.setTarget(&gc);
        pimpl->gui->draw();
        pimpl->graphics.setTarget(0);
    }

    void CGListener::onResize(int new_width, int new_height)
    {
        if (!pimpl->gui_enabled) return;
        
        if (pimpl->gui->getTop()) {
            pimpl->gui->getTop()->setSize(new_width, new_height);
            pimpl->window->invalidateAll();  // make sure it gets redrawn.
        }
    }

    void CGListener::onKey(KeyEvent ke, KeyCode kc, int character, int modifiers)
    {
        gcn::Key k;

        if (character >= 32 && character < 1000) {
            // Use 'character' if it is available.
            k = character;
        } else {
            // Convert 'kc' to a guichan value.
            switch (kc) {
            case KC_SPACE: k = gcn::Key::SPACE; break;
            case KC_TAB: k = gcn::Key::TAB; break;
            case KC_KP_ENTER: k = gcn::Key::ENTER; break;
            case KC_RETURN: k = gcn::Key::ENTER; break;
            case KC_LEFT_ALT: k = gcn::Key::LEFT_ALT; break;
            case KC_RIGHT_ALT: k = gcn::Key::RIGHT_ALT; break;
            case KC_LEFT_SHIFT: k = gcn::Key::LEFT_SHIFT; break;
            case KC_RIGHT_SHIFT: k = gcn::Key::RIGHT_SHIFT; break;
            case KC_LEFT_CONTROL: k = gcn::Key::LEFT_CONTROL; break;
            case KC_RIGHT_CONTROL: k = gcn::Key::RIGHT_CONTROL; break;
            case KC_LEFT_META: k = gcn::Key::LEFT_META; break;
            case KC_RIGHT_META: k = gcn::Key::RIGHT_META; break;
            case KC_LEFT_SUPER: k = gcn::Key::LEFT_SUPER; break;
            case KC_RIGHT_SUPER: k = gcn::Key::RIGHT_SUPER; break;
            case KC_INSERT: k = gcn::Key::INSERT; break;
            case KC_HOME: k = gcn::Key::HOME; break;
            case KC_PAGE_UP: k = gcn::Key::PAGE_UP; break;
            case KC_DELETE: k = gcn::Key::DELETE; break;
            case KC_END: k = gcn::Key::END; break;
            case KC_PAGE_DOWN: k = gcn::Key::PAGE_DOWN; break;
            case KC_ESCAPE: k = gcn::Key::ESCAPE; break;
            case KC_CAPS_LOCK: k = gcn::Key::CAPS_LOCK; break;
            case KC_BACKSPACE: k = gcn::Key::BACKSPACE; break;
            case KC_F1: k = gcn::Key::F1; break;
            case KC_F2: k = gcn::Key::F2; break;
            case KC_F3: k = gcn::Key::F3; break;
            case KC_F4: k = gcn::Key::F4; break;
            case KC_F5: k = gcn::Key::F5; break;
            case KC_F6: k = gcn::Key::F6; break;
            case KC_F7: k = gcn::Key::F7; break;
            case KC_F8: k = gcn::Key::F8; break;
            case KC_F9: k = gcn::Key::F9; break;
            case KC_F10: k = gcn::Key::F10; break;
            case KC_F11: k = gcn::Key::F11; break;
            case KC_F12: k = gcn::Key::F12; break;
            case KC_F13: k = gcn::Key::F13; break;
            case KC_F14: k = gcn::Key::F14; break;
            case KC_F15: k = gcn::Key::F15; break;
            case KC_PRINT: k = gcn::Key::PRINT_SCREEN; break;
            case KC_SCROLL_LOCK: k = gcn::Key::SCROLL_LOCK; break;
            case KC_PAUSE: k = gcn::Key::PAUSE; break;
            case KC_NUM_LOCK: k = gcn::Key::NUM_LOCK; break;
            case KC_LEFT: k = gcn::Key::LEFT; break;
            case KC_RIGHT: k = gcn::Key::RIGHT; break;
            case KC_UP: k = gcn::Key::UP; break;
            case KC_DOWN: k = gcn::Key::DOWN; break;
            default: return;   // Unknown key
            }
        }

        gcn::KeyInput key_input(k, ke == KE_RELEASED ? gcn::KeyInput::RELEASED : gcn::KeyInput::PRESSED);
        key_input.setAltPressed((modifiers & KM_ALT) != 0);
        key_input.setControlPressed((modifiers & KM_CONTROL) != 0);
        key_input.setMetaPressed((modifiers & KM_META) != 0);
        key_input.setShiftPressed((modifiers & KM_SHIFT) != 0);
        key_input.setNumericPad(kc >= KC_KP_0 && kc <= KC_KP_PLUS);
        pimpl->input.addKeyInput(key_input);
    }

    void CGListener::onMouseDown(int x, int y, MouseButton button)
    {
        if (!pimpl->gui_enabled) return;
        
        unsigned int b, t;
        switch (button) {
        case MB_LEFT:
            b = gcn::MouseInput::LEFT;
            t = gcn::MouseInput::PRESSED;
            break;
        case MB_RIGHT:
            b = gcn::MouseInput::RIGHT;
            t = gcn::MouseInput::PRESSED;
            break;
        case MB_MIDDLE:
            b = gcn::MouseInput::MIDDLE;
            t = gcn::MouseInput::PRESSED;
            break;
        case MB_WHEEL_UP:
            b = gcn::MouseInput::EMPTY;
            t = gcn::MouseInput::WHEEL_MOVED_UP;
            break;
        case MB_WHEEL_DOWN:
            b = gcn::MouseInput::EMPTY;
            t = gcn::MouseInput::WHEEL_MOVED_DOWN;
            break;
        default:
            return;
        }
        const int timestamp = pimpl->timer ? pimpl->timer->getMsec() : 0;

        pimpl->input.addMouseInput(gcn::MouseInput(b, t, x, y, timestamp));
        pimpl->last_mouse_x = x;
        pimpl->last_mouse_y = y;
    }

    void CGListener::onMouseUp(int x, int y, MouseButton button)
    {
        if (!pimpl->gui_enabled) return;
        
        unsigned int b;
        switch (button) {
        case MB_LEFT: b = gcn::MouseInput::LEFT; break;
        case MB_RIGHT: b = gcn::MouseInput::RIGHT; break;
        case MB_MIDDLE: b = gcn::MouseInput::MIDDLE; break;
        default: return;
        }
        const int timestamp = pimpl->timer ? pimpl->timer->getMsec() : 0;
        pimpl->input.addMouseInput(gcn::MouseInput(b, gcn::MouseInput::RELEASED, x, y, timestamp));
        pimpl->last_mouse_x = x;
        pimpl->last_mouse_y = y;
    }

    void CGListener::onMouseMove(int x, int y)
    {
        if (!pimpl->gui_enabled) return;
        
        const int timestamp = pimpl->timer ? pimpl->timer->getMsec() : 0;
        pimpl->input.addMouseInput(gcn::MouseInput(gcn::MouseInput::EMPTY, gcn::MouseInput::MOVED, x, y, timestamp));
        pimpl->last_mouse_x = x;
        pimpl->last_mouse_y = y;
    }

    void CGListener::repeatLastMouseInput()
    {
        onMouseMove(pimpl->last_mouse_x, pimpl->last_mouse_y);
    }
}
