/*
 * FILE:
 *   cg_graphics.cpp
 *
 * AUTHOR:
 *   Stephen Thompson, 2008
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *
 */

#include "cg_graphics.hpp"
#include "cg_image.hpp"
#include "../gfx/gfx_context.hpp"
#include "../gfx/rectangle.hpp"

namespace Coercri {

    bool CGGraphics::pushClipArea(gcn::Rectangle area)
    {
        if (!gfx_context) throw GCN_EXCEPTION("pushClipArea: no gfx context");

        // Use the base class to help keep track of clip areas
        const bool result = Graphics::pushClipArea(area);
        loadClipArea();
        return result;
    }

    void CGGraphics::popClipArea()
    {
        if (!gfx_context) throw GCN_EXCEPTION("popClipArea: no gfx context");
        Graphics::popClipArea();
        loadClipArea();
    }

    void CGGraphics::loadClipArea()
    {
        // Load the current clip area into Coercri
        if (mClipStack.empty()) {
            gfx_context->clearClipRectangle();
        } else {
            const gcn::Rectangle & curr_clip = mClipStack.top();
            gfx_context->setClipRectangle(Coercri::Rectangle(curr_clip.x,
                                                             curr_clip.y,
                                                             curr_clip.width,
                                                             curr_clip.height));
        }
    }
    
    void CGGraphics::drawImage(const gcn::Image *image, int srcX, int srcY, int dstX, int dstY, int width, int height)
    {
        if (!gfx_context) throw GCN_EXCEPTION("drawImage: no gfx context");
        
        if (srcX != 0 || srcY != 0 || width != image->getWidth() || height != image->getHeight()) {
            throw GCN_EXCEPTION("drawImage: not implemented for partial images");
        }

        const CGImage * coercri_image = dynamic_cast<const CGImage*>(image);
        if (!coercri_image) {
            throw GCN_EXCEPTION("drawImage: Incompatible image type");
        }

        transformPoint(dstX, dstY);
        gfx_context->drawGraphic(dstX, dstY, coercri_image->getGraphic());
    }

    void CGGraphics::drawImage(const gcn::Image *image, int dstX, int dstY)
    {
        drawImage(image, 0, 0, dstX, dstY, image->getWidth(), image->getHeight());
    }

    void CGGraphics::drawPoint(int x, int y)
    {
        if (!gfx_context) throw GCN_EXCEPTION("drawPoint: no gfx_context");
        transformPoint(x, y);
        gfx_context->plotPixel(x, y, getCoercriColor());
    }

    void CGGraphics::drawLine(int x1, int y1, int x2, int y2)
    {
        if (!gfx_context) throw GCN_EXCEPTION("drawLine: no gfx_context");
        transformPoint(x1, y1);
        transformPoint(x2, y2);
        gfx_context->drawLine(x1, y1, x2, y2, getCoercriColor());
    }

    void CGGraphics::drawRectangle(const gcn::Rectangle &rectangle)
    {
        if (!gfx_context) throw GCN_EXCEPTION("drawRectangle: no gfx_context");
        int x = rectangle.x, y = rectangle.y;
        transformPoint(x, y);
        Coercri::Rectangle rect(x, y, rectangle.width, rectangle.height);
        gfx_context->drawRectangle(rect, getCoercriColor());
    }

    void CGGraphics::fillRectangle(const gcn::Rectangle &rectangle)
    {
        if (!gfx_context) throw GCN_EXCEPTION("fillRectangle: no gfx_context");
        int x = rectangle.x, y = rectangle.y;
        transformPoint(x, y);        
        Coercri::Rectangle rect(x, y, rectangle.width, rectangle.height);
        gfx_context->fillRectangle(rect, getCoercriColor());
    }

    void CGGraphics::setColor(const gcn::Color &color)
    {
        curr_col = color;
    }

    const gcn::Color & CGGraphics::getColor() const
    {
        return curr_col;
    }

    void CGGraphics::transformPoint(int &x, int &y) const
    {
        if (!mClipStack.empty()) {
            x += mClipStack.top().xOffset;
            y += mClipStack.top().yOffset;
        }
    }
}
