/*
 * FILE:
 *   enet_udp_socket.cpp
 *
 * AUTHOR:
 *   Stephen Thompson, 2008
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *
 */

#include "enet_udp_socket.hpp"
#include "../core/coercri_error.hpp"

namespace Coercri {

    EnetUDPSocket::EnetUDPSocket(int p, bool reuseaddr)
        : bound_port(p)
    {
        sock = enet_socket_create(ENET_SOCKET_TYPE_DATAGRAM);

        if (sock == ENET_SOCKET_NULL) throw CoercriError("enet_socket_create failed");

        enet_socket_set_option(sock, ENET_SOCKOPT_REUSEADDR, reuseaddr ? 1 : 0);
        enet_socket_set_option(sock, ENET_SOCKOPT_NONBLOCK, 1);
        enet_socket_set_option(sock, ENET_SOCKOPT_BROADCAST, 1);

        if (bound_port != -1) {
            ENetAddress addr;
            addr.host = ENET_HOST_ANY;
            addr.port = bound_port;
            const int bind_result = enet_socket_bind(sock, &addr);
            if (bind_result != 0) throw CoercriError("enet_socket_bind failed");
        }
    }

    EnetUDPSocket::~EnetUDPSocket()
    {
        enet_socket_destroy(sock);
    }

    bool EnetUDPSocket::receive(std::string &address, std::string &msg)
    {
        ENetAddress addr;

        char mybuf[2048];
        ENetBuffer buffer;
        buffer.data = mybuf;
        buffer.dataLength = sizeof(mybuf);
        
        int num_bytes = enet_socket_receive(sock, &addr, &buffer, 1);

        if (num_bytes <= 0) {
            return false;
        } else {
            msg.assign(mybuf, num_bytes);
            enet_address_get_host_ip(&addr, mybuf, sizeof(mybuf));
            mybuf[sizeof(mybuf)-1] = 0;
            address = mybuf;
            return true;
        }
    }

    void EnetUDPSocket::send(const std::string &address, int port, const std::string &msg)
    {
        ENetAddress addr;
        enet_address_set_host(&addr, address.c_str());
        addr.port = port;

        ENetBuffer buffer;
        buffer.data = const_cast<char*>(msg.data());
        buffer.dataLength = msg.length();

        const int sent_length = enet_socket_send(sock, &addr, &buffer, 1);
        if (sent_length < 0) throw CoercriError("enet_socket_send failed");
    }
    
    void EnetUDPSocket::broadcast(int port, const std::string &msg)
    {
        ENetAddress addr;
        addr.host = ENET_HOST_BROADCAST;
        addr.port = port;

        ENetBuffer buffer;
        buffer.data = const_cast<char*>(msg.data());
        buffer.dataLength = msg.length();
        
        const int sent_length = enet_socket_send(sock, &addr, &buffer, 1);
        if (sent_length < 0) throw CoercriError("enet_socket_send failed");
    }    
}
