/*
 * FILE:
 *   enet_network_connection.cpp
 *
 * AUTHOR:
 *   Stephen Thompson, 2008
 *
 * COPYRIGHT:
 *   Usage of this file is permitted under the terms of the Boost
 *   Software License, version 1.0.
 *
 */

#include "enet_network_connection.hpp"

#include "../core/coercri_error.hpp"

namespace Coercri {

    EnetNetworkConnection::EnetNetworkConnection(ENetHost *host, const std::string &hostname, int port)
        : peer(0), state(PENDING)
    {
        ENetAddress address;
        enet_address_set_host(&address, hostname.c_str());
        address.port = port;

        peer = enet_host_connect(host, &address, 1);  // open 1 channel only
        if (!peer) {
            throw CoercriError("EnetNetworkConnection: enet_host_connect failed");
        }

        peer->data = this;
    }

    EnetNetworkConnection::EnetNetworkConnection(ENetPeer *peer_)
        : peer(peer_), state(CONNECTED)
    {
        peer->data = this;
    }

    EnetNetworkConnection::~EnetNetworkConnection()
    {
        while (!queued_packets.empty()) {
            enet_packet_destroy(queued_packets.front());
            queued_packets.pop();
        }
        if (peer) {
            // Use enet_peer_disconnect_now rather than enet_peer_reset
            // (the only real difference is that disconnect_now will take
            // one last shot at sending a DISCONNECT message before
            // resetting).
            enet_peer_disconnect_now(peer, 0);
            peer->data = 0;
        }
    }

    EnetNetworkConnection::State EnetNetworkConnection::getState() const
    {
        return state;
    }

    void EnetNetworkConnection::close()
    {
        if (peer && state != CLOSED && state != FAILED) {
            state = CLOSED;

            // use enet_peer_disconnect_later, this ensures any queued
            // packets are sent first.
            enet_peer_disconnect_later(peer, 0);
        }
        // NOTE: Keep the link to the peer open (ie dont zero out peer->data and peer),
        // as the peer is still "active" while we wait for the disconnect message.
    }

    void EnetNetworkConnection::send(const std::vector<unsigned char> &buf)
    {
        if (state == CONNECTED && peer) {
            ENetPacket *packet = enet_packet_create(&buf[0], buf.size(), ENET_PACKET_FLAG_RELIABLE);
            enet_peer_send(peer, 0, packet);
        } else if (state == PENDING) {
            // Can't send yet, but will send as soon as connection becomes established.
            outgoing_packets.push(buf);
        }
        // Packets send to a CLOSED or FAILED connection are silently dropped.
    }

    void EnetNetworkConnection::receive(std::vector<unsigned char> &buf)
    {
        buf.clear();
        while (!queued_packets.empty()) {
            unsigned char* ptr = static_cast<unsigned char*>(queued_packets.front()->data);
            std::copy(ptr, ptr + queued_packets.front()->dataLength, std::back_inserter(buf));
            enet_packet_destroy(queued_packets.front());
            queued_packets.pop();
        }
    }

    std::string EnetNetworkConnection::getAddress()
    {
        char buf[1024];
        enet_address_get_host_ip(&peer->address, buf, sizeof(buf));
        buf[sizeof(buf)-1] = 0;
        return std::string(buf);
    }

    void EnetNetworkConnection::onReceiveAcknowledgment()
    {
        state = CONNECTED;
        while (!outgoing_packets.empty()) {
            send(outgoing_packets.front());
            outgoing_packets.pop();
        }
    }

    void EnetNetworkConnection::onReceivePacket(ENetPacket *packet)
    {
        queued_packets.push(packet);
    }

    void EnetNetworkConnection::onDisconnect()
    {
        // This is called when we get an ENet disconnect event. This
        // means either an existing connection timed out or was
        // closed, or an attempt to establish a new connection timed
        // out or otherwise failed.
        if (state == CONNECTED) {
            state = CLOSED;
        } else {
            state = FAILED;
        }

        // Break the link to the peer.
        peer->data = 0;
        peer = 0;
    }
}
