/*
 * knights_client.hpp
 *
 * Class representing the connection from a client to a Knights
 * server.
 *
 * Messages can be sent to the server by calling the methods of this
 * class. This will put data into the output buffer, this should then
 * be read with getOutputData() and sent to the server.
 *
 * When data is received from the server it should be processed by
 * calling processInputData(). This will convert any server commands
 * received into calls to the callback interfaces.
 *
 * Copyright (c) Stephen Thompson, 2009.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef KNIGHTS_CLIENT_HPP
#define KNIGHTS_CLIENT_HPP

#include "boost/noncopyable.hpp"

#include <vector>

class ClientCallbacks;
class KnightsCallbacks;
class KnightsClientImpl;
class UserControl;

class KnightsClient : boost::noncopyable {
public:
    typedef unsigned char ubyte;

    KnightsClient();
    ~KnightsClient();

    //
    // Methods to handle incoming/outgoing data
    //

    // process input data
    // this will generate calls to the callback interfaces, if set (see below).
    void receiveInputData(const std::vector<ubyte> &data);

    // get the output data (and then clear the output buffer).
    // caller is then responsible for sending this data to the server.
    // any existing contents of 'data' are deleted.
    void getOutputData(std::vector<ubyte> &data);

    // this should be called if the network link is closed or lost.
    void connectionClosed();

    // this is like connectionClosed but is called when the attempt to establish the outgoing connection fails.
    void connectionFailed();


    //
    // Get/Set Callbacks
    //

    ClientCallbacks* getClientCallbacks() const;
    void setClientCallbacks(ClientCallbacks *client_callbacks);
    KnightsCallbacks* getKnightsCallbacks() const;
    void setKnightsCallbacks(KnightsCallbacks *knights_callbacks);
    

    //
    // Methods to send commands to the server (will write data to the
    // output buffer)
    // 

    void setPlayerName(const std::string &plyr_name);   // should be 1st cmd sent
    void requestGameList();   // request list of available games
    void joinGame(const std::string &game_name);   // attempt to join a game.
    void joinGameSplitScreen(const std::string &game_name);  // used for split screen mode.
    void leaveGame();   // attempt to leave a game (go back to "unjoined" state).

    void sendChatMessage(const std::string &msg);

    void setReady(bool ready);  // change my ready status (used in menu screen)
    void setHouseColour(int hse_col);
    void changePlayerNum(int player_num);   // attempt to set player num (0 or 1 for player, -1 for observer)
    void setMenuSelection(const std::string &key, int value);  // set a menu option (used in menu screen)

    void finishedLoading();  // tell server that we have finished loading and are ready to start game

    // send password to server
    // server will reply with initialPlayerList() if successful, or passwordRequested(false) if not.
    void sendPassword(const std::string &password);
    
    // Send a control to my knight
    // Note: continuous controls will act until cancelled (by a
    // sendControl(0);) while discontinuous controls will act once
    // only.
    void sendControl(int plyr, const UserControl *ctrl);

    void readyToEnd();   // tell server that we are ready to exit winner/loser screen
    void requestQuit();  // tell server that we want to quit game (go back to lobby).

    void setPauseMode(bool);  // only works for split screen games currently.

private:
    std::auto_ptr<KnightsClientImpl> pimpl;
};

#endif
