/*
 * knights_client.cpp
 *
 * Copyright (c) Stephen Thompson, 2009.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#include "misc.hpp"

#include "anim.hpp"
#include "client_callbacks.hpp"
#include "client_config.hpp"
#include "colour_change.hpp"
#include "dungeon_view.hpp"
#include "graphic.hpp"
#include "mini_map.hpp"
#include "overlay.hpp"
#include "sound.hpp"
#include "status_display.hpp"
#include "knights_callbacks.hpp"
#include "knights_client.hpp"
#include "protocol.hpp"
#include "user_control.hpp"
#include "version.hpp"

#include "network/byte_buf.hpp"  // coercri

namespace {
    void ReadRoomCoord(Coercri::InputByteBuf &buf, int &x, int &y)
    {
        buf.readNibbles(x, y);
        x--;
        y--;
    }

    void ReadTileInfo(Coercri::InputByteBuf &buf, int &depth, bool &cc)
    {
        int x = buf.readUbyte();
        cc = (x & 128) != 0;
        depth = (x & 127) - 64;
    }
}

class KnightsClientImpl {
public:
    // ctor
    KnightsClientImpl() : ndisplays(0), player(0), 
                          knights_callbacks(0), client_callbacks(0)
    { 
        for (int i = 0; i < 2; ++i) last_cts_ctrl[i] = 0;
    }

    // data
    int ndisplays;
    int player;         // currently 'selected' display
    std::vector<unsigned char> out;
    boost::shared_ptr<ClientConfig> client_config;
    KnightsCallbacks *knights_callbacks;
    ClientCallbacks *client_callbacks;
    const UserControl *last_cts_ctrl[2];
    
    // helper functions
    void receiveConfiguration(Coercri::InputByteBuf &buf);
    const Graphic * readGraphic(Coercri::InputByteBuf &buf) const;
    const Anim * readAnim(Coercri::InputByteBuf &buf) const;
    const Overlay * readOverlay(Coercri::InputByteBuf &buf) const;
    const Sound * readSound(Coercri::InputByteBuf &buf) const;
    const UserControl * getControl(int id) const;
};

KnightsClient::KnightsClient()
    : pimpl(new KnightsClientImpl)
{
    // Write the initial version string.
    Coercri::OutputByteBuf buf(pimpl->out);
    buf.writeString("Knights/" KNIGHTS_VERSION);
}

KnightsClient::~KnightsClient()
{
}

void KnightsClient::setClientCallbacks(ClientCallbacks *c)
{
    pimpl->client_callbacks = c;
}

ClientCallbacks* KnightsClient::getClientCallbacks() const
{
    return pimpl->client_callbacks;
}

void KnightsClient::setKnightsCallbacks(KnightsCallbacks *c)
{
    pimpl->knights_callbacks = c;
}

KnightsCallbacks* KnightsClient::getKnightsCallbacks() const
{
    return pimpl->knights_callbacks;
}

void KnightsClient::receiveInputData(const std::vector<ubyte> &data)
{
    // This is where we decode the messages coming from the server,
    // and turn them into calls to ClientCallbacks.

    Coercri::InputByteBuf buf(data);

    // some aliases (to save typing):
    ClientCallbacks * client_cb = pimpl->client_callbacks;
    KnightsCallbacks * knights_cb = pimpl->knights_callbacks;
    DungeonView * dungeon_view = knights_cb ? &knights_cb->getDungeonView(pimpl->player) : 0;
    MiniMap * mini_map = knights_cb ? &knights_cb->getMiniMap(pimpl->player) : 0;
    StatusDisplay * status_display = knights_cb ? &knights_cb->getStatusDisplay(pimpl->player) : 0;
    
    while (!buf.eof()) {
        const unsigned char msg_code = buf.readUbyte();
        switch (msg_code) {
        case SERVER_ERROR:
            {
                const std::string err = buf.readString();
                if (client_cb) client_cb->serverError(err);
            }
            break;
            
        case SERVER_GAME_LIST:
            {
                std::vector<GameInfo> game_infos;
                int ngames = buf.readVarInt();
                game_infos.reserve(ngames);
                for (int i = 0; i < ngames; ++i) {
                    GameInfo gi;
                    gi.game_name = buf.readString();
                    gi.player_name[0] = buf.readString();
                    gi.player_name[1] = buf.readString();
                    gi.num_observers = buf.readVarInt();
                    gi.status = GameStatus(buf.readVarInt());
                    game_infos.push_back(gi);
                }
                if (client_cb) client_cb->gameListReceived(game_infos);
            }
            break;

        case SERVER_JOIN_GAME_ACCEPTED:
            {
                pimpl->receiveConfiguration(buf);
                const int my_player_num = buf.readVarInt();
                
                const std::string player1 = buf.readString();
                const bool ready1 = buf.readUbyte() != 0;
                const int house_col_1 = buf.readUbyte();
                
                const std::string player2 = buf.readString();
                const bool ready2 = buf.readUbyte() != 0;
                const int house_col_2 = buf.readUbyte();
                
                const int n_obs = buf.readVarInt();
                std::vector<std::string> observers;
                if (n_obs < 0) throw ProtocolError("n_observers incorrect");
                observers.reserve(n_obs);
                for (int i = 0; i < n_obs; ++i) {
                    observers.push_back(buf.readString());
                }
                if (client_cb) client_cb->joinGameAccepted(pimpl->client_config,
                                                           my_player_num,
                                                           player1, ready1, house_col_1,
                                                           player2, ready2, house_col_2,
                                                           observers);
            }
            break;

        case SERVER_JOIN_GAME_DENIED:
            {
                const std::string reason = buf.readString();
                if (client_cb) client_cb->joinGameDenied(reason);
            }
            break;

        case SERVER_INITIAL_PLAYER_LIST:
            {
                const int nplayers = buf.readVarInt();
                std::vector<std::pair<std::string, std::string> > names;
                for (int i = 0; i < nplayers; ++i) {
                    const std::string player_name = buf.readString();
                    const std::string game_name = buf.readString();
                    names.push_back(std::make_pair(player_name, game_name));
                }
                if (client_cb) client_cb->initialPlayerList(names);
            }
            break;

        case SERVER_PLAYER_CONNECTED:
            {
                const std::string name = buf.readString();
                if (client_cb) client_cb->playerConnected(name);
            }
            break;

        case SERVER_PLAYER_DISCONNECTED:
            {
                const std::string name = buf.readString();
                if (client_cb) client_cb->playerDisconnected(name);
            }
            break;

        case SERVER_LEAVE_GAME:
            if (client_cb) client_cb->leaveGame();
            break;
            
        case SERVER_SET_MENU_SELECTION:
            {
                const std::string key = buf.readString();
                const int val = buf.readVarInt();
                std::vector<int> allowed_vals;
                const int num_allowed_vals = buf.readVarInt();
                allowed_vals.resize(num_allowed_vals);
                for (int i = 0; i < num_allowed_vals; ++i) {
                    allowed_vals[i] = buf.readVarInt();
                }
                if (client_cb) client_cb->setMenuSelection(key, val, allowed_vals);
            }
            break;

        case SERVER_SET_QUEST_DESCRIPTION:
            {
                const std::string quest_descr = buf.readString();
                if (client_cb) client_cb->setQuestDescription(quest_descr);
            }
            break;
            
        case SERVER_START_GAME:
            {
                pimpl->ndisplays = buf.readUbyte();
                pimpl->player = 0;
                if (client_cb) client_cb->startGame(pimpl->ndisplays);
            }
            break;

        case SERVER_GOTO_MENU:
            if (client_cb) client_cb->gotoMenu();
            break;

        case SERVER_PLAYER_JOINED_THIS_GAME:
            {
                const std::string name = buf.readString();
                const int pnum = buf.readVarInt();
                const int house_col = buf.readUbyte();
                if (client_cb) client_cb->playerJoinedThisGame(name, pnum, house_col);
            }
            break;

        case SERVER_PLAYER_LEFT_THIS_GAME:
            {
                const std::string name = buf.readString();
                if (client_cb) client_cb->playerLeftThisGame(name);
            }
            break;

        case SERVER_SET_READY:
            {
                const std::string name = buf.readString();
                const int ready = buf.readUbyte();
                if (client_cb) client_cb->setReady(name, ready != 0);
            }
            break;

        case SERVER_SET_HOUSE_COLOUR:
            {
                const std::string name = buf.readString();
                const int x = buf.readUbyte();
                if (client_cb) client_cb->setPlayerHouseColour(name, x);
            }
            break;

        case SERVER_SET_AVAILABLE_HOUSE_COLOURS:
            {
                const int n = buf.readUbyte();
                std::vector<Coercri::Color> cols;
                cols.reserve(n);
                for (int i = 0; i < n; ++i) {
                    const unsigned char r = buf.readUbyte();
                    const unsigned char g = buf.readUbyte();
                    const unsigned char b = buf.readUbyte();
                    cols.push_back(Coercri::Color(r,g,b));
                }
                if (client_cb) client_cb->setAvailableHouseColours(cols);
            }
            break;

        case SERVER_PLAYER_JOINED_GAME:
            {
                const std::string player = buf.readString();
                const std::string game = buf.readString();
                if (client_cb) client_cb->playerJoinedGame(player, game);
            }
            break;

        case SERVER_PLAYER_LEFT_GAME:
            {
                const std::string player = buf.readString();
                const std::string game = buf.readString();
                if (client_cb) client_cb->playerLeftGame(player, game);
            }
            break;

        case SERVER_SET_PLAYER_NUM:
            {
                const std::string player = buf.readString();
                const int new_num = buf.readVarInt();
                const bool is_me = (buf.readUbyte() != 0);
                if (client_cb) client_cb->setPlayerNum(player, new_num, is_me);
            }
            break;

        case SERVER_CHAT:
            {
                const std::string whofrom = buf.readString();

                // we currently don't parse all the chat codes, we only look for 2 (Observer)
                // and use this to print "(Observer)" after the player's name...
                const bool is_observer = (buf.readUbyte() == 2);
                const std::string msg = buf.readString();
                if (client_cb) client_cb->chat(whofrom, is_observer, msg);
            }
            break;

        case SERVER_ANNOUNCEMENT:
            {
                const std::string msg = buf.readString();
                if (client_cb) client_cb->announcement(msg);
            }
            break;

        case SERVER_REQUEST_PASSWORD:
            {
                const bool first_attempt = buf.readUbyte() != 0;
                if (client_cb) client_cb->passwordRequested(first_attempt);
            }
            break;

        case SERVER_PLAY_SOUND:
            {
                const Sound *sound = pimpl->readSound(buf);
                const int freq = buf.readVarInt();
                if (knights_cb && sound) knights_cb->playSound(pimpl->player, *sound, freq);
            }
            break;

        case SERVER_WIN_GAME:
            if (knights_cb) knights_cb->winGame(pimpl->player);
            break;

        case SERVER_LOSE_GAME:
            if (knights_cb) knights_cb->loseGame(pimpl->player);
            break;

        case SERVER_SET_AVAILABLE_CONTROLS:
            {
                const int n = buf.readUbyte();
                std::vector<std::pair<const UserControl*, bool> > avail;
                avail.reserve(n);
                for (int i = 0; i < n; ++i) {
                    const int x = buf.readUbyte();
                    const UserControl * ctrl = pimpl->getControl(x & 0x7f);
                    const bool primary = (x & 0x80) != 0;
                    avail.push_back(std::make_pair(ctrl, primary));
                }
                if (knights_cb) knights_cb->setAvailableControls(pimpl->player, avail);
            }
            break;

        case SERVER_SET_MENU_HIGHLIGHT:
            {
                const UserControl * ctrl = pimpl->getControl(buf.readUbyte());
                if (knights_cb) knights_cb->setMenuHighlight(pimpl->player, ctrl);
            }
            break;

        case SERVER_FLASH_SCREEN:
            {
                const int delay = buf.readVarInt();
                if (knights_cb) knights_cb->flashScreen(pimpl->player, delay);
            }
            break;

        case SERVER_SET_CURRENT_ROOM:
            {
                const int room = buf.readVarInt();
                int width, height;
                ReadRoomCoord(buf, width, height);
                if (dungeon_view) dungeon_view->setCurrentRoom(room, width, height);
            }
            break;

        case SERVER_ADD_ENTITY:
            {
                const int id = buf.readVarInt();
                int x, y;
                ReadRoomCoord(buf, x, y);
                int ht, facing;
                buf.readNibbles(ht, facing);
                const Anim *anim = pimpl->readAnim(buf);
                const Overlay *overlay = pimpl->readOverlay(buf);
                int af, z;
                buf.readNibbles(af, z);
                const MotionType motion_type = MotionType(z >> 1);
                const bool ainvuln = ((z & 1) != 0);
                const int atz_diff = af == 0 ? 0 : buf.readShort();
                const int cur_ofs = buf.readUshort();
                const int motion_time_remaining = motion_type == MT_NOT_MOVING ? 0 : buf.readUshort();
                if (dungeon_view) dungeon_view->addEntity(id, x, y, MapHeight(ht), MapDirection(facing),
                                                          anim, overlay, af, atz_diff, ainvuln,
                                                          cur_ofs, motion_type, motion_time_remaining);
            }
            break;

        case SERVER_RM_ENTITY:
            {
                const int id = buf.readVarInt();
                if (dungeon_view) dungeon_view->rmEntity(id);
            }
            break;

        case SERVER_REPOSITION_ENTITY:
            {
                const int id = buf.readVarInt();
                int x, y;
                ReadRoomCoord(buf, x, y);
                if (dungeon_view) dungeon_view->repositionEntity(id, x, y);
            }
            break;

        case SERVER_MOVE_ENTITY:
            {
                const int id = buf.readVarInt();
                int motion_type, missile_mode;
                buf.readNibbles(motion_type, missile_mode);
                const int motion_duration = buf.readUshort();
                if (dungeon_view) dungeon_view->moveEntity(id, MotionType(motion_type), motion_duration, missile_mode!=0);
            }
            break;

        case SERVER_FLIP_ENTITY_MOTION:
            {
                const int id = buf.readVarInt();
                const int initial_delay = buf.readUshort();
                const int motion_duration = buf.readUshort();
                if (dungeon_view) dungeon_view->flipEntityMotion(id, initial_delay, motion_duration);
            }
            break;

        case SERVER_SET_ANIM_DATA:
            {
                const int id = buf.readVarInt();
                const Anim *anim = pimpl->readAnim(buf);
                const Overlay *overlay = pimpl->readOverlay(buf);
                int af, z;
                buf.readNibbles(af, z);
                const bool ainvuln = ((z & 2) != 0);
                const bool currently_moving = ((z & 1) != 0);
                const int atz_diff = buf.readShort();
                if (dungeon_view) dungeon_view->setAnimData(id, anim, overlay, af, atz_diff, ainvuln, currently_moving);
            }
            break;

        case SERVER_SET_FACING:
            {
                const int id = buf.readVarInt();
                const MapDirection facing = MapDirection(buf.readUbyte());
                if (dungeon_view) dungeon_view->setFacing(id, facing);
            }
            break;

        case SERVER_CLEAR_TILES:
            {
                int x, y;
                ReadRoomCoord(buf, x, y);
                if (dungeon_view) dungeon_view->clearTiles(x, y);
            }
            break;

        case SERVER_SET_TILE:
            {
                int x, y;
                ReadRoomCoord(buf, x, y);
                int depth;
                bool cc_set;
                ReadTileInfo(buf, depth, cc_set);
                const Graphic *graphic = pimpl->readGraphic(buf);
                boost::shared_ptr<ColourChange> cc;
                if (cc_set) cc.reset(new ColourChange(buf));
                if (dungeon_view) dungeon_view->setTile(x, y, depth, graphic, cc);
            }
            break;

        case SERVER_SET_ITEM:
            {
                int x, y;
                ReadRoomCoord(buf, x, y);
                const Graphic * graphic = pimpl->readGraphic(buf);
                if (dungeon_view) dungeon_view->setItem(x, y, graphic);
            }
            break;

        case SERVER_PLACE_ICON:
            {
                int x, y;
                ReadRoomCoord(buf, x, y);
                const Graphic * graphic = pimpl->readGraphic(buf);
                const int duration = buf.readUshort();
                if (dungeon_view) dungeon_view->placeIcon(x, y, graphic, duration);
            }
            break;

        case SERVER_FLASH_MESSAGE:
            {
                const std::string msg = buf.readString();
                const int ntimes = buf.readUbyte();
                if (dungeon_view) dungeon_view->flashMessage(msg, ntimes);
            }
            break;

        case SERVER_CANCEL_CONTINUOUS_MESSAGES:
            if (dungeon_view) dungeon_view->cancelContinuousMessages();
            break;

        case SERVER_ADD_CONTINUOUS_MESSAGE:
            {
                const std::string msg = buf.readString();
                if (dungeon_view) dungeon_view->addContinuousMessage(msg);
            }
            break;

        case SERVER_SET_MAP_SIZE:
            {
                const int width = buf.readUbyte();
                const int height = buf.readUbyte();
                if (mini_map) mini_map->setSize(width, height);
            }
            break;

        case SERVER_SET_COLOUR:
            {
                const int nruns = buf.readVarInt();
                for (int i = 0; i < nruns; ++i) {
                    const int start_x = buf.readUbyte();
                    const int y = buf.readUbyte();
                    const int nx = buf.readUbyte();
                    for (int x = start_x; x < start_x + nx; ++x) {
                        const MiniMapColour col = MiniMapColour(buf.readUbyte());
                        if (mini_map) mini_map->setColour(x, y, col);
                    }
                }
            }
            break;

        case SERVER_WIPE_MAP:
            if (mini_map) mini_map->wipeMap();
            break;

        case SERVER_MAP_KNIGHT_LOCATION:
            {
                const int knight_id = buf.readUbyte();
                const int x = buf.readUbyte();
                if (x == 255) {
                    if (mini_map) mini_map->mapKnightLocation(knight_id, -1, -1);
                } else {
                    const int y = buf.readUbyte();
                    if (mini_map) mini_map->mapKnightLocation(knight_id, x, y);
                }
            }
            break;

        case SERVER_MAP_ITEM_LOCATION:
            {
                const int x = buf.readUbyte();
                const int y = buf.readUbyte();
                const bool flag = buf.readUbyte() != 0;
                if (mini_map) mini_map->mapItemLocation(x, y, flag);
            }
            break;

        case SERVER_SET_BACKPACK:
            {
                const int slot = buf.readUbyte();
                const Graphic * gfx = pimpl->readGraphic(buf);
                const Graphic * ovr = pimpl->readGraphic(buf);
                const int no_carried = buf.readUbyte();
                const int no_max = buf.readUbyte();
                if (status_display) status_display->setBackpack(slot, gfx, ovr, no_carried, no_max);
            }
            break;

        case SERVER_ADD_SKULL:
            if (status_display) status_display->addSkull();
            break;

        case SERVER_SET_HEALTH:
            {
                const int h = buf.readVarInt();
                if (status_display) status_display->setHealth(h);
            }
            break;

        case SERVER_SET_POTION_MAGIC:
            {
                const int x = buf.readUbyte();
                if (status_display) status_display->setPotionMagic(PotionMagic(x & 0x7f), (x & 0x80) != 0);
            }
            break;

        case SERVER_SET_QUEST_MESSAGE:
            {
                const std::string msg = buf.readString();
                if (status_display) status_display->setQuestMessage(msg);
            }
            break;

        case SERVER_SET_QUEST_ICONS:
            {
                int num_icons = buf.readUbyte();
                std::vector<StatusDisplay::QuestIconInfo> icons;
                icons.reserve(num_icons);
                for (int i = 0; i < num_icons; ++i) {
                    StatusDisplay::QuestIconInfo qi;
                    qi.num_held = buf.readUbyte();
                    qi.num_required = buf.readUbyte();
                    qi.gfx_missing = pimpl->readGraphic(buf);
                    qi.gfx_held = pimpl->readGraphic(buf);
                    icons.push_back(qi);
                }
                if (status_display) status_display->setQuestIcons(icons);
            }
            break;
            
        case SERVER_SWITCH_PLAYER:
            {
                const int new_player = buf.readUbyte();
                if (new_player >= pimpl->ndisplays) throw ProtocolError("display number out of range");
                pimpl->player = new_player;
                if (knights_cb) {
                    dungeon_view = &knights_cb->getDungeonView(new_player);
                    mini_map = &knights_cb->getMiniMap(new_player);
                    status_display = &knights_cb->getStatusDisplay(new_player);
                }
            }
            break;
            
        default:
            throw ProtocolError("Unknown message code from server");
        }
    }
}

void KnightsClient::getOutputData(std::vector<ubyte> &data)
{
    data.swap(pimpl->out);
    pimpl->out.clear();
}

void KnightsClient::connectionClosed()
{
    if (pimpl->client_callbacks) pimpl->client_callbacks->connectionLost();
}

void KnightsClient::connectionFailed()
{
    if (pimpl->client_callbacks) pimpl->client_callbacks->connectionFailed();
}

void KnightsClient::setPlayerName(const std::string &name)
{
    Coercri::OutputByteBuf buf(pimpl->out);
    buf.writeUbyte(CLIENT_SET_PLAYER_NAME);
    buf.writeString(name);
}

void KnightsClient::requestGameList()
{
    pimpl->out.push_back(CLIENT_REQUEST_GAME_LIST);
}

void KnightsClient::joinGame(const std::string &game_name)
{
    Coercri::OutputByteBuf buf(pimpl->out);
    buf.writeUbyte(CLIENT_JOIN_GAME);
    buf.writeString(game_name);
}

void KnightsClient::joinGameSplitScreen(const std::string &game_name)
{
    Coercri::OutputByteBuf buf(pimpl->out);
    buf.writeUbyte(CLIENT_JOIN_GAME_SPLIT_SCREEN);
    buf.writeString(game_name);
}

void KnightsClient::leaveGame()
{
    pimpl->out.push_back(CLIENT_LEAVE_GAME);
}

void KnightsClient::sendChatMessage(const std::string &msg)
{
    Coercri::OutputByteBuf buf(pimpl->out);
    buf.writeUbyte(CLIENT_CHAT);
    buf.writeString(msg);
}

void KnightsClient::setReady(bool ready)
{
    Coercri::OutputByteBuf buf(pimpl->out);
    buf.writeUbyte(CLIENT_SET_READY);
    buf.writeUbyte(ready);
}

void KnightsClient::setHouseColour(int x)
{
    Coercri::OutputByteBuf buf(pimpl->out);
    buf.writeUbyte(CLIENT_SET_HOUSE_COLOUR);
    buf.writeUbyte(x);
}

void KnightsClient::changePlayerNum(int x)
{
    Coercri::OutputByteBuf buf(pimpl->out);
    buf.writeUbyte(CLIENT_CHANGE_PLAYER_NUM);
    buf.writeVarInt(x);
}

void KnightsClient::setMenuSelection(const std::string &key, int value)
{
    Coercri::OutputByteBuf buf(pimpl->out);
    buf.writeUbyte(CLIENT_SET_MENU_SELECTION);
    buf.writeString(key);
    buf.writeVarInt(value);
}

void KnightsClient::finishedLoading()
{
    pimpl->out.push_back(CLIENT_FINISHED_LOADING);
}

void KnightsClient::sendPassword(const std::string &password)
{
    Coercri::OutputByteBuf buf(pimpl->out);
    buf.writeUbyte(CLIENT_SEND_PASSWORD);
    buf.writeString(password);
}

void KnightsClient::sendControl(int plyr, const UserControl *ctrl)
{
    int id = 0;
    if (ctrl) {
        id = ctrl->getID();
        if (id < 1 || id > 127) throw ProtocolError("control ID out of range (sendControl)");
    }
    if (plyr < 0 || plyr > 1) throw ProtocolError("player number out of range (sendControl)");

    // optimization: do not send "repeats" of continuous controls.
    if (ctrl == 0 || ctrl->isContinuous()) {
        if (ctrl == pimpl->last_cts_ctrl[plyr]) return;
        pimpl->last_cts_ctrl[plyr] = ctrl;
    }
    
    Coercri::OutputByteBuf buf(pimpl->out);
    buf.writeUbyte(CLIENT_SEND_CONTROL);

    if (plyr == 1) id += 128;
    buf.writeUbyte(id);
}

void KnightsClient::readyToEnd()
{
    pimpl->out.push_back(CLIENT_READY_TO_END);
}

void KnightsClient::requestQuit()
{
    pimpl->out.push_back(CLIENT_QUIT);
}

void KnightsClient::setPauseMode(bool p)
{
    pimpl->out.push_back(CLIENT_SET_PAUSE_MODE);
    pimpl->out.push_back(p ? 1 : 0);
}

void KnightsClientImpl::receiveConfiguration(Coercri::InputByteBuf &buf)
{
    client_config.reset(new ClientConfig);  // wipe out the old client config if there is one.
    
    const int n_graphics = buf.readVarInt();
    client_config->graphics.reserve(n_graphics);
    for (int i = 0; i < n_graphics; ++i) {
        client_config->graphics.push_back(new Graphic(i+1, buf));
    }

    const int n_anims = buf.readVarInt();
    client_config->anims.reserve(n_anims);
    for (int i = 0; i < n_anims; ++i) {
        client_config->anims.push_back(new Anim(i+1, buf, client_config->graphics));
    }

    const int n_overlays = buf.readVarInt();
    client_config->overlays.reserve(n_overlays);
    for (int i = 0; i < n_overlays; ++i) {
        client_config->overlays.push_back(new Overlay(i+1, buf, client_config->graphics));
    }

    const int n_sounds = buf.readVarInt();
    client_config->sounds.reserve(n_sounds);
    for (int i = 0; i < n_sounds; ++i) {
        client_config->sounds.push_back(new Sound(i+1, buf));
    }

    const int n_standard_controls = buf.readVarInt();
    client_config->standard_controls.reserve(n_standard_controls);
    for (int i = 0; i < n_standard_controls; ++i) {
        client_config->standard_controls.push_back(new UserControl(i+1, buf, client_config->graphics));
    }
    
    const int n_other_controls = buf.readVarInt();
    client_config->other_controls.reserve(n_other_controls);
    for (int i = 0; i < n_other_controls; ++i) {
        client_config->other_controls.push_back(new UserControl(i+1+n_standard_controls, buf, client_config->graphics));
    }

    client_config->menu.reset(new Menu(buf));

    client_config->approach_offset = buf.readVarInt();
}

const Graphic * KnightsClientImpl::readGraphic(Coercri::InputByteBuf &buf) const
{
    const int id = buf.readVarInt();
    if (id < 0 || id > client_config->graphics.size()) throw ProtocolError("graphic id out of range");
    if (id == 0) return 0;
    return client_config->graphics.at(id-1);
}

const Anim * KnightsClientImpl::readAnim(Coercri::InputByteBuf &buf) const
{
    const int id = buf.readVarInt();
    if (id < 0 || id > client_config->anims.size()) throw ProtocolError("anim id out of range");
    if (id == 0) return 0;
    return client_config->anims.at(id-1);
}

const Overlay * KnightsClientImpl::readOverlay(Coercri::InputByteBuf &buf) const
{
    const int id = buf.readVarInt();
    if (id < 0 || id > client_config->overlays.size()) throw ProtocolError("overlay id out of range");
    if (id == 0) return 0;
    return client_config->overlays.at(id-1);
}

const Sound * KnightsClientImpl::readSound(Coercri::InputByteBuf &buf) const
{
    const int id = buf.readVarInt();
    if (id < 0 || id > client_config->sounds.size()) throw ProtocolError("sound id out of range");
    if (id == 0) return 0;
    return client_config->sounds.at(id-1);
}

const UserControl * KnightsClientImpl::getControl(int id) const
{
    if (id == 0) return 0;
    if (id <= client_config->standard_controls.size()) {
        return client_config->standard_controls.at(id-1);
    } else if (id <= client_config->standard_controls.size() + client_config->other_controls.size()) {
        return client_config->other_controls.at(id - 1 - client_config->standard_controls.size()); 
    } else {
        throw ProtocolError("control id out of range");
    }
}
