/*
 * client_callbacks.hpp
 *
 * This must be implemented by the client to handle responses from the
 * server. Each possible server response corresponds to a method call
 * on this interface.
 *
 * Copyright (c) Stephen Thompson, 2009.
 * Licensed for non-commercial use only. See LICENCE.txt for details.
 *
 */

#ifndef CLIENT_CALLBACKS_HPP
#define CLIENT_CALLBACKS_HPP

#include "game_info.hpp"

#include "gfx/color.hpp" // coercri

#include "boost/shared_ptr.hpp"
#include <string>
#include <vector>

class ClientConfig;

class ClientCallbacks {
public:
    virtual ~ClientCallbacks() { }

    //
    // Callbacks that can be received at any time
    //

    // connection is being closed (or was never successfully opened)
    virtual void connectionLost() = 0;
    virtual void connectionFailed() = 0;

    // special
    virtual void serverError(const std::string &error) = 0;
    
    
    //
    // "Pre-join-game" callbacks
    //
    
    // game list
    virtual void gameListReceived(const std::vector<GameInfo> &games) = 0;

    // join game success/failure
    // NOTE: The returned ClientConfig will remain valid for as long as we are connected to the game
    virtual void joinGameAccepted(boost::shared_ptr<const ClientConfig> conf,
                                  int my_player_num,
                                  const std::string &player1, bool rdy1, int house_col_1,
                                  const std::string &player2, bool rdy2, int house_col_2,
                                  const std::vector<std::string> &observers) = 0;
    virtual void joinGameDenied(const std::string &reason) = 0;

    // called when we first connect, if there are other players already present
    // (fst = player name, snd = game name)
    virtual void initialPlayerList(const std::vector<std::pair<std::string, std::string> > & players) = 0;

    // called if the server wants us to enter a password before proceeding.
    virtual void passwordRequested(bool first_attempt) = 0;
    
    // called when other players connect to or disconnect from the server.
    virtual void playerConnected(const std::string &name) = 0;
    virtual void playerDisconnected(const std::string &name) = 0;

    // called when players join or leave any game.
    virtual void playerJoinedGame(const std::string &player, const std::string &game) = 0;
    virtual void playerLeftGame(const std::string &player, const std::string &game) = 0;
    
    
    //
    // "Post-join-game" callbacks
    //

    // called when I leave the game.
    virtual void leaveGame() = 0;
    
    // menu
    virtual void setMenuSelection(const std::string &key, int val, const std::vector<int> &allowed_values) = 0;
    virtual void setQuestDescription(const std::string &quest_descr) = 0;

    // switching between menu and in-game states
    virtual void startGame(int ndisplays) = 0;
    virtual void gotoMenu() = 0;

    // called when players join/leave my current game, or change state.
    // player_num is 0 or 1 for active players, or -1 for observers.
    virtual void playerJoinedThisGame(const std::string &name, int player_num, int house_col) = 0;
    virtual void playerLeftThisGame(const std::string &name) = 0;
    virtual void playerChangedName(const std::string &new_name, int player_num) = 0;
    virtual void setPlayerHouseColour(const std::string &name, int house_col) = 0;
    virtual void setAvailableHouseColours(const std::vector<Coercri::Color> &cols) = 0;
    virtual void setReady(const std::string &name, bool ready) = 0;

    // called when a player, in the current game, changes his player_num.
    virtual void setPlayerNum(const std::string &name, int new_num, bool is_me) = 0;
    
    // chat, and "announcements".
    virtual void chat(const std::string &whofrom, bool observer, const std::string &msg) = 0;
    virtual void announcement(const std::string &msg) = 0;
};

#endif
