/**
  This file belong to the KMPlayer project, a movie player plugin for Konqueror
  Copyright (C) 2007  Koos Vriezen <koos.vriezen@gmail.com>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**/

#include <string.h>
#include <map>
#include <cairo.h>
#include <librsvg/rsvg.h>
#include <librsvg/rsvg-cairo.h>

#include "actor.h"
#include "kmplayercontrol.h"
#include "kmplayerprocess.h"
#include "viewarea.h"

using namespace KMPlayer;

namespace {
    typedef std::map <String, ImageDataPtrW> ImageDataMap;
    static ImageDataMap *image_data_map;
}

//------------------------%<----------------------------------------------------

ActorAgent::ActorAgent (Control *ctrl) : m_control (ctrl) {
    if (!image_data_map)
        image_data_map = new ImageDataMap;
}

ActorAgent::~ActorAgent () {
    if (image_data_map->size () == 0) {
        delete image_data_map;
        image_data_map = 0;
    }
}

//------------------------%<----------------------------------------------------

Actor::Actor (ActorAgent *manager, Node *node)
 : m_agent (manager), m_node (node) {
   //manager->medias ().push_back (this);
}

Actor::~Actor () {
    //m_agent->medias ().remove (this);
}

KDE_NO_EXPORT void Actor::dismiss () {
    delete this;
}

//------------------------%<----------------------------------------------------

static bool isPlayListMime (const String & mime) {
    const char * mimestr = (const char *) mime;
    //debugLog () << "isPlayListMime  mime:" << mimestr << endl;
    return mimestr && (!strcmp (mimestr, "audio/mpegurl") ||
            !strcmp (mimestr, "audio/x-mpegurl") ||
            !strncmp (mimestr, "video/x-ms", 10) ||
            !strncmp (mimestr, "audio/x-ms", 10) ||
            //!strcmp (mimestr, "video/x-ms-wmp") ||
            //!strcmp (mimestr, "video/x-ms-asf") ||
            //!strcmp (mimestr, "video/x-ms-wmv") ||
            //!strcmp (mimestr, "video/x-ms-wvx") ||
            //!strcmp (mimestr, "video/x-msvideo") ||
            !strcmp (mimestr, "video/x-real") ||
            !strcmp (mimestr, "audio/x-scpls") ||
            !strcmp (mimestr, "audio/x-shoutcast-stream") ||
            !strcmp (mimestr, "audio/x-pn-realaudio") ||
            !strcmp (mimestr, "audio/vnd.rn-realaudio") ||
            !strcmp (mimestr, "audio/m3u") ||
            !strcmp (mimestr, "audio/x-m3u") ||
            !strncmp (mimestr, "text/", 5) ||
            (!strncmp (mimestr, "application/", 12) &&
             strstr (mimestr + 12,"+xml")) ||
            !strncasecmp (mimestr, "application/smil", 16) ||
            !strncasecmp (mimestr, "application/xml", 15) ||
            !strcmp (mimestr, "application/news_reader") ||
            !strcmp (mimestr, "application/rss+xml") ||
            !strcmp (mimestr, "application/atom+xml") ||
            !strcmp (mimestr, "application/ram") ||
            !strcmp (mimestr, "image/vnd.rn-realpix") ||
            !strcmp (mimestr, "image/svg+xml") ||
            !strcmp (mimestr, "application/x-mplayer2"));
}

static bool isAudioOnlyMime (const String & mime) {
    const char * mimestr = (const char *) mime;
    return mimestr && (!strncmp ((const char *) mime, "audio/", 6));
}

MediaInfo::MediaInfo (Node *n, ActorAgent::ActorType t)
 : node (n), media (NULL), type (t), job (NULL) {
}

MediaInfo::~MediaInfo () {
    clearData ();
    if (media)
        media->dismiss ();
}

KDE_NO_EXPORT void MediaInfo::killWGet () {
    if (job) {
        job->kill (); // quiet, no result signal
        job = 0L;
        clearData (); // assume data is invalid
    }
}

/**
 * Gets contents from url and puts it in data
 */
KDE_NO_EXPORT bool MediaInfo::wget (const String &str) {
    clearData ();
    url = str;

    if (ActorAgent::Any == type || ActorAgent::Image == type) {
        ImageDataMap::iterator i = image_data_map->find (str);
        if (i != image_data_map->end ()) {
            media = new ImageActor (node, i->second);
            type = ActorAgent::Image;
            ready ();
            return true;
        }
    }

    Mrl *mrl = node->mrl ();
    if (mrl && (ActorAgent::Any == type || ActorAgent::AudioVideo == type))
    {
        mime = mrl->mimetype;
#ifdef __ARMEL__
        if (NpPlayer::isFlashMimeType (mime)) {
            ready ();
            return true; // FIXME
        }
#endif
    }

    URL uri (str);
    if (mime.isEmpty () && uri.isLocalFile ()) {
        setMimetype (MimeType::findByURL (uri));
        if (mrl)
            mrl->mimetype = mime;
    }

    bool only_playlist = false;
    bool maybe_playlist = false;
    if (ActorAgent::Audio == type || ActorAgent::AudioVideo == type) {
        only_playlist = true;
        if (!mime.isEmpty ())
            maybe_playlist = isPlayListMime (mime);
    }

    if (mrl) {
        if (mime.isEmpty ()) {
            for (Node *p = mrl->parentNode (); p; p = p->parentNode ()) {
                Mrl *m = p->mrl ();
                if (m && m->audio_only) {
                    mrl->audio_only = true;
                    break;
                }
            }
        } else {
            mrl->audio_only |= isAudioOnlyMime (mime);
        }
    }
    debugLog () << "Actor::wget " << str << endl;
    if (uri.isLocalFile ()) {
        for (Node *p = node->parentNode (); p; p = p->parentNode ()) {
            Mrl *m = p->mrl ();
            if (m && !m->src.isEmpty () &&
                    m->src != "Playlist://" &&
                    !URL (m->src).isLocalFile ()) {
                errorLog () << "local acces denied\n";
                ready ();
                return true;
            }
        }
        File file (uri.path ());
        if (file.exists () && file.open (IO_ReadOnly)) {
            if (only_playlist) {
                maybe_playlist &= file.size () < 2000000;
                if (maybe_playlist) {
                    char buf[256];
                    off_t nr = file.readBlock (buf, sizeof (buf) - 1);
                    if (nr > 0) {
                        buf[nr] = 0;
                        int sz = nr > 512 ? 512 : nr;
                        if (MimeType::isBufferBinaryData (buf, sz) ||
                                !strncmp (buf, "RIFF", 4))
                            maybe_playlist = false;
                    }
                }
                if (!maybe_playlist) {
                    ready ();
                    return true;
                }
            }
            data = file.readAll ();
            file.close ();
        }
        ready ();
        return true;
    }

    String protocol = uri.protocol ();
    if (protocol == "mms" || protocol == "rtsp" || protocol == "rtp" ||
            (only_playlist && !maybe_playlist && !mime.isEmpty ())) {
        ready ();
        return true;
    }

    job = asyncGet (this, url);
    job->start ();
    return false;
}

KDE_NO_EXPORT bool MediaInfo::readChildDoc () {
    TextStream textstream (&data);
    String line;
    NodePtr cur_elm = node;
    do {
        line = textstream.readLine ();
    } while (!line.isNull () && line.stripWhiteSpace ().isEmpty ());
    if (!line.isNull ()) {
        bool pls_groupfound =
            line.startsWith ("[") && line.endsWith ("]") &&
            line.mid (1, line.length () - 2).stripWhiteSpace () == "playlist";
        if ((pls_groupfound &&
                    cur_elm->mrl ()->mimetype.startsWith ("audio/")) ||
                cur_elm->mrl ()->mimetype == "audio/x-scpls") {
            int nr = -1;
            struct Entry {
                String url, title;
            } * entries = 0L;
            do {
                line = line.stripWhiteSpace ();
                if (!line.isEmpty ()) {
                    if (line.startsWith ("[") && line.endsWith ("]")) {
                        if (line.mid (1, line.length () - 2).stripWhiteSpace () == "playlist")
                            pls_groupfound = true;
                        else
                            break;
                    } else if (pls_groupfound) {
                        int eq_pos = line.indexOf (Char ('='));
                        if (eq_pos > 0) {
                            if (line.lower ().startsWith (String ("numberofentries"))) {
                                nr = line.mid (eq_pos + 1).stripWhiteSpace ().toInt ();
                                if (nr > 0 && nr < 1024)
                                    entries = new Entry[nr];
                                else
                                    nr = 0;
                            } else if (nr > 0) {
                                String ll = line.lower ();
                                if (ll.startsWith (String ("file"))) {
                                    int i = line.mid (4, eq_pos-4).toInt ();
                                    if (i > 0 && i <= nr)
                                        entries[i-1].url = line.mid (eq_pos + 1).stripWhiteSpace ();
                                } else if (ll.startsWith (String ("title"))) {
                                    int i = line.mid (5, eq_pos-5).toInt ();
                                    if (i > 0 && i <= nr)
                                        entries[i-1].title = line.mid (eq_pos + 1).stripWhiteSpace ();
                                }
                            }
                        }
                    }
                }
                line = textstream.readLine ();
            } while (!line.isNull ());
            NodePtr doc = node->document ();
            for (int i = 0; i < nr; i++)
                if (!entries[i].url.isEmpty ()) {
                    GenericURL *gen = new GenericURL (doc,
                            URL::decode_string (entries[i].url),
                            entries[i].title);
                    cur_elm->appendChild (gen);
                    gen->opener = cur_elm;
                }
            delete [] entries;
        } else if (line.stripWhiteSpace ().startsWith (Char ('<'))) {
            readXML (cur_elm, textstream, line);
            //cur_elm->normalize ();
        } else if (line.lower ().startsWith ("[reference]")) {
            NodePtr doc = node->document ();
            for (line = textstream.readLine ();
                    !line.isNull ();
                    line = textstream.readLine ()) {
                if (line.lower ().startsWith ("ref")) {
                    int p = line.indexOf ("=", 3);
                    if (p > -1) {
                        URL u (line.mid (p + 1));
                        if (u.protocol() == "http" &&
                                line.indexOf("MSWMExt=.asf") > -1)
                            cur_elm->appendChild (new GenericURL (doc,
                                        String ("mms") + u.url().mid (4)));
                        else if (u.protocol() == "mms")
                            cur_elm->appendChild (new GenericURL(doc, u.url()));
                    }
                }
            }
        } else if (line.lower () != "[reference]") {
            bool extm3u = line.startsWith ("#EXTM3U");
            String title;
            NodePtr doc = node->document ();
            if (extm3u)
                line = textstream.readLine ();
            while (!line.isNull ()) {
             /* TODO && m_document.size () < 1024 / * support 1k entries * /);*/
                String mrl = line.stripWhiteSpace ();
                if (line == "--stop--")
                    break;
                if (mrl.lower ().startsWith (String ("asf ")))
                    mrl = mrl.mid (4).stripWhiteSpace ();
                if (!mrl.isEmpty ()) {
                    if (extm3u && mrl.startsWith (Char ('#'))) {
                        if (line.startsWith ("#EXTINF:"))
                            title = line.mid (9);
                        else
                            title = mrl.mid (1);
                    } else if (!line.startsWith (Char ('#'))) {
                        GenericURL *gen = new GenericURL (doc, mrl, title);
                        cur_elm->appendChild (gen);
                        gen->opener = cur_elm;
                        title.truncate (0);
                    }
                }
                line = textstream.readLine ();
            }
        }
    }
    return !cur_elm->isPlayable ();
}

void MediaInfo::setMimetype (const String &mt)
{
    mime = mt;

    Mrl *mrl = node ? node->mrl () : NULL;
    if (mrl && mrl->mimetype.isEmpty ())
        mrl->mimetype = mt;

    if (!mt.isEmpty () && ActorAgent::Any == type) {
        if (mimetype ().startsWith ("image/"))
            type = ActorAgent::Image;
        else if (mime.startsWith ("audio/"))
            type = ActorAgent::Audio;
        else
            type = ActorAgent::AudioVideo;
    }
}

KDE_NO_EXPORT String MediaInfo::mimetype () {
    if (data.size () > 0 && mime.isEmpty ())
        mime = MimeType::findByContent (data.data (), data.size ());
    return mime;
}

KDE_NO_EXPORT void MediaInfo::clearData () {
    killWGet ();
    url.truncate (0);
    mime.truncate (0);
    data.resize (0);
}

KDE_NO_EXPORT bool MediaInfo::downloading () const {
    return !!job;
}

KDE_NO_EXPORT void MediaInfo::create () {
    ActorAgent *mgr = (ActorAgent *)node->document()->role(RoleActorAgent);
    if (!media) {
        switch (type) {
            case ActorAgent::Audio:
            case ActorAgent::AudioVideo:
                debugLog() << "create AudioVideo " << data.size () << endl;
                if (!data.size () || !readChildDoc ())
                    media = new AudioVideoActor (mgr, node);
                break;
            case ActorAgent::Image:
                if (data.size () && mime == "image/svg+xml") {
                    readChildDoc ();
                    if (node->firstChild () &&
                            id_node_svg == node->lastChild ()->id) {
                        media = new ImageActor (node);
                        break;
                    }
                }
                if (data.size () &&
                        (!(!MimeType::isBufferBinaryData (
                                data.data (),
                                data.size () > 512 ? 512 : data.size ()) ||
                           mime == "image/vnd.rn-realpix") ||
                         !readChildDoc ()))
                    media = new ImageActor (mgr, node, url, data);
                break;
            case ActorAgent::Text:
                if (data.size ())
                    media = new TextActor (mgr, node, data);
                break;
            default: // Any
                break;
        }
    }
    data = ByteArray ();
}

KDE_NO_EXPORT void MediaInfo::ready () {
    create ();
    node->document()->post (new Posting (node, node, MsgMediaReady));
}

static bool validDataFormat (ActorAgent::ActorType type, const ByteArray &ba) {
    int sz = ba.size () > 512 ? 512 : ba.size();

    switch (type) {
    case ActorAgent::Any:
        //TODO
    case ActorAgent::Audio:
    case ActorAgent::AudioVideo: {
        bool kill (ba.size () > 2000000 ||            // too big
                (ba.size () > 4 &&
                 !strncmp (ba.data (), "RIFF", 4))); // falsely seen as text
        if (!kill && ba.size() > 14) {
            int sz = ba.size () > 512 ? 512 : ba.size ();
            kill = MimeType::isBufferBinaryData (ba.data(), sz) ||
                (!strncasecmp (ba.data (), "<html", 5) || // html admin
                 !strncasecmp (ba.data (), "<!doctype html", 14));
        }
        return !kill;
    }
    default:
        return true;
    }
}

KDE_NO_EXPORT void MediaInfo::jobResult (IOJob *jb) {
    if (jb->error ())
        data.resize (0);
    job = 0L; // signal KIO::IOJob::result deletes itself
    if (mime.isEmpty ()) {
        if (data.size () > 0) {
            setMimetype (MimeType::findByContent (data.data(), data.size ()));
            if (!validDataFormat (type, data))
                data.resize (0);
        }
        if (node->mrl ()) {
            node->mrl ()->mimetype = mime;
            node->mrl ()->audio_only = isAudioOnlyMime (mime);
        }
    }
    ready ();
}

KDE_NO_EXPORT void MediaInfo::jobData (IOJob *jb, ByteArray &ba) {
    //debugLog () << "ActorTypeRuntime::jobData " << data.size () << endl;
    if (ba.size ()) {
        int old_size = data.size ();
        int newsize = old_size + ba.size ();

        data.resize (newsize);
        memcpy (data.data () + old_size, ba.data (), ba.size ());

        if (old_size < 512 && newsize >= 512) {
            setMimetype (MimeType::findByContent (data.data (), 512));
            if (!validDataFormat (type, data)) {
                data.resize (0);
                job->kill (false); // not quiet, wants jobResult
            }
        }
    }
}

KDE_NO_EXPORT void MediaInfo::redirected (IOJob *, const String &url) {
    Mrl *mrl = node ? node->mrl () : NULL;
    if (mrl)
        mrl->src = url;
}

//------------------------%<----------------------------------------------------

AudioVideoActor::AudioVideoActor (ActorAgent *manager, Node *node)
 : Actor (manager, node) {
}

AudioVideoActor::~AudioVideoActor () {
    stop ();
    debugLog() << "AudioVideoActor::~AudioVideoActor" << endl;
}

bool AudioVideoActor::play () {
    Process *process = m_agent->control ()->process ();
    if (process && process->playing () && process->mrl () == sourceMrl ()) {
        errorLog () << "play: already playing " << sourceMrl ()->src << endl;
    } else if (m_agent->control ()->requestPlayURL (sourceMrl ())) {
        if (!sourceMrl ()->audio_only)
            m_agent->control ()->viewArea ()->setAudioVideoNode (m_node);
        m_node->defer ();
        return true;
    }
    return false;
}

void AudioVideoActor::stop () {
    Process *process = m_agent->control ()->process ();
    if (process && process->mrlBackEndNotify () == this) {
        process->stop ();
        process->setMrlBackEndNotify (NULL);
    }
    m_agent->control ()->viewArea ()->setAudioVideoNode (NULL);
}

void AudioVideoActor::pause () {
    Process *process = m_agent->control ()->process ();
    if (process && process->playing ())
        process->pause ();
}

void AudioVideoActor::unpause () {
    Process *process = m_agent->control ()->process ();
    if (process && process->playing ())
        process->play (String ());
}

Mrl *AudioVideoActor::sourceMrl ()
{
    return m_node ? m_node->mrl () : NULL;
}

void AudioVideoActor::referenceMrl (const String &uri)
{
    Mrl *_mrl = sourceMrl ();
    if (_mrl && !uri.isEmpty ()) {
        for (Node *n = _mrl; n; n = n->parentNode ()) {
            Mrl *m = n->mrl ();
            if (m && m->src == uri)
                return;
        }
        for (Node *n = _mrl->firstChild (); n; n = n->nextSibling ()) {
            Mrl *m = n->mrl ();
            if (m && m->src == uri)
                return;
        }

        NodePtr doc = _mrl->document ();
        Node *n = new GenericURL (doc, uri, String ());
        n->id = id_node_ref_url;
        n->setAuxiliaryNode (true);
        _mrl->appendChild (n);

        m_agent->control ()->updatePlaylistView ();
    }
}

void AudioVideoActor::infoMessage (const String &msg) {
    if (sourceMrl ())
        sourceMrl ()->document ()->message (MsgInfoString, (void *) &msg);
}

void AudioVideoActor::readyToPlay (Process *process)
{
    Mrl *mrl = sourceMrl ();
    if (mrl) {
        process->setMrlBackEndNotify (this);
        process->setRepeat (mrl->repeat);
        process->setHasVideo (!mrl->audio_only);
        // TODO set aspect/crop/http_object args
        process->play (mrl->absolutePath ());
    }
}

//------------------------%<----------------------------------------------------

#include <cairo.h>

ImageData::ImageData( const String & img)
 : width (0),
   height (0),
   flags (0),
   has_alpha (false),
   image (0L),
   surface (NULL),
   url (img) {
    //if (img.isEmpty ())
    //    //debugLog() << "New ImageData for " << this << endl;
    //else
    //    //debugLog() << "New ImageData for " << img << endl;
}

ImageData::~ImageData() {
    if (!url.isEmpty ())
        image_data_map->erase (url);
    if (surface)
        cairo_surface_destroy (surface);
    delete image;
}

void ImageData::setImage (Image *img) {
    if (image != img) {
        delete image;
        if (surface) {
            cairo_surface_destroy (surface);
            surface = NULL;
        }
        image = img;
        if (img) {
            width = img->width ();
            height = img->height ();
            has_alpha = false;
        }
    }
    if (!img)
        width = height = 0;
}

void ImageData::setImage (cairo_surface_t *s, int w, int h) {
    delete image;
    image = NULL;
    if (surface)
        cairo_surface_destroy (surface);
    surface = s;
    width = w;
    height = h;
    has_alpha = true;
}

ImageActor::ImageActor (ActorAgent *manager, Node *node,
        const String &url, const ByteArray &ba)
 : Actor (manager, node),
   img_movie (NULL),
   img_movie_iter (NULL),
   svg_renderer (NULL),
   img_movie_timer (0),
   timeout (0),
   update_render (false) {
    setupImage (url, ba);
}

ImageActor::ImageActor (Node *node, ImageDataPtr id)
 : Actor ((ActorAgent *)node->document()->role(RoleActorAgent),
         node),
   img_movie (NULL),
   img_movie_iter (NULL),
   svg_renderer (NULL),
   img_movie_timer (0),
   timeout (0),
   update_render (false) {
    if (!id) {
        Node *c = findChildWithId (node, id_node_svg);
        if (c) {
            svg_renderer = rsvg_handle_new ();
            String xml = c->outerXML();
            const char *cxml = (const char*) xml;
            bool success = rsvg_handle_write (svg_renderer,
                    (const guchar*)cxml, strlen (cxml), NULL);
            rsvg_handle_close (svg_renderer, NULL);
            if (success) {
                cached_img = new ImageData (String ());
                cached_img->flags = ImageData::ImageScalable;
            } else {
                g_object_unref (svg_renderer);
                svg_renderer = NULL;
            }
        }
    } else {
        cached_img = id;
    }
}

ImageActor::~ImageActor () {
    stop ();
    if (img_movie_iter)
        g_object_unref (img_movie_iter);
    if (svg_renderer)
        g_object_unref (svg_renderer);
    else if (img_movie)
        g_object_unref (img_movie);
}

KDE_NO_EXPORT bool ImageActor::play () {
    if (img_movie)
        unpause ();
    return img_movie;
}

KDE_NO_EXPORT void ImageActor::stop () {
    pause ();
}

void ImageActor::pause () {
    if (img_movie_timer)
        g_source_remove (img_movie_timer);
    img_movie_timer = 0;
}

KDE_NO_EXPORT
unsigned ImageActor::render (cairo_surface_t *similar, ISize sz) {
    unsigned penalty = 0;
    if (svg_renderer && update_render) {
        g_object_unref (svg_renderer);
        svg_renderer = NULL;
        Node *c = findChildWithId (m_node, id_node_svg);
        if (c) {
            RsvgHandle *r = rsvg_handle_new ();
            String xml = c->outerXML();
            const char *cxml = (const char*) xml;
            size_t len = strlen (cxml);
            bool success = rsvg_handle_write (r,
                    (const guchar*)cxml, len, NULL);
            rsvg_handle_close (r, NULL);
            if (success) {
                cached_img->setImage (NULL);
                svg_renderer = r;
            } else {
                g_object_unref (r);
            }
            penalty += len / 1024;
        }
        update_render = false;
    }
    if (svg_renderer &&
           (cached_img->width != sz.width || cached_img->height != sz.height)) {
        RsvgDimensionData dim;
        rsvg_handle_get_dimensions (svg_renderer, &dim);
        cairo_surface_t *surf = cairo_surface_create_similar (similar,
                CAIRO_CONTENT_COLOR_ALPHA, sz.width, sz.height);
        cairo_t *cr = cairo_create (surf);
        cairo_scale (cr, 1.0*sz.width / dim.width, 1.0*sz.height / dim.height);
        rsvg_handle_render_cairo (svg_renderer, cr);
        cairo_destroy (cr);
        cached_img->setImage (surf, sz.width, sz.height);

        penalty += sz.width * sz.height / (10 * 480);
        //Image *img = new Image (GPixbuf (
        //            rsvg_handle_get_pixbuf (svg_renderer), true));
        //cached_img->setImage (img);
    }
    return penalty;
}

KDE_NO_EXPORT void ImageActor::updateRender () {
    update_render = true;
    if (m_node)
        m_node->document()->post(new Posting (m_node, m_node, MsgMediaUpdated));
}

KDE_NO_EXPORT void ImageActor::sizes (SSize &size) {
    if (svg_renderer) {
        RsvgDimensionData dim;
        rsvg_handle_get_dimensions (svg_renderer, &dim);
        size.width = dim.width;
        size.height = dim.height;
    } else if (cached_img) {
        size.width = cached_img->width;
        size.height = cached_img->height;
    } else {
        size.width = 0;
        size.height = 0;
    }
}

gboolean movieUpdateTimeout (void * p) {
#ifdef DEBUG_TIMERS
    debugLog() << "movieUpdateTimeout" << endl;
#endif
    return ((ImageActor *)p)->movieTimer ();
}

bool ImageActor::movieTimer () {
    bool ret = false;
    if (gdk_pixbuf_animation_iter_advance (img_movie_iter, 0L)) {
        timeout = 0;
        cached_img->setImage (new Image (GPixbuf (
                    gdk_pixbuf_animation_iter_get_pixbuf (img_movie_iter))));
        cached_img->flags = (int) (ImageData::ImagePixmap | ImageData::ImageAnimated);
        int to = gdk_pixbuf_animation_iter_get_delay_time (img_movie_iter);
        if (to == timeout) {
            ret = true; // re-use timer
        } else {
            timeout = to;
            if (to > 0)
                img_movie_timer = g_timeout_add (to, movieUpdateTimeout, this);
        }
        if (m_node)
            m_node->document()->post (new Posting (m_node, m_node, MsgMediaUpdated));
    } else if (m_node) {
        m_node->document()->post (new Posting(m_node, m_node, MsgMediaFinished));
    }
    return ret;
}

void ImageActor::unpause () {
    if (img_movie_iter && !img_movie_timer) {
        timeout = gdk_pixbuf_animation_iter_get_delay_time (img_movie_iter);
        if (timeout > 0)
            img_movie_timer = g_timeout_add (timeout, movieUpdateTimeout, this);
    }
}

KDE_NO_EXPORT
void ImageActor::setupImage (const String &url, const ByteArray &data) {
    GdkPixbufLoader *pix_loader = gdk_pixbuf_loader_new ();
    if (gdk_pixbuf_loader_write (pix_loader,
                (const guchar *) data.data (),
                data.size (), NULL) &&
            gdk_pixbuf_loader_close (pix_loader, 0L)) {
        cached_img = ImageDataPtr (new ImageData (url));
        img_movie = gdk_pixbuf_loader_get_animation (pix_loader);
        if (img_movie)
            g_object_ref (img_movie);
    }

    if (img_movie && !gdk_pixbuf_animation_is_static_image (img_movie)) {
        img_movie_iter = gdk_pixbuf_animation_get_iter (img_movie, NULL);
        cached_img->setImage (new Image (GPixbuf (
                    gdk_pixbuf_animation_iter_get_pixbuf (img_movie_iter))));
        cached_img->flags = (int) (ImageData::ImagePixmap | ImageData::ImageAnimated);
    } else if (img_movie) {
        cached_img->setImage (new Image (GPixbuf (
                    gdk_pixbuf_animation_get_static_image (img_movie))));
        cached_img->flags = (int) ImageData::ImagePixmap;
        (*image_data_map)[url] = ImageDataPtrW (cached_img);
    }
    g_object_unref (pix_loader);
}

bool ImageActor::isEmpty () const {
    return !cached_img || (!svg_renderer && cached_img->isEmpty ());
}

//------------------------%<----------------------------------------------------

static int default_font_size = 11;

TextActor::TextActor (ActorAgent *manager, Node *node, const ByteArray &ba)
 : Actor (manager, node) {
    if (!ba.isEmpty ())
        text = String (ba.data (), ba.size ());
}

TextActor::~TextActor () {
}

bool TextActor::play () {
    return !text.isEmpty ();
}

int TextActor::defaultFontSize () {
    return default_font_size;
}
