/*
* Copyright (C) 2010  Koos Vriezen <koos.vriezen@gmail.com>
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#ifndef _KMPLAYER_IO_H_
#define _KMPLAYER_IO_H_

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>

#include <glib.h>
#include <gdk/gdk.h>
#include <gconf/gconf-client.h>

#include "kmplayerstring.h"

typedef struct _GFile GFILE;

namespace KMPlayer {

class GIOFilePrivate;


enum FileMode { IO_ReadOnly=0x01, IO_WriteOnly=0x02, IO_ReadWrite=0x03 };

class GIOFile {
    GIOFilePrivate * d;
    friend class GioUrl;
public:
    GIOFile (const String & path);
    ~GIOFile ();
    bool open (FileMode mode);
    void close ();
    ByteArray readAll ();
    off_t readBlock (char *, off_t max);
    off_t writeBlock (const char * data, off_t length);
    bool exists ();
    off_t size ();
};

typedef GIOFile File;


class GioUrl {
    GFile *m_file;
public:
    GioUrl ();
    GioUrl (const String &url);
    GioUrl (const File &file);
    GioUrl (const GioUrl &base, const String &rel);
    GioUrl (const GioUrl &url);
    ~GioUrl ();
    GioUrl & operator = (const String &url);
    GioUrl & operator = (const GioUrl &url);
    bool operator == (const GioUrl &u) const { return url() == u.url(); }
    String url () const;
    String path () const;
    String protocol () const;
    static String decode_string (const String &str);
    static String encode_string (const String &str);
    bool isLocalFile () const;
};

typedef GioUrl URL;


class IOJob;
class IOJobListener;
typedef class CurlGetJob IOJobPrivate;

IOJob *asyncGet (IOJobListener *receiver, const String &url);

class IOJob {
    friend IOJob *asyncGet (IOJobListener *receiver, const String &url);
    friend class CurlGetJob;

public:
    ~IOJob ();

    void setHeader (const String &header);
    void setHttpPostData (const ByteArray &data);
    void start (off_t pos=0);
    void kill (bool quiet = true);
    bool error ();
    void setError ();
    unsigned long contentLength() const;
    String contentType () const;

private:
    IOJob (IOJobPrivate *);

    IOJobPrivate *d;
};

class IOJobListener {
public:
    virtual void jobData (IOJob * job, ByteArray & data) = 0;
    virtual void jobResult (IOJob * job) = 0;
    virtual void redirected (IOJob * job, const String & uri) = 0;
};


//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT inline GioUrl::GioUrl () : m_file (0L) {}
} // namespace

#endif
