/**
 * Copyright (C) 2005 by Koos Vriezen <koos ! vriezen ? xs4all ! nl>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License version 2 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 **/

//#include <config.h>
#include <string.h>
#include "kmplayer_atom.h"
#include "kmplayer_smil.h"

using namespace KMPlayer;

Node *ATOM::Feed::childFromTag (const String & tag) {
    if (!strcmp ((const char *) tag, "entry"))
        return new ATOM::Entry (m_doc);
    else if (!strcmp ((const char *) tag, "link"))
        return new ATOM::Link (m_doc);
    else if (!strcmp ((const char *) tag, "title"))
        return new DarkNode (m_doc, tag, id_node_title);
    return NULL;
}

void ATOM::Feed::closed () {
    for (Node *c = firstChild (); c; c = c->nextSibling ())
        if (c->id == id_node_title) {
            title = c->innerText ().stripWhiteSpace ();
            break;
        }
    Element::closed ();
}

void *ATOM::Feed::role (RoleType msg, void *content)
{
    if (RolePlaylist == msg)
        return !title.isEmpty () ? (PlaylistRole *) this : NULL;
    return Element::role (msg, content);
}

Node *ATOM::Entry::childFromTag (const String &tag) {
    const char *cstr = (const char *) tag;
    if (!strcmp (cstr, "link"))
        return new ATOM::Link (m_doc);
    else if (!strcmp (cstr, "content"))
        return new ATOM::Content (m_doc);
    else if (!strcmp (cstr, "title"))
        return new DarkNode (m_doc, tag, id_node_title);
    else if (!strcmp (cstr, "summary"))
        return new DarkNode (m_doc, tag, id_node_summary);
    else if (!strcmp (cstr, "media:group"))
        return new MediaGroup (m_doc);
    else if (!strcmp (cstr, "gd:rating") ||
            !strcmp (cstr, "category") ||
            !strcmp (cstr, "author:") ||
            !strcmp (cstr, "id") ||
            !strcmp (cstr, "updated") ||
            !strncmp (cstr, "yt:", 3) ||
            !strncmp (cstr, "gd:", 3))
        return new DarkNode (m_doc, tag, id_node_ignored);
    return NULL;
}

void ATOM::Entry::closed () {
    MediaGroup *group = NULL;
    for (Node *c = firstChild (); c; c = c->nextSibling ())
        if (c->id == id_node_title) {
            title = c->innerText ().stripWhiteSpace ();
        } else if (c->id == id_node_media_group) {
            group = static_cast <MediaGroup *> (c);
        }
    Element::closed ();
}

void *ATOM::Entry::role (RoleType msg, void *content)
{
    if (RolePlaylist == msg)
        return !title.isEmpty () ? (PlaylistRole *) this : NULL;
    return Element::role (msg, content);
}

String ATOM::Entry::description () const {
    for (Node *c = firstChild (); c; c = c->nextSibling ())
        if (c->id == id_node_content) {
            Mrl *mrl = c->mrl ();
            if (mrl->mimetype.startsWith ("text/") ||
                    mrl->mimetype.indexOf ("html") > -1)
                return mrl->innerText ();
            break;
        }
    return String ();
}

Node::PlayType ATOM::Link::playType () {
    return src.isEmpty () ? play_type_none : play_type_unknown;
}

void ATOM::Link::closed () {
    String href;
    String rel;
    for (Attribute *a = attributes ().first (); a; a = a->nextSibling ()) {
        if (a->name () == Ids::attr_href)
            href = a->value ();
        else if (a->name () == Ids::attr_title)
            title = a->value ();
        else if (a->name () == "rel")
            rel = a->value ();
    }
    if (!href.isEmpty () && rel == "enclosure")
        src = href;
    else if (title.isEmpty ())
        title = href;
    Mrl::closed ();
}

void ATOM::Content::closed () {
    for (Attribute *a = attributes ().first (); a; a = a->nextSibling ()) {
        if (a->name () == Ids::attr_src)
            src = a->value ();
        else if (a->name () == Ids::attr_type) {
            String v = a->value ().lower ();
            if (v == "text")
                mimetype = "text/plain";
            else if (v == "html")
                mimetype = "text/html";
            else if (v == "xhtml")
                mimetype = "application/xhtml+xml";
            else
                mimetype = v;
        }
    }
    Mrl::closed ();
}

Node::PlayType ATOM::Content::playType () {
    if (!hasChildNodes () && !src.isEmpty ())
        return play_type_unknown;
    return play_type_none;
}

Node *ATOM::MediaGroup::childFromTag (const String &tag) {
    const char *cstr = (const char *) tag;
    if (!strcmp (cstr, "media:content"))
        return new ATOM::MediaContent (m_doc);
    else if (!strcmp (cstr, "media:title"))
        return new DarkNode (m_doc, tag, id_node_media_title);
    else if (!strcmp (cstr, "media:description"))
        return new DarkNode (m_doc, tag, id_node_media_description);
    else if (!strncmp (cstr, "media:", 6))
        return new DarkNode (m_doc, tag, id_node_ignored);
    else if (!strcmp (cstr, "smil"))
        return new SMIL::Smil (m_doc);
    return NULL;
}

void ATOM::MediaGroup::closed () {
    String title;
    for (Node *c = firstChild (); c; c = c->nextSibling ())
        if (id_node_media_title == c->id) {
            title = c->innerText ();
            break;
        }
    if (!title.isEmpty ())
        for (Node *c = firstChild (); c; c = c->nextSibling ())
            if (id_node_media_content == c->id) {
                Mrl *mrl = c->mrl ();
                if (mrl && !mrl->src.isEmpty ()) {
                    String s = title;
                    if (!mrl->mimetype.isEmpty ())
                        s += Char (' ') + c->mrl ()->mimetype;
                    c->mrl ()->setCaption (s);
                }
            }
}

void ATOM::MediaGroup::message (MessageType msg, void *content) {
    if (MsgChildFinished == msg &&
            id_node_media_content == ((Posting *) content)->source->id)
        finish (); // only play one
    Element::message (msg, content);
}

void ATOM::MediaContent::closed () {
    for (Attribute *a = attributes ().first (); a; a = a->nextSibling ()) {
        if (a->name () == Ids::attr_url)
            src = a->value();
        else if (a->name () == Ids::attr_type)
            mimetype = a->value ();
        else if (a->name () == Ids::attr_height)
            size.height = a->value ().toInt ();
        else if (a->name () == Ids::attr_width)
            size.width = a->value ().toInt ();
        else if (a->name () == Ids::attr_width)
            size.width = a->value ().toInt ();
    }
    Mrl::closed ();
}
