/**
 * Copyright (C) 2004-2007 by Koos Vriezen <koos.vriezen@gmail.com>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License version 2 as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 **/

#include <config.h>
#include <time.h>
#include <strings.h>
#include <string.h>
#include <map>
#ifdef HAVE_EXPAT
#include <expat.h>
#endif
#include "kmplayerplaylist.h"
#ifdef KMPLAYER_XML_FORMATS
#include "kmplayer_asx.h"
#include "kmplayer_atom.h"
#include "kmplayer_rp.h"
#include "kmplayer_rss.h"
#include "kmplayer_smil.h"
#include "kmplayer_xspf.h"
#else
#include <vector>
typedef std::vector <KMPlayer::String> StringList;
#endif
#include "actor.h"

#ifdef SHAREDPTR_DEBUG
int shared_data_count;
#endif

#define TIMEOUT_OFF -1

using namespace KMPlayer;

//-----------------------------------------------------------------------------

Node * KMPlayer::fromXMLDocumentTag (NodePtr & d, const String & tag) {
    const char * const name = (const char *) tag;
#ifdef KMPLAYER_XML_FORMATS
    if (!strcmp (name, "smil"))
        return new SMIL::Smil (d);
    else if (!strcasecmp (name, "asx"))
        return new ASX::Asx (d);
    else if (!strcasecmp (name, "imfl"))
        return new RP::Imfl (d);
    else if (!strcasecmp (name, "rss"))
        return new RSS::Rss (d);
    else if (!strcasecmp (name, "feed"))
        return new ATOM::Feed (d);
    else if (!strcasecmp (name, "playlist"))
        return new XSPF::Playlist (d);
    else
#endif
    if (!strcasecmp (name, "url"))
        return new GenericURL (d, String ());
    else if (!strcasecmp (name, "mrl") ||
            !strcasecmp (name, "document"))
        return new GenericMrl (d);
    return 0L;
}

//-----------------------------------------------------------------------------

static const Char char_lt ('<');
static const Char char_gt ('>');
static const Char char_quot ('"');
static const Char char_amp ('&');
static const Char char_nl ('\n');

TextStream &KMPlayer::operator << (TextStream &out, const XMLStringlet &txt) {
    int len = int (txt.str.length ());
    for (int i = 0; i < len; ++i) {
        if (txt.str [i] == char_lt) {
            out <<  "&lt;";
        } else if (txt.str [i] == char_gt) {
            out <<  "&gt;";
        } else if (txt.str [i] == char_quot) {
            out <<  "&quot;";
        } else if (txt.str [i] == char_amp) {
            out <<  "&amp;";
        } else
            out << txt.str [i];
    }
    return out;
}

//-----------------------------------------------------------------------------

Connection::Connection (Node *invoker, Node *receiver, VirtualVoid *pl)
 : connectee (invoker), connecter (receiver), payload (pl) {
#ifdef KMPLAYER_TEST_CONNECTION
    connection_counter++;
#endif
}

ConnectionLink::ConnectionLink () : connection (NULL) {}

ConnectionLink::~ConnectionLink () {
    disconnect ();
}

bool ConnectionLink::connect (Node *send, MessageType msg, Node *rec,
        VirtualVoid *payload) {
    disconnect ();
    ConnectionList *list = nodeMessageReceivers (send, msg);
    if (list) {
        connection = new Connection (send, rec, payload);
        connection->list = list;
        connection->link = &connection;
        connection->prev = list->link_last;
        connection->next = NULL;
        if (list->link_last)
            list->link_last->next = connection;
        else
            list->link_first = connection;
        list->link_last = connection;
    }
    return list;
}

void ConnectionLink::disconnect () const {
    if (connection) {
        Connection *tmp = connection;
        if (tmp->prev)
            tmp->prev->next = tmp->next;
        else
            tmp->list->link_first = tmp->next;
        if (tmp->next)
            tmp->next->prev = tmp->prev;
        else
            tmp->list->link_last = tmp->prev;
        *tmp->link = NULL;
        if (tmp->list->link_next == tmp)
            tmp->list->link_next = tmp->next;
        delete tmp;
    }
    ASSERT (!connection);
}

void ConnectionLink::assign (const ConnectionLink *link) const {
    disconnect ();
    connection = link->connection;
    link->connection = NULL;
    if (connection)
        connection->link = &connection;
}

Node *ConnectionLink::signaler () const {
    return connection ? connection->connectee.ptr () : NULL;
}

ConnectionList::ConnectionList ()
    : link_first (NULL), link_last (NULL), link_next (NULL) {}

ConnectionList::~ConnectionList () {
    clear ();
}

void ConnectionList::clear () {
    while (link_first) {
        Connection *tmp = link_first;
        link_first = tmp->next;
        *tmp->link = NULL;
        delete tmp;
    }
    link_last = link_next = NULL;
}

//-----------------------------------------------------------------------------

KDE_NO_CDTOR_EXPORT Node::Node (NodePtr & d, short _id)
 : m_doc (d), state (state_init), id (_id),
   auxiliary_node (false), open (false) {}

Node::~Node () {
    clear ();
}

Document * Node::document () {
    return convertNode <Document> (m_doc);
}

Mrl * Node::mrl () {
    return 0L;
}

KDE_NO_EXPORT const char * Node::nodeName () const {
    return "node";
}

void Node::setState (State nstate) {
    if (state != nstate && (state_init == nstate || state != state_resetting)) {
        State old = state;
        state = nstate;
        if (document ()->notify_listener)
            document()->notify_listener->stateElementChanged (this, old, state);
    }
}

String Node::description () const {
    return String ();
}

void Node::activate () {
    //kdDebug () << nodeName () << " Node::activate" << endl;
    setState (state_activated);
    if (firstChild ())
        firstChild ()->activate (); // activate only the first
    else
        finish (); // a quicky :-)
}

void Node::begin () {
    if (active ()) {
        setState (state_began);
    } else
        errorLog() << nodeName() << " begin call on not active element" << endl;
}

void Node::defer () {
    if (active ()) {
        setState (state_deferred);
    } else
        errorLog() << "Node::defer () call on not activated element" << endl;
}

void Node::undefer () {
    if (state == state_deferred) {
        if (firstChild () && firstChild ()->state > state_init) {
            setState (state_began);
        } else {
            setState (state_activated);
            // don't call activate; if this is the document, it will start again
            //activate ();
        }
    } else
        warningLog() <<"Node::undefer () call on not defered element"<< endl;
}

void Node::finish () {
    if (active ()) {
        setState (state_finished);
        if (m_parent)
            document ()->post (new Posting (this, m_parent, MsgChildFinished));
        else
            deactivate (); // document deactivates itself on finish
    } else
        warningLog() <<"Node::finish () call on not active element"<< endl;
}

void Node::deactivate () {
    //kdDebug () << nodeName () << " Node::deactivate" << endl;
    bool need_finish (unfinished ());
    if (state_resetting != state)
        setState (state_deactivated);
    for (NodePtr e = firstChild (); e; e = e->nextSibling ()) {
        if (e->state > state_init && e->state < state_deactivated)
            e->deactivate ();
        else
            break; // remaining not yet activated
    }
    if (need_finish && m_parent && m_parent->active ())
        document ()->post (new Posting (this, m_parent, MsgChildFinished));
}

void Node::reset () {
    //kdDebug () << nodeName () << " Node::reset" << endl;
    if (active ()) {
        setState (state_resetting);
        deactivate ();
    }
    setState (state_init);
    for (NodePtr e = firstChild (); e; e = e->nextSibling ()) {
        if (e->state != state_init)
            e->reset ();
        // else
        //    break; // rest not activated yet
    }
}

void Node::clear () {
    clearChildren ();
}

void Node::clearChildren () {
    if (m_doc)
        document()->m_tree_version++;
    while (m_first_child != m_last_child) {
        // avoid stack abuse with 10k children derefing each other
        m_last_child->m_parent = 0L;
        m_last_child = m_last_child->m_prev;
        m_last_child->m_next = 0L;
    }
    if (m_first_child)
        m_first_child->m_parent = 0L;
    m_first_child = m_last_child = 0L;
}

template <>
void TreeNode<Node>::appendChild (Node *c) {
    static_cast <Node *> (this)->document()->m_tree_version++;
    ASSERT (!c->parentNode ());
    appendChildImpl (c);
}

template <>
KDE_NO_EXPORT void TreeNode<Node>::insertBefore (Node *c, Node *b) {
    ASSERT (!c->parentNode ());
    static_cast <Node *> (this)->document()->m_tree_version++;
    insertBeforeImpl (c, b);
}

template <>
void TreeNode<Node>::removeChild (NodePtr c) {
    static_cast <Node *> (this)->document()->m_tree_version++;
    removeChildImpl (c);
}

KDE_NO_EXPORT void Node::insertBefore (const String &xml, Node *b) {
    Node *last = lastChild ();
    TextStream inxml (&xml);
    readXML (this, inxml, String(), false);
    if (lastChild () != last) {
        NodePtr n = lastChild ();
        n->normalize ();
        if (b) {
            removeChild (n);
            insertBefore (n, b);
        }
    }
}

KDE_NO_EXPORT void Node::replaceChild (NodePtr _new, NodePtr old) {
    document()->m_tree_version++;
    if (old->m_prev) {
        old->m_prev->m_next = _new;
        _new->m_prev = old->m_prev;
        old->m_prev = 0L;
    } else {
        _new->m_prev = 0L;
        m_first_child = _new;
    }
    if (old->m_next) {
        old->m_next->m_prev = _new;
        _new->m_next = old->m_next;
        old->m_next = 0L;
    } else {
        _new->m_next = 0L;
        m_last_child = _new;
    }
    _new->m_parent = this;
    old->m_parent = 0L;
}

Node *Node::childFromTag (const String &) {
    return NULL;
}

void Node::normalize () {
    Node *e = firstChild ();
    while (e) {
        Node *tmp = e->nextSibling ();
        if (!e->isElementNode () && e->id == id_node_text) {
            String val = e->nodeValue (); //.simplifyWhiteSpace ();
            if (val.isEmpty ())
                removeChild (e);
            else
                static_cast <TextNode *> (e)->setText (val);
        } else
            e->normalize ();
        e = tmp;
    }
}

static void getInnerText (const Node *p, TextStream & out) {
    for (Node *e = p->firstChild (); e; e = e->nextSibling ()) {
        if (e->id == id_node_text || e->id == id_node_cdata)
            out << e->nodeValue ();
        else
            getInnerText (e, out);
    }
}

String Node::innerText () const {
    TextStream out;
    getInnerText (this, out);
    return out.release ();
}

static
void getOuterXML (const Node *p, TextStream & out, int depth, bool inter) {
    if (!p->isElementNode ()) { // #text or #cdata
        if (p->id == id_node_cdata)
            out << "<![CDATA[" << p->nodeValue () << "]]>" << char_nl;
        else
            out << XMLStringlet (p->nodeValue ()) << char_nl;
    } else {
        const Element *e = static_cast <const Element *> (p);
        String indent (String ().fill (Char (' '), depth));
        out << indent << char_lt << XMLStringlet (e->nodeName ());
        for (Attribute *a = e->attributes().first(); a; a = a->nextSibling())
            out << " " << XMLStringlet (a->name ().toString ()) <<
                "=\"" << XMLStringlet (a->value ()) << "\"";
        if (e->hasChildNodes ()) {
            out << char_gt << char_nl;
            for (Node *c = e->firstChild (); c; c = c->nextSibling ()) {
                if (!inter && c->mrl () && c->mrl ()->opener.ptr () == e)
                    continue;
                getOuterXML (c, out, depth + 1, inter);
            }
            out << indent << String("</") << XMLStringlet (e->nodeName()) <<
                char_gt << char_nl;
        } else
            out << String ("/>") << char_nl;
    }
}

String Node::innerXML () const {
    TextStream out;
    for (Node *e = firstChild (); e; e = e->nextSibling ())
        getOuterXML (e, out, 0, true);
    return out.release ();
}

String Node::outerXML (bool inter) const {
    TextStream out;
    getOuterXML (this, out, 0, inter);
    return out.release ();
}

Node::PlayType Node::playType () {
    return play_type_none;
}

void Node::opened () {
    open = true;
}

void Node::closed () {
    open = false;
}

void Node::message (MessageType msg, void *content) {
    switch (msg) {

    case MsgChildFinished: {
        Posting *post = (Posting *) content;
        if (unfinished ()) {
            if (post->source && post->source->state == state_finished)
                post->source->deactivate ();
            if (post->source && post->source->nextSibling ())
                post->source->nextSibling ()->activate ();
            else
                finish (); // we're done
        }
        break;
    }

    default:
        break;
    }
}

void *Node::role (RoleType msg, void *content) {
    switch (msg) {
    case RoleReady:
        return MsgBool (true);
    default:
        break;
    }
    return NULL;
}

KDE_NO_EXPORT void Node::deliver (MessageType msg, void *content) {
    ConnectionList *nl = nodeMessageReceivers (this, msg);
    if (nl)
        for (Connection *c = nl->first(); c; c = nl->next ())
            if (c->connecter)
                c->connecter->message (msg, content);
}

void Node::accept (Visitor * v) {
    v->visit (this);
}

String Node::nodeValue () const {
    return innerText ().stripWhiteSpace ();
}

template <> void List<NodeRefItem>::remove (NodeRefItem *c) {
        removeImpl (c);
}

//-----------------------------------------------------------------------------

namespace {
    struct KMPLAYER_NO_EXPORT ParamValue {
        String val;
        StringList  * modifications;
        ParamValue (const String & v) : val (v), modifications (0L) {}
        ~ParamValue () { delete modifications; }
        String value ();
        void setValue (const String & v) { val = v; }
    };
    typedef std::map <TrieString, ParamValue *> ParamMap;
}

namespace KMPlayer {
    class KMPLAYER_NO_EXPORT ElementPrivate {
    public:
        ~ElementPrivate ();
        ParamMap params;
        void clear ();
    };
}

KDE_NO_EXPORT String ParamValue::value () {
    return modifications && modifications->size ()
        ? modifications->back () : val;
}

KDE_NO_CDTOR_EXPORT ElementPrivate::~ElementPrivate () {
    clear ();
}

KDE_NO_EXPORT void ElementPrivate::clear () {
    const ParamMap::iterator e = params.end ();
    for (ParamMap::iterator i = params.begin (); i != e; ++i)
        delete i->second;
    params.clear ();
}

Element::Element (NodePtr & d, short id)
    : Node (d, id), d (new ElementPrivate) {}

Element::~Element () {
    delete d;
}

void Element::setParam (const TrieString &name, const String &val, int *mid) {
    ParamValue * pv = d->params [name];
    if (!pv) {
        pv = new ParamValue (mid ? getAttribute (name) : val);
        d->params [name] = pv;
    }
    if (mid) {
        if (!pv->modifications)
            pv->modifications = new StringList;
        if (*mid >= 0 && *mid < int (pv->modifications->size ())) {
            (*pv->modifications) [*mid] = val;
        } else {
            *mid = pv->modifications->size ();
            pv->modifications->push_back (val);
        }
    } else {
        pv->setValue (val);
    }
    parseParam (name, val);
}

String Element::param (const TrieString & name) {
    ParamValue * pv = d->params [name];
    if (pv)
        return pv->value ();
    return getAttribute (name);
}

void Element::resetParam (const TrieString &name, int mid) {
    ParamValue * pv = d->params [name];
    if (pv && pv->modifications) {
        if (int (pv->modifications->size ()) > mid && mid > -1) {
            (*pv->modifications) [mid].clear ();
            while (pv->modifications->size () > 0 &&
                    pv->modifications->back ().isNull ())
                pv->modifications->pop_back ();
        }
        String val = pv->value ();
        if (pv->modifications->size () == 0) {
            delete pv->modifications;
            pv->modifications = 0L;
            if (val.isNull ()) {
                delete pv;
                d->params.erase (name);
            }
        }
        parseParam (name, val);
    } else
        errorLog () << "resetting " << name.toString() << " that doesn't exists" << endl;
}

void Element::setAttribute (const TrieString & name, const String & value) {
    for (Attribute *a = m_attributes.first (); a; a = a->nextSibling ())
        if (name == a->name ()) {
            if (value.isNull ())
                m_attributes.remove (a);
            else
                a->setValue (value);
            return;
        }
    if (!value.isNull ())
        m_attributes.append (new Attribute (TrieString (), name, value));
}

String Element::getAttribute (const TrieString & name) {
    for (Attribute *a = m_attributes.first (); a; a = a->nextSibling ())
        if (name == a->name ())
            return a->value ();
    return String ();
}

void Element::init () {
    d->clear();
    for (Attribute *a = attributes ().first (); a; a = a->nextSibling ()) {
        String v = a->value ();
        int p = v.indexOf (Char ('{'));
        if (p > -1) {
            int q = v.indexOf (Char ('}'), p + 1);
            if (q > -1)
                continue;
        }
        parseParam (a->name (), v);
    }
}

void Element::reset () {
    d->clear();
    Node::reset ();
}

void Element::clear () {
    m_attributes = AttributeList (); // remove attributes
    d->clear();
    Node::clear ();
}

void Element::setAttributes (const AttributeList &attrs) {
    m_attributes = attrs;
}

void Element::accept (Visitor * v) {
    v->visit (this);
}

//-----------------------------------------------------------------------------

Attribute::Attribute (const TrieString &ns, const TrieString &n, const String &v)
  : m_namespace (ns), m_name (n), m_value (v) {}

void Attribute::setName (const TrieString & n) {
    m_name = n;
}

void Attribute::setValue (const String & v) {
    m_value = v;
}

//-----------------------------------------------------------------------------

String PlaylistRole::caption () const {
    return title;
}

void PlaylistRole::setCaption (const String &s) {
    title = s;
}

//-----------------------------------------------------------------------------

static bool hasMrlChildren (const NodePtr & e) {
    for (Node *c = e->firstChild (); c; c = c->nextSibling ())
        if (c->isPlayable () || hasMrlChildren (c))
            return true;
    return false;
}

Mrl::Mrl (NodePtr & d, short id)
    : Element (d, id), cached_ismrl_version (~0),
      media_info (NULL),
      aspect (0), repeat (0),
      clip_begin (0),
      view_mode (SingleMode),
      resolved (false), bookmarkable (true),
      audio_only (false) {}

Mrl::~Mrl () {
    if (media_info)
        delete media_info;
}

Node::PlayType Mrl::playType () {
    if (cached_ismrl_version != document()->m_tree_version) {
        bool ismrl = !hasMrlChildren (this);
        cached_play_type = ismrl ? play_type_unknown : play_type_none;
        cached_ismrl_version = document()->m_tree_version;
    }
    return cached_play_type;
}

String Mrl::absolutePath () {
    int p = src.indexOf (":/");
    if (p > -1 && p < 15)
        return src;
    String path = src;
    if (!path.isEmpty()) {
        for (Node *e = parentNode (); e; e = e->parentNode ()) {
            Mrl *mrl = e->mrl ();
            if (mrl && !mrl->src.isEmpty () &&
                    mrl->src != src && !mrl->src.startsWith ("Playlist:/")) {
                path = URL (mrl->absolutePath (), path).url ();
                break;
            }
        }
    }
    return path.isEmpty () ? src : path;
}

Node *Mrl::childFromTag (const String & tag) {
    Node * elm = fromXMLDocumentTag (m_doc, tag);
    if (elm)
        return elm;
    return NULL;
}

Mrl * Mrl::mrl () {
    return this;
}

void Mrl::message (MessageType msg, void *content) {
    switch (msg) {
    case MsgMediaReady:
        resolved = true;
        if (state == state_deferred) {
            if (isPlayable ()) {
                setState (state_activated);
                begin ();
            } else {
                Element::activate ();
            }
        }
        break;

    case MsgMediaFinished:
        if (state == state_deferred &&
                !isPlayable () && firstChild ()) {//if backend added child links
            state = state_activated;
            firstChild ()->activate ();
        } else
            finish ();
        break;

    default:
        break;
    }
    Node::message (msg, content);
}

void *Mrl::role (RoleType msg, void *content) {
    switch (msg) {

    case RoleChildDisplay:
        for (Node *p = parentNode (); p; p = p->parentNode ())
            if (p->mrl ())
                return p->role (msg, content);
        return NULL;

    case RolePlaylist:
        if (title.isEmpty ())
            title = src;
        return !title.isEmpty () ? (PlaylistRole *) this : NULL;

    default:
        break;
    }
    return Node::role (msg, content);
}

void Mrl::activate () {
    if (!resolved && isPlayable ()) {
        setState (state_deferred);
        media_info = new MediaInfo (this, ActorAgent::AudioVideo);
        resolved = media_info->wget (absolutePath ());
    } else if (isPlayable ()) {
        setState (state_activated);
        begin ();
    } else {
        Element::activate ();
    }
}

void Mrl::begin () {
    if (!src.isEmpty ()) {
        if (!media_info)
            media_info = new MediaInfo (this, ActorAgent::AudioVideo);
        if (!media_info->media)
            media_info->create ();
        if (media_info->media && media_info->media->play ())
            setState (state_began);
        else
            deactivate ();
    } else {
        deactivate (); // nothing to activate
    }
}

void Mrl::defer () {
    if (media_info && media_info->media)
        media_info->media->pause ();
    Node::defer ();
}

void Mrl::undefer () {
    if (media_info && media_info->media) {
        media_info->media->unpause ();
        setState (state_began);
    } else {
        Node::undefer ();
    }
}

void Mrl::deactivate () {
    if (media_info) {
        delete media_info;
        media_info = NULL;
    }
    Node::deactivate ();
}

unsigned int Mrl::parseTimeString (const String &ts) {
    String s (ts);
    int multiply[] = { 1, 60, 60 * 60, 24 * 60 * 60, 0 };
    int mpos = 0;
    double d = 0;
    while (!s.isEmpty () && multiply[mpos]) {
        int p = s.lastIndexOf (Char (':'));
        String t = p >= 0 ? s.mid (p + 1) : s;
        d += multiply[mpos++] * t.toDouble();
        s = p >= 0 ? s.left (p) : String ();
    }
    if (d > 0.01)
        return (unsigned int) (d * 100);
    return 0;
}

String *Mrl::parsePanZoomString (const String &pz, String *coord) {
    int i;
    gchar **list = g_strsplit ((const char *) pz, ",", -1);
    for (i = 0; list [i] && i < 4; ++i)
        coord[i] = list [i];
    if (i < 4)
        warningLog () << "panZoom less then four nubmers" << endl;
    g_strfreev (list);
    return i < 4 ? NULL : coord;
}

//----------------------%<-----------------------------------------------------

void Mrl::parseParam (const TrieString & para, const String & val) {
    if (para == StringPool::attr_src && !src.startsWith ("#")) {
        String abs = absolutePath ();
        if (abs != src)
            src = val;
        else
            src = URL (abs, val).url ();
        for (NodePtr c = firstChild (); c; c = c->nextSibling ())
            if (c->mrl () && c->mrl ()->opener.ptr () == this) {
                removeChild (c);
                c->reset();
            }
        resolved = false;
    }
}

//-----------------------------------------------------------------------------

CommandData::CommandData (Command *e, CommandData *n)
 : event (e), next (n) {}

CommandData::~CommandData () {
    if (event)
        event->removed ();
}
//-----------------------------------------------------------------------------

Postpone::Postpone (NodePtr doc) : m_doc (doc) {
    if (m_doc)
        m_doc->document ()->timeOfDay (postponed_time);
}

Postpone::~Postpone () {
    if (m_doc)
        m_doc->document ()->proceed (postponed_time);
}

//-----------------------------------------------------------------------------

static NodePtr dummy_element;

Document::Document (const String & s, PlayListNotify * n)
 : Mrl (dummy_element, id_node_document),
   notify_listener (n),
   m_tree_version (0),
   event_queue (NULL),
   paused_queue (NULL),
   cur_event (NULL),
   cur_time_interval (TIMEOUT_OFF) {
    m_doc = m_self; // just-in-time setting fragile m_self to m_doc
    src = s;
    if (!s.isEmpty ())
        setAttribute (StringPool::attr_src, s);
}

Document::~Document () {
    //debugLog () << "~Document" << endl;
}

static Node *getElementByIdImpl (Node *n, const String & id, bool inter) {
    Node *elm = NULL;
    if (!n->isElementNode ())
        return NULL;
    Element *e = static_cast <Element *> (n);
    if (e->getAttribute (StringPool::attr_id) == id)
        return n;
    for (Node *c = e->firstChild (); c; c = c->nextSibling ()) {
        if (!inter && c->mrl () && c->mrl ()->opener.ptr () == n)
            continue;
        if ((elm = getElementByIdImpl (c, id, inter)))
            break;
    }
    return elm;
}

Node *Document::getElementById (const String & id) {
    return getElementByIdImpl (this, id, true);
}

Node *Document::getElementById (Node *n, const String & id, bool inter) {
    return getElementByIdImpl (n, id, inter);
}

KDE_NO_EXPORT Node *Document::childFromTag (const String & tag) {
    Node * elm = fromXMLDocumentTag (m_doc, tag);
    if (elm)
        return elm;
    return NULL;
}

void Document::dispose () {
    clear ();
    m_doc = 0L;
}

void Document::activate () {
    first_event_time.tv_sec = 0;
    cur_time_interval = TIMEOUT_OFF;
    last_event_time = 0;
    Mrl::activate ();
}

void Document::defer () {
    if (resolved)
        postpone_lock = postpone ();
    Mrl::defer ();
}

void Document::undefer () {
    postpone_lock = 0L;
    Mrl::undefer ();
}

void Document::reset () {
    Mrl::reset ();
    if (notify_listener && TIMEOUT_OFF != cur_time_interval) {
        notify_listener->setTimeout (-1);
        cur_time_interval = TIMEOUT_OFF;
    }
    if (event_queue) {
        while (event_queue) {
            CommandData *ed = event_queue;
            event_queue = ed->next;
            delete ed;
        }
    }
    postpone_lock = 0L;
}

static inline
int diffTime (const struct timeval & tv1, const struct timeval & tv2) {
    //debugLog () << "diffTime sec:" << ((tv1.tv_sec - tv2.tv_sec) * 1000) << " usec:" << ((tv1.tv_usec - tv2.tv_usec) /1000) << endl;
    return (tv1.tv_sec - tv2.tv_sec) * 1000 + (tv1.tv_usec - tv2.tv_usec) /1000;
}

static inline void addTime (struct timeval & tv, int ms) {
    if (ms >= 1000) {
        tv.tv_sec += ms / 1000;
        ms %= 1000;
    }
    tv.tv_sec += (tv.tv_usec + ms*1000) / 1000000;
    tv.tv_usec = (tv.tv_usec + ms*1000) % 1000000;
}

//-----------------------------------------------------------------------------

void Posting::apply (struct timeval &) {
    if (target)
        target->message (message, this);
    delete this;
}

bool Posting::postponable () const {
    return message == MsgEventTimer ||
        message == MsgEventStarted ||
        message == MsgEventStopped;
}

int Posting::delay () const {
    return 0;
}

void Posting::removed () {
    delete this;
}

KDE_NO_CDTOR_EXPORT TimerPosting::TimerPosting (Node *t, int ms, unsigned eid)
 : Posting (NULL, t, MsgEventTimer),
   event_id (eid),
   milli_sec (ms),
   interval (false) {}

void TimerPosting::apply (struct timeval &timeout) {
    if (target)
        target->message (message, this);
    if (target && interval) {
        interval = false; // reset interval
        addTime (timeout, milli_sec);
        target->document ()->insertPosting (this, timeout);
    } else {
        delete this;
    }
}

int TimerPosting::delay () const {
    return milli_sec;
}

KDE_NO_CDTOR_EXPORT UpdateEvent::UpdateEvent (Document *doc, unsigned int skip)
 : skipped_time (skip) {
    struct timeval tv;
    doc->timeOfDay (tv);
    cur_event_time = doc->last_event_time;
}

PostponedEvent::PostponedEvent (bool postponed)
    : is_postponed (postponed) {}

//-----------------------------------------------------------------------------
/*static inline void subtractTime (struct timeval & tv, int ms) {
    int sec = ms / 1000;
    int msec = ms % 1000;
    tv.tv_sec -= sec;
    if (tv.tv_usec / 1000 >= msec) {
        tv.tv_usec -= msec * 1000;
    } else {
        tv.tv_sec--;
        tv.tv_usec = 1000000 - (msec - tv.tv_usec / 1000 );
    }
}*/

void Document::timeOfDay (struct timeval & tv) {
    gettimeofday (&tv, 0L);
    if (!first_event_time.tv_sec) {
        first_event_time = tv;
        last_event_time = 0;
    } else {
        last_event_time = diffTime (tv, first_event_time);
    }
}

void Document::insertPosting (Command *e, const struct timeval &tv) {
    if (!notify_listener)
        return;
    bool postponed_sensible = e->postponable ();
    CommandData *prev = NULL;
    CommandData *ed = event_queue;
    for (; ed; ed = ed->next) {
        int diff = diffTime (ed->timeout, tv);
        bool psens = ed->event->postponable ();
        if ((diff > 0 && postponed_sensible == psens) || (!postponed_sensible && psens))
            break;
        prev = ed;
    }
    ed = new CommandData (e, ed);
    ed->timeout = tv;
    if (prev)
        prev->next = ed;
    else
        event_queue = ed;
    //debugLog () << "setTimeout " << ms << " at:" << pos << " tv:" << tv.tv_sec << "." << tv.tv_usec << endl;
}

void Document::setNextTimeout (const struct timeval &now) {
    if (!cur_event) {              // if we're not processing events
        int timeout = 0x7FFFFFFF;
        if (event_queue && active () &&
                (!postpone_ref || !event_queue->event->postponable ()))
            timeout = diffTime (event_queue->timeout, now);
        timeout = 0x7FFFFFFF != timeout
            ? (timeout > 0 ? timeout : 0)
            : TIMEOUT_OFF;
        if (TIMEOUT_OFF != cur_time_interval && TIMEOUT_OFF != timeout) {
            if (cur_time_interval > timeout - 5 && cur_time_interval < timeout + 5)
                return; // too small, let the timer repeat
        }
        cur_time_interval = timeout;
        notify_listener->setTimeout (timeout);
    }
}

void Document::updateTimeout () {
    if (!postpone_ref && event_queue && notify_listener) {
        struct timeval now;
        if (cur_event)
            now = cur_event->timeout;
        else
            timeOfDay (now);
        setNextTimeout (now);
    }
}

Command *Document::post (Command *e) {
    int ms = e->delay ();
    struct timeval now, tv;
    if (cur_event)
        now = cur_event->timeout;
    else
        timeOfDay (now);
    tv = now;
    addTime (tv, ms);
    insertPosting (e, tv);
    if (postpone_ref || event_queue->event == e)
        setNextTimeout (now);
    return e;
}

static CommandData *findPosting (CommandData *queue, CommandData **prev, const Command *e) {
    *prev = NULL;
    for (CommandData *ed = queue; ed; ed = ed->next) {
        if (e == ed->event)
            return ed;
        *prev = ed;
    }
    return NULL;
}

void Document::cancelPosting (Command *e) {
    if (cur_event && cur_event->event == e) {
        cur_event->event->removed ();
        cur_event->event = NULL;
    } else {
        CommandData *prev;
        CommandData **queue = &event_queue;
        CommandData *ed = findPosting (event_queue, &prev, e);
        if (!ed) {
            ed = findPosting (paused_queue, &prev, e);
            queue = &paused_queue;
        }
        if (ed) {
            if (prev) {
                prev->next = ed->next;
            } else {
                *queue = ed->next;
                if (!cur_event && queue == &event_queue) {
                    struct timeval now;
                    if (event_queue) // save a sys call
                        timeOfDay (now);
                    setNextTimeout (now);
                }
            }
            delete ed;
        } else {
            errorLog () << "Command not found" << endl;
        }
    }
}

void Document::pausePosting (Command *e) {
    if (cur_event && cur_event->event == e) {
        paused_queue = new CommandData (cur_event->event, paused_queue);
        paused_queue->timeout = cur_event->timeout;
        cur_event->event = NULL;
    } else {
        CommandData *prev;
        CommandData *ed = findPosting (event_queue, &prev, e);
        if (ed) {
            if (prev)
                prev->next = ed->next;
            else
                event_queue = ed->next;
            ed->next = paused_queue;
            paused_queue = ed;
        } else {
            errorLog () << "pauseEvent not found";
        }
    }
}

void Document::unpausePosting (Command *e, int ms) {
    CommandData *prev;
    CommandData *ed = findPosting (paused_queue, &prev, e);
    if (ed) {
        if (prev)
            prev->next = ed->next;
        else
            paused_queue = ed->next;
        addTime (ed->timeout, ms);
        insertPosting (ed->event, ed->timeout);
        ed->event = NULL;
        delete ed;
    } else {
        errorLog () << "pausePosting not found" << endl;
    }
}

void Document::timer () {
    struct timeval now;
    cur_event = event_queue;
    if (cur_event) {
        NodePtrW guard = this;
        struct timeval start = cur_event->timeout;
        timeOfDay (now);

        // handle max 100 timeouts with timeout set to now
        for (int i = 0; i < 100 && active (); ++i) {
            if (postpone_ref && cur_event->event->postponable ())
                break;
            // remove from queue
            event_queue = cur_event->next;

            CommandData *ed = cur_event;
            cur_event->event->apply (cur_event->timeout);
            ed->event = NULL;
            if (!guard) {
                delete ed;
                return;
            }
            delete cur_event;
            cur_event = event_queue;
            if (!cur_event || diffTime (cur_event->timeout, start) > 5)
                break;
        }
        cur_event = NULL;
    }
    setNextTimeout (now);
}

PostponePtr Document::postpone () {
    if (postpone_ref)
        return postpone_ref;
    debugLog () << "postponed " << endl;
    PostponePtr p = new Postpone (this);
    postpone_ref = p;
    PostponedEvent event (true);
    deliver (MsgEventPostponed, &event);
    if (notify_listener)
        notify_listener->enableRepaintUpdaters (false, 0);
    if (!cur_event) {
        struct timeval now;
        if (event_queue) // save a sys call
            timeOfDay (now);
        setNextTimeout (now);
    }
    return p;
}

void Document::proceed (const struct timeval &postponed_time) {
    debugLog () << "proceed" << endl;
    postpone_ref = NULL;
    struct timeval now;
    timeOfDay (now);
    int diff = diffTime (now, postponed_time);
    if (event_queue) {
        for (CommandData *ed = event_queue; ed; ed = ed->next)
            if (ed->event && ed->event->postponable ())
                addTime (ed->timeout, diff);
        setNextTimeout (now);
    }
    if (notify_listener)
        notify_listener->enableRepaintUpdaters (true, diff);
    PostponedEvent event (false);
    deliver (MsgEventPostponed, &event);
}

void *Document::role (RoleType msg, void *content) {
    if (RoleReceivers == msg) {
        MessageType m = (MessageType) (long) content;
        if (MsgEventPostponed == m)
            return &m_PostponedListeners;
    }
    return Mrl::role (msg, content);
}

//-----------------------------------------------------------------------------

KDE_NO_CDTOR_EXPORT TextNode::TextNode (NodePtr & d, const String & s, short i)
 : Node (d, i), text (s) {}

void TextNode::appendText (const String & s) {
    text += s;
}

String TextNode::nodeValue () const {
    return text;
}

//-----------------------------------------------------------------------------

KDE_NO_CDTOR_EXPORT
CData::CData (NodePtr & d, const String & s) : TextNode (d, s, id_node_cdata) {}

//-----------------------------------------------------------------------------

DarkNode::DarkNode (NodePtr & d, const String & n, short id)
 : Element (d, id), name (n) {
}

Node *DarkNode::childFromTag (const String & tag) {
    return new DarkNode (m_doc, tag);
}

//-----------------------------------------------------------------------------

GenericURL::GenericURL (NodePtr & d, const String & s, const String & name)
 : Mrl (d) {
    src = s;
    if (!src.isEmpty ())
        setAttribute (StringPool::attr_src, src);
    title = name;
    if (!name.isEmpty ())
        setAttribute (StringPool::attr_title, name);
}

KDE_NO_EXPORT void GenericURL::closed () {
    if (src.isEmpty ())
        src = getAttribute (StringPool::attr_src);
    Mrl::closed ();
}

//-----------------------------------------------------------------------------

GenericMrl::GenericMrl (NodePtr & d, const String & s, const String & name, const String & tag)
 : Mrl (d), node_name (tag) {
    src = s;
    if (!src.isEmpty ())
        setAttribute (StringPool::attr_src, src);
    title = name;
    if (!name.isEmpty ())
        setAttribute (StringPool::attr_name, name);
}

void GenericMrl::closed () {
    if (src.isEmpty ()) {
        src = getAttribute (StringPool::attr_src);
        if (src.isEmpty ())
            src = getAttribute (StringPool::attr_url);
    }
    if (title.isEmpty ())
        title = getAttribute (StringPool::attr_name);
    Mrl::closed ();
}

void *GenericMrl::role (RoleType msg, void *content)
{
    if (RolePlaylist == msg)
        return !title.isEmpty () || //return false if no title and only one
            previousSibling () || nextSibling ()
            ? (PlaylistRole *) this : NULL;
    return Mrl::role (msg, content);
}

//-----------------------------------------------------------------------------

void Visitor::visit (Element *elm) {
    visit (static_cast <Node *> (elm));
}

void Visitor::visit (TextNode *text) {
    visit (static_cast <Node *> (text));
}

//-----------------------------------------------------------------------------

namespace KMPlayer {

class KMPLAYER_NO_EXPORT DocumentBuilder {
    int m_ignore_depth;
    bool m_set_opener;
    bool m_root_is_first;
    NodePtr m_node;
    NodePtr m_root;
public:
    DocumentBuilder (NodePtr d, bool set_opener);
    ~DocumentBuilder () {}
    bool startTag (const String & tag, const AttributeList &attr);
    bool endTag (const String & tag);
    bool characterData (const String &data, bool append_only=false);
    bool cdataData (const String & data);
#ifdef HAVE_EXPAT
    void cdataStart ();
    void cdataEnd ();
private:
    bool in_cdata;
    String cdata;
#endif
};

} // namespace KMPlayer

DocumentBuilder::DocumentBuilder (NodePtr d, bool set_opener)
 : m_ignore_depth (0), m_set_opener (set_opener), m_root_is_first (false),
   m_node (d), m_root (d)
#ifdef HAVE_EXPAT
    , in_cdata (false)
#endif
{}

bool DocumentBuilder::startTag(const String &tag, const AttributeList &attr) {
    if (m_ignore_depth) {
        m_ignore_depth++;
        //kdDebug () << "Warning: ignored tag " << tag.latin1 () << " ignore depth = " << m_ignore_depth << endl;
    } else {
        NodePtr n = m_node ? m_node->childFromTag (tag) : NULL;
        if (!n) {
            //kdDebug () << "Warning: unknown tag " << tag.latin1 () << endl;
            NodePtr doc = m_root->document ();
            n = new DarkNode (doc, tag);
        }
        //kdDebug () << "Found tag " << tag << endl;
        if (n->isElementNode ())
            convertNode <Element> (n)->setAttributes (attr);
        if (m_node == n && m_node == m_root)
            m_root_is_first = true;
        else
            m_node->appendChild (n.ptr ());
        if (m_set_opener && m_node == m_root) {
            Mrl * mrl = n->mrl ();
            if (mrl)
                mrl->opener = m_root;
        }
        n->opened ();
        m_node = n;
    }
    return true;
}

bool DocumentBuilder::endTag (const String & tag) {
    if (m_ignore_depth) { // endtag to ignore
        m_ignore_depth--;
        //kdDebug () << "Warning: ignored end tag " << " ignore depth = " << m_ignore_depth <<  endl;
    } else {  // endtag
        NodePtr n = m_node;
        while (n) {
            if (!strcasecmp (n->nodeName (), (const char *) tag) &&
                    (m_root_is_first || n != m_root)) {
                while (n != m_node) {
                    warningLog()<< m_node->nodeName () << " not closed" << endl;
                    if (m_node->parentNode () == m_root)
                        break;
                    m_node->closed ();
                    m_node = m_node->parentNode ();
                }
                break;
            }
            if (n == m_root) {
                if (n == m_node) {
                    errorLog() << "m_node == m_doc, stack underflow " << endl;
                    return false;
                }
                warningLog()<<"endtag: no match " << (const char *) tag << endl;
                break;
            } else
                 warningLog() <<"tag "<<tag<< " not " << n->nodeName () << endl;
            n = n ->parentNode ();
        }
        //kdDebug () << "end tag " << tag << endl;
        m_node->closed ();
        m_node = m_node->parentNode ();
    }
    return true;
}

bool DocumentBuilder::characterData (const String &data, bool append_only) {
    if (!m_ignore_depth && m_node) {
#ifdef HAVE_EXPAT
        if (in_cdata)
            cdata += data;
        else
#endif
        {
            if (!data.isEmpty ()) {
                Node *last_node = m_node->lastChild ();
                if (append_only) {
                    if (last_node && last_node->id == id_node_text)
                        static_cast <TextNode *> (last_node)->appendText (data);
                } else {
                    if (!last_node || last_node->id != id_node_text) {
                        NodePtr d = m_node->document ();
                        last_node = new TextNode (d, String ());
                        m_node->appendChild (last_node);
                    }
                    static_cast <TextNode *> (last_node)->appendText (data);
                }
            }
        }
    }
    //kdDebug () << "characterData " << d.latin1() << endl;
    return true;
}

bool DocumentBuilder::cdataData (const String & data) {
    if (!m_ignore_depth) {
        NodePtr d = m_node->document ();
        m_node->appendChild (new CData (d, data));
    }
    //kdDebug () << "characterData " << d.latin1() << endl;
    return true;
}

#ifdef HAVE_EXPAT

void DocumentBuilder::cdataStart () {
    cdata.truncate (0);
    in_cdata = true;
}

void DocumentBuilder::cdataEnd () {
    cdataData (cdata);
    cdata.truncate (0);
    in_cdata = false;
}

static void startTag (void *data, const char * tag, const char **attr) {
    DocumentBuilder * builder = static_cast <DocumentBuilder *> (data);
    AttributeList attributes;
    if (attr && attr [0]) {
        for (int i = 0; attr[i]; i += 2)
            attributes.append (new Attribute (
                        TrieString(),
                        String::fromUtf8 (attr [i]),
                        String::fromUtf8 (attr [i+1])));
    }
    builder->startTag (String::fromUtf8 (tag), attributes);
}

static void endTag (void *data, const char * tag) {
    DocumentBuilder * builder = static_cast <DocumentBuilder *> (data);
    builder->endTag (String::fromUtf8 (tag));
}

static void characterData (void *data, const char *s, int len) {
    DocumentBuilder * builder = static_cast <DocumentBuilder *> (data);
    char * buf = new char [len + 1];
    strncpy (buf, s, len);
    buf[len] = 0;
    builder->characterData (String::fromUtf8 (buf));
    delete [] buf;
}

static void cdataStart (void *data) {
    DocumentBuilder * builder = static_cast <DocumentBuilder *> (data);
    builder->cdataStart ();
}

static void cdataEnd (void *data) {
    DocumentBuilder * builder = static_cast <DocumentBuilder *> (data);
    builder->cdataEnd ();
}

KMPLAYER_EXPORT
void KMPlayer::readXML (NodePtr root, TextStream & in, const String & firstline, bool set_opener) {
    bool ok = true;
    DocumentBuilder builder (root, set_opener);
    XML_Parser parser = XML_ParserCreate (0L);
    XML_SetUserData (parser, &builder);
    XML_SetElementHandler (parser, startTag, endTag);
    XML_SetCharacterDataHandler (parser, characterData);
    XML_SetCdataSectionHandler (parser, cdataStart, cdataEnd);
    if (!firstline.isEmpty ()) {
        String str (firstline + char_nl);
        QCString buf = str.utf8 ();
        ok = XML_Parse(parser, buf, strlen (buf), false) != XML_STATUS_ERROR;
        if (!ok)
            warningLog() << XML_ErrorString(XML_GetErrorCode(parser)) << " at " << XML_GetCurrentLineNumber(parser) << " col " << XML_GetCurrentColumnNumber(parser) << endl;
    }
    if (ok && !in.atEnd ()) {
        QCString buf = in.read ().utf8 ();
        ok = XML_Parse(parser, buf, strlen (buf), true) != XML_STATUS_ERROR;
        if (!ok)
            warningLog() << XML_ErrorString(XML_GetErrorCode(parser)) << " at " << XML_GetCurrentLineNumber(parser) << " col " << XML_GetCurrentColumnNumber(parser) << endl;
    }
    XML_ParserFree(parser);
    root->normalize ();
    //return ok;
}

//-----------------------------------------------------------------------------
#else // HAVE_EXPAT

namespace {

class SimpleSAXParser {
    enum Token { tok_empty, tok_text, tok_white_space, tok_angle_open,
        tok_equal, tok_double_quote, tok_single_quote, tok_angle_close,
        tok_slash, tok_exclamation, tok_amp, tok_hash, tok_colon,
        tok_semi_colon, tok_question_mark, tok_cdata_start };
public:
    struct TokenInfo {
        TokenInfo () : token (tok_empty) {}
        void *operator new (size_t);
        void operator delete (void *);
        Token token;
        String string;
        SharedPtr <TokenInfo> next;
    };
    typedef SharedPtr <TokenInfo> TokenInfoPtr;
    SimpleSAXParser (DocumentBuilder & b) : builder (b), position (0), equal_seen (false), in_dbl_quote (false), in_sngl_quote (false), have_error (false), no_entitity_look_ahead (false), have_next_char (false) {}
    ~SimpleSAXParser ();
    bool parse (TextStream & d);
private:
    TextStream * data;
    DocumentBuilder & builder;
    int position;
    Char next_char;
    enum State {
        InTag, InStartTag, InPITag, InDTDTag, InEndTag, InAttributes, InContent, InCDATA, InComment
    };
    struct StateInfo {
        StateInfo (State s, SharedPtr <StateInfo> n) : state (s), next (n) {}
        State state;
        String data;
        SharedPtr <StateInfo> next;
    };
    SharedPtr <StateInfo> m_state;
    TokenInfoPtr next_token, token, prev_token;
    // for element reading
    String tagname;
    AttributeList m_attributes;
    String attr_namespace, attr_name, attr_value;
    String cdata;
    bool equal_seen;
    bool in_dbl_quote;
    bool in_sngl_quote;
    bool have_error;
    bool no_entitity_look_ahead;
    bool have_next_char;

    bool readTag ();
    bool readEndTag ();
    bool readAttributes ();
    bool readPI ();
    bool readDTD ();
    bool readCDATA ();
    bool readComment ();
    bool nextToken ();
    void push ();
    void push_attribute ();
};

} // namespace


static FixedSizeAllocator *token_info_pool;

inline void *SimpleSAXParser::TokenInfo::operator new (size_t s) {
    if (!token_info_pool)
        token_info_pool = new FixedSizeAllocator (sizeof (TokenInfo));

    return token_info_pool->alloc ();
}

inline void SimpleSAXParser::TokenInfo::operator delete (void *p) {
    token_info_pool->dealloc (p);
}

static const Char char_semi (';');
static const Char char_eq ('=');
static const Char char_squot ('\'');
static const Char char_qu ('?');
static const Char char_hash ('#');
static const Char char_bang ('!');
static const Char char_slash ('/');

inline SimpleSAXParser::~SimpleSAXParser () {};

KMPLAYER_EXPORT
void KMPlayer::readXML (NodePtr root, TextStream & in, const String & firstline, bool set_opener) {
    DocumentBuilder builder (root, set_opener);
    SimpleSAXParser parser (builder);
    if (!firstline.isEmpty ()) {
        String str (firstline + char_nl);
        TextStream fl_in (&str /*, IO_ReadOnly*/);
        parser.parse (fl_in);
    }
    if (!in.atEnd ())
        parser.parse (in);
    for (NodePtr e = root->parentNode (); e; e = e->parentNode ()) {
        if (e->open)
            break;
        e->closed ();
    }
    //doc->normalize ();
    //kdDebug () << root->outerXML ();
}

void SimpleSAXParser::push () {
    if (!next_token->string.isEmpty ()) {
        prev_token = token;
        token = next_token;
        if (prev_token)
            prev_token->next = token;
        next_token = TokenInfoPtr (new TokenInfo);
        //kdDebug () << "push " << token->string << endl;
    }
}

void SimpleSAXParser::push_attribute () {
    //kdDebug () << "attribute " << attr_name.latin1 () << "=" << attr_value.latin1 () << endl;
    m_attributes.append(new Attribute (attr_namespace, attr_name, attr_value));
    attr_namespace.clear ();
    attr_name.truncate (0);
    attr_value.truncate (0);
    equal_seen = in_sngl_quote = in_dbl_quote = false;
}

bool SimpleSAXParser::nextToken () {
    TokenInfoPtr cur_token = token;
    while ((have_next_char || !data->atEnd ()) &&
           cur_token == token && !(token && token->next)) {
        const gchar *at = NULL;
        if (have_next_char) {
            have_next_char = false;
        } else {
            at = data->at ();
            *data >> next_char;
        }
        bool append_char = true;
        if (next_char.isSpace ()) {
            if (next_token->token != tok_white_space)
                push ();
            next_token->token = tok_white_space;
        } else if (!next_char.isLetterOrNumber ()) {
            if (next_char == char_hash) {
                //if (next_token->token == tok_empty) { // check last item on stack &
                    push ();
                    next_token->token = tok_hash;
                //}
            } else if (next_char == char_slash) {
                push ();
                next_token->token = tok_slash;
            } else if (next_char == char_bang) {
                push ();
                next_token->token = tok_exclamation;
            } else if (next_char == char_qu) {
                push ();
                next_token->token = tok_question_mark;
            } else if (next_char == char_lt) {
                push ();
                next_token->token = tok_angle_open;
            } else if (next_char == char_gt) {
                push ();
                next_token->token = tok_angle_close;
            } else if (InAttributes == m_state->state &&
                    next_char == Char (':')) {
                push ();
                next_token->token = tok_colon;
            } else if (next_char == char_semi) {
                push ();
                next_token->token = tok_semi_colon;
            } else if (next_char == char_eq) {
                push ();
                next_token->token = tok_equal;
            } else if (next_char == char_quot) {
                push ();
                next_token->token = tok_double_quote;
            } else if (next_char == char_squot) {
                push ();
                next_token->token = tok_single_quote;
            } else if (next_char == char_amp) {
                push ();
                if (no_entitity_look_ahead) {
                    have_next_char = true;
                    break;
                }
                append_char = false;
                at = NULL;
                no_entitity_look_ahead = true;
                TokenInfoPtr tmp = token;
                TokenInfoPtr prev_tmp = prev_token;
                if (nextToken () && token->token == tok_text &&
                        nextToken () && token->token == tok_semi_colon) {
                    if (prev_token->string == "amp")
                        token->string = char_amp;
                    else if (prev_token->string == "lt")
                        token->string = char_lt;
                    else if (prev_token->string == "gt")
                        token->string = char_gt;
                    else if (prev_token->string == "quot")
                        token->string = char_quot;
                    else if (prev_token->string == "apos")
                        token->string = char_squot;
                    else if (prev_token->string == "copy")
                        token->string = Char (169);
                    else
                        token->string = char_qu;// TODO lookup more ..
                    token->token = tok_text;
                    if (tmp) { // cut out the & xxx ; tokens
                        tmp->next = token;
                        token = tmp;
                    }
                    //kdDebug () << "entity found "<<prev_token->string << endl;
                } else if (token->token == tok_hash &&
                        nextToken () && token->token == tok_text &&
                        nextToken () && token->token == tok_semi_colon) {
                    //kdDebug () << "char entity found " << prev_token->string << prev_token->string.toInt (0L, 16) << endl;
                    token->token = tok_text;
                    if (!prev_token->string.startsWith (Char ('x')))
                        token->string = Char (prev_token->string.toInt ());
                    else
                        token->string = Char (prev_token->string.mid (1).toInt (0L, 16));
                    if (tmp) { // cut out the '& # xxx ;' tokens
                        tmp->next = token;
                        token = tmp;
                    }
                } else {
                    token = tmp; // restore and insert the lost & token
                    tmp = TokenInfoPtr (new TokenInfo);
                    tmp->token = tok_amp;
                    tmp->string += char_amp;
                    tmp->next = token->next;
                    if (token)
                        token->next = tmp;
                    else
                        token = tmp; // hmm
                }
                no_entitity_look_ahead = false;
                prev_token = prev_tmp;
            } else if (next_token->token != tok_text) {
                push ();
                next_token->token = tok_text;
            }
        } else if (next_token->token != tok_text) {
            push ();
            next_token->token = tok_text;
        }
        if (at && !data->atEnd () && tok_text == next_token->token) {
            const gchar *end;
            do {
                end = data->at ();
                *data >> next_char;
            } while (!data->atEnd () && next_char.isLetterOrNumber ());
            next_token->string.append (at, end - at);
            have_next_char = true;
        } else if (append_char) {
            next_token->string += next_char;
        }
        if (next_token->token == tok_text &&
                next_char == Char ('[' ) && next_token->string == "[CDATA[") {
            next_token->token = tok_cdata_start;
            break;
        }
    }
    if (token == cur_token) {
        if (token && token->next) {
            prev_token = token;
            token = token->next;
        } else if (!next_token->string.isEmpty ()) {
            push (); // last token
        } else
            return false;
        return true;
    }
    return true;
}

bool SimpleSAXParser::readAttributes () {
    bool closed = false;
    while (true) {
        if (!nextToken ()) return false;
        //kdDebug () << "readAttributes " << token->string.latin1() << endl;
        if ((in_dbl_quote && token->token != tok_double_quote) ||
                    (in_sngl_quote && token->token != tok_single_quote)) {
            attr_value += token->string;
        } else if (token->token == tok_equal) {
            if (attr_name.isEmpty ())
                return false;
            if (equal_seen)
                attr_value += token->string; // EQ=a=2c ???
            //kdDebug () << "equal_seen"<< endl;
            equal_seen = true;
        } else if (token->token == tok_white_space) {
            if (!attr_value.isEmpty ())
                push_attribute ();
        } else if (token->token == tok_single_quote) {
            if (!equal_seen)
                attr_name += token->string; // D'OH=xxx ???
            else if (in_sngl_quote) { // found one
                push_attribute ();
            } else if (attr_value.isEmpty ())
                in_sngl_quote = true;
            else
                attr_value += token->string;
        } else if (token->token == tok_colon) {
            if (equal_seen) {
                attr_value += token->string;
            } else {
                attr_namespace = attr_name;
                attr_name.clear();
            }
        } else if (token->token == tok_double_quote) {
            if (!equal_seen)
                attr_name += token->string; // hmm
            else if (in_dbl_quote) { // found one
                push_attribute ();
            } else if (attr_value.isEmpty ())
                in_dbl_quote = true;
            else
                attr_value += token->string;
            //kdDebug () << "in_dbl_quote:"<< in_dbl_quote << endl;
        } else if (token->token == tok_slash) {
            TokenInfoPtr mark_token = token;
            if (nextToken () &&
                    (token->token != tok_white_space || nextToken()) &&//<e / >
                    token->token == tok_angle_close) {
            //kdDebug () << "close mark:"<< endl;
                closed = true;
                break;
            } else {
                token = mark_token;
            //kdDebug () << "not end mark:"<< equal_seen << endl;
                if (equal_seen)
                    attr_value += token->string; // ABBR=w/o ???
                else
                    attr_name += token->string;
            }
        } else if (token->token == tok_angle_close) {
            if (!attr_name.isEmpty ())
                push_attribute ();
            break;
        } else if (equal_seen) {
            attr_value += token->string;
        } else {
            attr_name += token->string;
        }
    }
    m_state = m_state->next;
    if (m_state->state == InPITag) {
        if (tagname == "xml") {
            /*const AttributeMap::const_iterator e = attr.end ();
            for (AttributeMap::const_iterator i = attr.begin (); i != e; ++i)
                if (!strcasecmp (i.key ().latin1 (), "encoding"))
                  kdDebug () << "encodeing " << i.data().latin1() << endl;*/
        }
    } else {
        have_error = builder.startTag (tagname, m_attributes);
        if (closed)
            have_error &= builder.endTag (tagname);
        //kdDebug () << "readTag " << tagname << " closed:" << closed << " ok:" << have_error << endl;
    }
    m_state = m_state->next; // pop Node or PI
    return true;
}

bool SimpleSAXParser::readPI () {
    // TODO: <?xml .. encoding="ENC" .. ?>
    if (!nextToken ()) return false;
    if (token->token == tok_text && !token->string.compare ("xml")) {
        m_state = new StateInfo (InAttributes, m_state);
        return readAttributes ();
    } else {
        while (nextToken ())
            if (token->token == tok_angle_close) {
                m_state = m_state->next;
                return true;
            }
    }
    return false;
}

bool SimpleSAXParser::readDTD () {
    //TODO: <!ENTITY ..>
    if (!nextToken ()) return false;
    if (token->token == tok_text && token->string.startsWith (String ("--"))) {
        m_state = new StateInfo (InComment, m_state->next); // note: pop DTD
        return readComment ();
    }
    //kdDebug () << "readDTD: " << token->string.latin1 () << endl;
    if (token->token == tok_cdata_start) {
        m_state = new StateInfo (InCDATA, m_state->next); // note: pop DTD
        if (token->next) {
            cdata = token->next->string;
            token->next = 0;
        } else {
            cdata = next_token->string;
            next_token->string.truncate (0);
            next_token->token = tok_empty;
        }
        return readCDATA ();
    }
    while (nextToken ())
        if (token->token == tok_angle_close) {
            m_state = m_state->next;
            return true;
        }
    return false;
}

bool SimpleSAXParser::readCDATA () {
    while (!data->atEnd ()) {
        *data >> next_char;
        if (next_char == char_gt && cdata.endsWith (String ("]]"))) {
            cdata.truncate (cdata.length () - 2);
            m_state = m_state->next;
            if (m_state->state == InContent)
                have_error = builder.cdataData (cdata);
            else if (m_state->state == InAttributes) {
                if (equal_seen)
                    attr_value += cdata;
                else
                    attr_name += cdata;
            }
            cdata.truncate (0);
            return true;
        }
        cdata += next_char;
    }
    return false;
}

bool SimpleSAXParser::readComment () {
    while (nextToken ()) {
        if (token->token == tok_angle_close && prev_token)
            if (prev_token->string.endsWith (String ("--"))) {
                m_state = m_state->next;
                return true;
            }
    }
    return false;
}

bool SimpleSAXParser::readEndTag () {
    if (!nextToken ()) return false;
    if (token->token == tok_white_space)
        if (!nextToken ()) return false;
    tagname = token->string;
    if (!nextToken ()) return false;
    if (token->token == tok_white_space)
        if (!nextToken ()) return false;
    if (token->token != tok_angle_close)
        return false;
    have_error = builder.endTag (tagname);
    m_state = m_state->next;
    return true;
}

// TODO: <!ENTITY ..> &#1234;
bool SimpleSAXParser::readTag () {
    if (!nextToken ()) return false;
    if (token->token == tok_exclamation) {
        m_state = new StateInfo (InDTDTag, m_state->next);
    //kdDebug () << "readTag: " << token->string.latin1 () << endl;
        return readDTD ();
    }
    if (token->token == tok_white_space)
        if (!nextToken ()) return false; // allow '< / foo', '<  foo', '< ? foo'
    if (token->token == tok_question_mark) {
        m_state = new StateInfo (InPITag, m_state->next);
        return readPI ();
    }
    if (token->token == tok_slash) {
        m_state = new StateInfo (InEndTag, m_state->next);
        return readEndTag ();
    }
    if (token->token != tok_text)
        return false; // FIXME entities
    tagname = token->string;
    //kdDebug () << "readTag " << tagname.latin1() << endl;
    m_state = new StateInfo (InAttributes, m_state);
    return readAttributes ();
}

bool SimpleSAXParser::parse (TextStream & d) {
    data = &d;
    if (!next_token) {
        next_token = TokenInfoPtr (new TokenInfo);
        m_state = new StateInfo (InContent, m_state);
    }
    bool ok = true;
    bool in_character_data = false;
    String white_space;
    while (ok) {
        switch (m_state->state) {
            case InTag:
                ok = readTag ();
                break;
            case InPITag:
                ok = readPI ();
                break;
            case InDTDTag:
                ok = readDTD ();
                break;
            case InEndTag:
                ok = readEndTag ();
                break;
            case InAttributes:
                ok = readAttributes ();
                break;
            case InCDATA:
                ok = readCDATA ();
                break;
            case InComment:
                ok = readComment ();
                break;
            default:
                if ((ok = nextToken ())) {
                    if (token->token == tok_angle_open) {
                        attr_name.truncate (0);
                        attr_value.truncate (0);
                        m_attributes = AttributeList ();
                        equal_seen = in_sngl_quote = in_dbl_quote = false;
                        m_state = new StateInfo (InTag, m_state);
                        ok = readTag ();
                        in_character_data = false;
                        have_error = builder.characterData (white_space, true);
                        white_space.truncate (0);
                    } else if (token->token == tok_white_space) {
                        white_space += token->string;
                    } else {
                        if (!white_space.isEmpty ()) {
                            if (!in_character_data) {
                                int pos = white_space.lastIndexOf (char_nl);
                                if (pos > -1)
                                    white_space = white_space.mid (pos + 1);
                            }
                            have_error = builder.characterData (white_space);
                            white_space.truncate (0);
                        }
                        have_error = builder.characterData (token->string);
                        in_character_data = true;
                    }
                }
        }
        if (!m_state)
            return true; // end document
    }
    return false; // need more data
}

#endif // HAVE_EXPAT
/*
#include <stdio.h>

int main () {
    const char * xmldata =
        //"<document><head><layout></layout></head><body></document>";
        "<ASX><entry><ref href='foo' dummy='yes'/></Asx>";
    TextStream in;
    in << xmldata;
    //TextStream in (String (xmldata));
    NodePtr doc = new Document (String ("foobar"), 0L);
    String str;
    {
    readXML (doc, in, str);
    }
    printf ("out: %s\n", (const char *) doc->outerXML ());
    doc->document ()->dispose ();
    return 0;
}
*/
