/* This file is part of the KMPlayer project
 *
 * Copyright (C) 2006 Koos Vriezen <koos.vriezen@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _KMPLAYER_LISTS_H_
#define _KMPLAYER_LISTS_H_

#include "kmplayerplaylist.h"
#include "kmplayercontrol.h"

typedef struct _Application Application;

namespace KMPlayer {

class UnixProcess;
class UnixProcessListener;

static const short id_node_recent_document = 30;
static const short id_node_recent_node = 31;
static const short id_node_disk_document = 32;
static const short id_node_disk_node = 33;
static const short id_node_html_object = 34;
static const short id_node_html_embed = 35;
static const short id_node_gen_generator = 36;
static const short id_node_gen_input = 37;
static const short id_node_gen_uri = 38;
static const short id_node_gen_literal = 39;
static const short id_node_gen_ask = 40;
static const short id_node_gen_title = 41;
static const short id_node_gen_description = 42;
static const short id_node_gen_process = 43;
static const short id_node_gen_program = 44;
static const short id_node_gen_argument = 45;
static const short id_node_gen_predefined = 46;
static const short id_node_gen_document = 47;
static const short id_node_gen_http_get = 48;
static const short id_node_gen_http_key_value = 49;
static const short id_node_gen_sequence = 50;

inline bool isListItem (short id) {
    return id_node_recent_node == id ||
        id_node_playlist_item == id ||
        id_node_html_object == id;
}

class KMPLAYER_NO_EXPORT FileDocument : public KMPlayer::SourceDocument {
public:
    FileDocument (short id, const String &, KMPlayer::Control * notify = 0L);
    KMPlayer::Node *childFromTag (const String & tag);
    void readFromFile (const String & file);
    void writeToFile (const String & file);
    void sync (const String & file);
    unsigned int load_tree_version;
};

class KMPLAYER_NO_EXPORT Recents : public FileDocument {
public:
    Recents (Application *a);
    void defer ();
    void activate ();
    void message (KMPlayer::MessageType msg, void *content=NULL);
    KMPlayer::Node *childFromTag (const String & tag);
    KDE_NO_EXPORT const char * nodeName () const { return "playlist"; }
    String path;
    Application * app;
};

class KMPLAYER_NO_EXPORT Recent : public KMPlayer::Mrl {
public:
    Recent (KMPlayer::NodePtr & doc, Application *a, const String &url = String());
    void activate ();
    void closed ();
    KDE_NO_EXPORT const char * nodeName () const { return "item"; }
    Application * app;
};

class KMPLAYER_NO_EXPORT Group : public Element, public PlaylistRole
{
public:
    Group (KMPlayer::NodePtr &doc, Application *a, const String &pn=String());
    KMPlayer::Node *childFromTag (const String & tag);
    void closed ();
    void *role (KMPlayer::RoleType msg, void *content=NULL);
    KDE_NO_EXPORT const char * nodeName () const { return "group"; }
    Application * app;
};

class KMPLAYER_NO_EXPORT Playlist : public FileDocument {
public:
    Playlist (Application *a, KMPlayer::Control *n, bool plmod = false);
    void message (KMPlayer::MessageType msg, void *content=NULL);
    void defer ();
    void activate ();
    KMPlayer::Node *childFromTag (const String & tag);
    KDE_NO_EXPORT const char * nodeName () const { return "playlist"; }
    Application * app;
    String path;
    bool playmode;
};

class KMPLAYER_NO_EXPORT PlaylistItemBase : public KMPlayer::Mrl {
public:
    PlaylistItemBase (KMPlayer::NodePtr &d, short id, Application *a, bool pm);
    void activate ();
    void closed ();
    Application * app;
    bool playmode;
};

class KMPLAYER_NO_EXPORT PlaylistItem : public PlaylistItemBase {
public:
    PlaylistItem (KMPlayer::NodePtr & doc, Application *a, bool playmode, const String &url = String());
    void closed ();
    void begin ();
    KDE_NO_EXPORT const char * nodeName () const { return "item"; }
};

class KMPLAYER_NO_EXPORT PlaylistGroup : public Element, public PlaylistRole
{
public:
    PlaylistGroup (KMPlayer::NodePtr &doc, Application *a, const String &pn);
    PlaylistGroup (KMPlayer::NodePtr &doc, Application *a, bool plmode=false);
    KMPlayer::Node *childFromTag (const String & tag);
    void closed ();
    void *role (KMPlayer::RoleType msg, void *content=NULL);
    KDE_NO_EXPORT const char * nodeName () const { return "group"; }
    Application * app;
    bool playmode;
};

class KMPLAYER_NO_EXPORT HtmlObject : public PlaylistItemBase {
public:
    HtmlObject (KMPlayer::NodePtr & doc, Application *a, bool playmode);
    void activate ();
    void closed ();
    KMPlayer::Node *childFromTag (const String & tag);
    KDE_NO_EXPORT const char * nodeName () const { return "object"; }
};

class KMPLAYER_NO_EXPORT Generator
    : public FileDocument, public UnixProcessListener, public IOJobListener {
public:
    Generator (KMPlayer::Control *src)
        : FileDocument (id_node_gen_document, String (), src),
          unix_process (NULL), input_job (NULL), data (NULL), buffer (NULL) {}
    void activate ();
    void begin ();
    void deactivate ();
    Node *childFromTag (const String & tag);
    KDE_NO_EXPORT const char *nodeName () const { return "generator"; }

    void readEvent (UnixProcess *);
    void writeEvent (UnixProcess *);
    void processExited (UnixProcess *);

    void jobData (IOJob *job, ByteArray &data);
    void jobResult (IOJob *job);
    void redirected (IOJob*, const String&) {}

private:
    String genReadProcess (Control *ctrl, Node *n);
    String genReadInput (Control *ctrl, Node *n);
    String genReadString (Control *ctr, Node *n);
    String genReadUriGet (Control *ctr, Node *n);
    String genReadAsk (Control *ctr, Node *n);

    UnixProcess *unix_process;
    IOJob *input_job;
    TextStream *data;
    ByteArray *buffer;
    String process;
    bool canceled;
    bool quote;
};

class KMPLAYER_NO_EXPORT GeneratorElement : public Element {
public:
    GeneratorElement (NodePtr &doc, const String &t, short id)
        : Element (doc, id), tag (t) {}
    Node *childFromTag (const String & tag);
    KDE_NO_EXPORT const char *nodeName () const { return (const char *) tag; }
    String tag;
};

}  // KMPlayer namespace

#endif //_KMPLAYER_LISTS_H_

