/*
    This file is part of Kismet

    Kismet is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    Kismet is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Kismet; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "config.h"

#ifdef HAVE_LIBLOCATION

#include "gpsliblocation.h"
#include "configfile.h"
#include "soundcontrol.h"
#include "packetchain.h"

// liblocation callback functions
void liblocation_changed(LocationGPSDevice *device, gpointer userdata) {
	((GPSLibLocation*) userdata)->LocChanged(device);
}
void liblocation_connected(LocationGPSDevice *new_device, gpointer userdata) {
	((GPSLibLocation*) userdata)->LocConnected();
}
void liblocation_disconnected(LocationGPSDevice *new_device, gpointer userdata) {
	((GPSLibLocation*) userdata)->LocDisconnected();
}
void liblocation_started(LocationGPSDControl *control, gpointer userdata) {
	((GPSLibLocation*) userdata)->LocStarted();
}
void liblocation_stopped(LocationGPSDControl *control, gpointer userdata) {
	((GPSLibLocation*) userdata)->LocStopped();
}
void liblocation_error(LocationGPSDControl *control, gpointer userdata) {
	((GPSLibLocation*) userdata)->LocError();
}

// event loop thread
void *event_loop(void *arg) {
	/* Clear the thread sigmask so we don't catch sigterm weirdly */
	sigset_t sset;
	sigfillset(&sset);
	pthread_sigmask(SIG_BLOCK, &sset, NULL);

	g_main_loop_run((GMainLoop *) arg);

	return NULL;
}

GPSLibLocation::GPSLibLocation() {
    fprintf(stderr, "FATAL OOPS: gpsliblocation called with no globalreg\n");
	exit(-1);
}

GPSLibLocation::GPSLibLocation(GlobalRegistry *in_globalreg) : GPSCore(in_globalreg) {
	// init
	g_type_init();
	loop = g_main_loop_new(NULL, FALSE);

	device = (LocationGPSDevice*) g_object_new(LOCATION_TYPE_GPS_DEVICE, NULL);
	control = location_gpsd_control_get_default();

	//g_object_set(G_OBJECT(control), "preferred-method", LOCATION_METHOD_CWP | LOCATION_METHOD_ACWP | LOCATION_METHOD_GNSS | LOCATION_METHOD_AGNSS, NULL);
	//g_object_set(G_OBJECT(control), "preferred-interval", LOCATION_INTERVAL_1S, NULL);

	//last_disconnect = 0;

	ScanOptions();
	RegisterComponents();

	// setup callback functions
	sig_handler_error = g_signal_connect(control, "error", G_CALLBACK(liblocation_error), this);
	sig_handler_started = g_signal_connect(control, "gpsd-running", G_CALLBACK(liblocation_started), this);
	sig_handler_stopped = g_signal_connect(control, "gpsd-stopped", G_CALLBACK(liblocation_stopped), this);
	sig_handler_changed = g_signal_connect(device, "changed", G_CALLBACK(liblocation_changed), this);
	sig_handler_connected = g_signal_connect(device, "connected", G_CALLBACK(liblocation_connected), this);
	sig_handler_disconnected = g_signal_connect(device, "disconnected", G_CALLBACK(liblocation_disconnected), this);

	if (Reconnect() < 0)
		return;

	_MSG("Using liblocation", MSGFLAG_INFO);
}

GPSLibLocation::~GPSLibLocation() {
	// Unregister ourselves from the main tcp service loop
	globalreg->RemovePollableSubsys(this);

	Shutdown();

	g_signal_handler_disconnect(device, sig_handler_started);
	g_signal_handler_disconnect(device, sig_handler_connected);
	g_signal_handler_disconnect(device, sig_handler_changed);
	g_signal_handler_disconnect(device, sig_handler_disconnected);
	g_signal_handler_disconnect(device, sig_handler_stopped);
	g_signal_handler_disconnect(device, sig_handler_error);
}

int GPSLibLocation::Shutdown() {
	_MSG("stopping liblocation", MSGFLAG_DEBUG);
	location_gpsd_control_stop(control);

	g_main_loop_quit(loop);

	pthread_join(event_loop_thread, NULL);

	return 1;
}

int GPSLibLocation::Reconnect() {
	_MSG("starting liblocation", MSGFLAG_DEBUG);

	pthread_create(&event_loop_thread, NULL, event_loop, loop);

	location_gpsd_control_start(control);

	return 1;
}

void GPSLibLocation::LocStarted() {
	_MSG("liblocation started", MSGFLAG_DEBUG);
	GPSCore::Timer();
}

void GPSLibLocation::LocStopped() {
	_MSG("liblocation stopped", MSGFLAG_DEBUG);
	GPSCore::Timer();
}

void GPSLibLocation::LocError() {
	_MSG("liblocation error", MSGFLAG_DEBUG);
	gps_connected = 0;
	GPSCore::Timer();
}

void GPSLibLocation::LocConnected() {
	_MSG("liblocation connected", MSGFLAG_DEBUG);
	gps_connected = 1;
	GPSCore::Timer();
}

void GPSLibLocation::LocDisconnected() {
	_MSG("liblocation disconnected", MSGFLAG_DEBUG);
	gps_connected = 0;
	GPSCore::Timer();
}

void GPSLibLocation::LocChanged(LocationGPSDevice *device) {
	//_MSG("liblocation changed", MSGFLAG_DEBUG);
	if (!device) {
		return;
	}
	
	if (device->fix) {
		mode = device->fix->mode;
		if (device->fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET) {
			lat = device->fix->latitude; // degree
			lon = device->fix->longitude; // degree
			hdop = vdop = device->fix->eph / 100; // cm -> m
		}
		if (device->fix->fields & LOCATION_GPS_DEVICE_ALTITUDE_SET) {
			alt = device->fix->altitude; // m
		}
		if (device->fix->fields & LOCATION_GPS_DEVICE_SPEED_SET) {
			spd = device->fix->speed / 3.6; // km/h -> m/s
		}
		if (device->fix->fields & LOCATION_GPS_DEVICE_TRACK_SET) {
			hed = device->fix->track; // degree
		}
	}
	if (device->satellites) {
		LocationGPSDeviceSatellite *sat;
		GPSCore::sat_pos sp;
		sat_pos_map.clear();

		for (int i = 0; i < device->satellites_in_view; i++) {
			sat = (LocationGPSDeviceSatellite*) g_ptr_array_index(device->satellites, i);
			sp.prn = sat->prn;
			sp.elevation = sat->elevation;
			sp.azimuth = sat->azimuth;
			sp.snr = sat->signal_strength;
			sat_pos_map[sp.prn] = sp;
		}
	}

	GPSCore::Timer();
}

#endif

