#include "check_libkimi_module.h"
#include "kimi.h"

#include <check.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>


#define BUF_SIZE 512

Kimi* kimi = NULL;

void setup()
{
    GPtrArray* a = g_ptr_array_new();
    KimiUICallbacks back;
    back.show_options = check_libkimi_settings_callback;
    kimi = kimi_init(a, back, NULL);
    check_libkimi_initialize(kimi);
    g_ptr_array_free(a, TRUE);
}

void teardown()
{
    CHData* data = check_libkimi_get_data(kimi);

    if (data->stored_event)
        kimi_event_free(data->stored_event, NULL);

    kimi_deinit(kimi);
    kimi = NULL;
}


START_TEST(check_init)
{
    fail_unless(kimi != NULL, "Cannot initialize kimi");
    CHData* data = check_libkimi_get_data(kimi);
    fail_unless(data != NULL, "Cannot get module data");
    fail_unless(data->test_value == 256, "Module data is wrong");
}
END_TEST

START_TEST(create_event)
{
    GError* err = NULL;
    Event* ev = kimi_event_new(&err);
    if (err) {
        g_error_free(err);
        fail("%s\n", err->message);
        err = NULL;
    }
    kimi_event_free(ev, NULL);
}
END_TEST

START_TEST(create_event_long)
{
    Period* per = kimi_per_create_period();
    Event* ev = kimi_event_new(NULL);
    kimi_event_fill(ev,
                    MOD_ID,
                    "id",
                    "QWERTY", 
                    "DESCRIPTION",
                    "LOCATION",
                    kimi_per_get_time_t("20100101T123434"),
                    kimi_per_get_time_t("20100101T133434"),
                    per,
                    NULL);

    fail_unless(strcmp(ev->service_event_id, MOD_ID) == 0);
    
    fail_unless(strcmp(ev->title, "QWERTY") == 0);
    fail_unless(strcmp(ev->description, "DESCRIPTION") == 0);
    fail_unless(strcmp(ev->location, "LOCATION") == 0);

    fail_unless(ev->start_time == kimi_per_get_time_t("20100101T123434"));
    fail_unless(ev->end_time == kimi_per_get_time_t("20100101T133434"));
    
    fail_unless(strcmp(ev->id, "id") == 0);
    fail_unless(ev->per == per);

    kimi_event_free(ev, NULL);
}
END_TEST

START_TEST(store_event)
{
    GError* err = NULL;
    CHData* data = check_libkimi_get_data(kimi);
    Event* ev = kimi_event_new(NULL);
    int code = kimi_event_fill(ev,
            "ch",
            NULL,
            "title",
            "description",
            "location",
            10000000,
            10000001,
            NULL,
            &err    
            );
    fail_unless(code == 0, "kimi_event_fill returned -1");
    code = kimi_store(ev, kimi, &err);

    fail_unless(ev == data->stored_event);
}
END_TEST

START_TEST(store_event_to_wrong_module)
{
    GError* err = NULL;
    CHData* data = check_libkimi_get_data(kimi);
    Event* ev = kimi_event_new(NULL);
    int code = kimi_event_fill(ev,
            "�,*��oD��`�:n�",
            NULL,
            "title",
            "description",
            "location",
            10000000,
            10000001,
            NULL,
            &err
            );
    fail_unless(code == 0, "kimi_event_fill returned -1");
    code = kimi_store(ev, kimi, &err);
    fail_unless((code == -1) == (err != NULL));
    fail_unless(ev != data->stored_event);

    if (!code) {
        fail_unless(err->domain == KIMI_ERROR);
        fail_unless(err->code == KIMI_NO_SUCH_MODULE);
        g_error_free(err);
        err = NULL;
    }
}
END_TEST

START_TEST(store_event_with_error)
{  
    GError* err = NULL;
    CHData* data = check_libkimi_get_data(kimi);
    Event* ev = kimi_event_new(NULL);
    int code = kimi_event_fill(ev,
            "ch",
            NULL,
            "title",
            "description",
            "location",
            10000000,
            10000001,
            NULL,
            &err
            );
    fail_unless(code == 0, "kimi_event_fill returned -1");
    
    data->error = 1;
    code = kimi_store(ev, kimi, &err);
    fail_unless((code == -1) == (err != NULL));
    
    if (err) {
        fail_unless(err->domain == KIMI_ERROR);
        fail_unless(err->code == KIMI_STORE_EVENT_FAILED);
        g_error_free(err);
        err = NULL;
    } else {
        fail("Kimi hasn't returned error");
    }
}
END_TEST

START_TEST(remove_event_from_wrong_module)
{
    int i;
    GError* err = NULL;
    static const int size = 10000;
    char buf[size];
    
    for (i = 0; i < size; i++) {
        buf[i] = rand() % 256;
    }
    buf[size - 1] = '\0';

    int code = kimi_event_remove("Suite *s = suite_create",
            buf,
            kimi, 
            &err);

    fail_unless((code == -1) == (err != NULL));

    if (!code) {
        fail_unless(err->domain == KIMI_ERROR);
        fail_unless(err->code == KIMI_NO_SUCH_MODULE);
        g_error_free(err);
        err = NULL;
    }
}
END_TEST

START_TEST(remove_event)
{
    CHData* data = check_libkimi_get_data(kimi);

    kimi_event_remove("Test event", MOD_ID, kimi, NULL);
    
    fail_unless(data->deleted_event != NULL);
    fail_unless(strcmp(data->deleted_event, "Test event") == 0);
}
END_TEST

START_TEST(free_event_array)
{
    int i;
    GPtrArray* array = g_ptr_array_new();
    for (i = 0; i < 15; i++)
        g_ptr_array_add(array, kimi_event_new(NULL));
    
    kimi_free_array(array);
}
END_TEST

START_TEST(get_wrong_module)
{
    fail_unless(kimi_get_module_data("qwerty", kimi, NULL) == NULL);
}
END_TEST

START_TEST(period_test)
{
    time_t t = kimi_per_get_time_t("20040615T120000");
    fail_unless(t == 1087300800);
    
    
    Period* per = kimi_per_create_period();
    
    Rule* rul = kimi_per_create_rule_from_ical("FREQ=DAILY;COUNT=4;INTERVAL=2", 0, 1);
    kimi_per_add_rule(per, rul);
    
    GArray* times = kimi_per_generate_instance_times(per,
                                                     kimi_per_get_time_t("20040615T120000"),
                                                     kimi_per_get_time_t("20040617T120000"),
                                                     kimi_per_get_time_t("20040614T120000"));
    fail_unless(times->len == 1);
    fail_unless(g_array_index(times, time_t, 0) == kimi_per_get_time_t("20040616T000000"));
    
    g_array_free(times, TRUE);
    kimi_per_free_period(per);
}
END_TEST

static void check_period(const char* ical, const char* st, const char* end, const char* evt, const char* result, ...)
{
    Period* per = kimi_per_create_period();
    Rule* rul = kimi_per_create_rule_from_ical(ical, 0, 1);
    kimi_per_add_rule(per, rul);
    
    GArray* times = kimi_per_generate_instance_times(per,
                                                     kimi_per_get_time_t(st),
                                                     kimi_per_get_time_t(end),
                                                     kimi_per_get_time_t(evt));
    fail_unless(times != NULL);
    
    va_list va;
    va_start(va, result);
    int i = 0;
    while (result != 0) {
        fail_unless(i < times->len); /* Too many results */
        fail_unless(g_array_index(times, time_t, i) == kimi_per_get_time_t(result));
        result = va_arg(va, const char*);
        i++;
    }
    va_end(va);
    
    fail_unless(i == times->len);
    g_array_free(times, TRUE);
    kimi_per_free_period(per);
}

static void check_period_2(const char* ical, const char* ical2, const char* st, const char* end, const char* evt, const char* result, ...)
{
    Period* per = kimi_per_create_period();
    Rule* rul = kimi_per_create_rule_from_ical(ical, 0, 1);
    Rule* rul2 = kimi_per_create_rule_from_ical(ical2, 0, 1);
    
    kimi_per_add_rule(per, rul2);
    kimi_per_add_rule(per, rul);
    
    GArray* times = kimi_per_generate_instance_times(per,
                                                     kimi_per_get_time_t(st),
                                                     kimi_per_get_time_t(end),
                                                     kimi_per_get_time_t(evt));
    fail_unless(times != NULL);
    
    va_list va;
    va_start(va, result);
    int i = 0;

    while (result != 0) {
        fail_unless(i < times->len); /* Too many results */
        fail_unless(g_array_index(times, time_t, i) == kimi_per_get_time_t(result));
        result = va_arg(va, const char*);
        i++;
    }
    va_end(va);
    
    fail_unless(i == times->len);
    g_array_free(times, TRUE);
    kimi_per_free_period(per);
}

START_TEST(period_test_2)
{
    check_period("FREQ=DAILY;COUNT=4;INTERVAL=2", "20040615", "20040617", "20040614",
                 "20040616", 0);
                 
    check_period("FREQ=DAILY;COUNT=4;INTERVAL=1", "20040615", "20040617", "20040614",
                 "20040615", "20040616", "20040617", 0);
                 
    check_period("FREQ=WEEKLY;COUNT=4;INTERVAL=1;BYDAY=TH", "20100316", "20100326", "20100317",
                 "20100317", "20100318", "20100325", 0);
                 
    check_period("FREQ=WEEKLY;COUNT=4;INTERVAL=1;BYDAY=TH,WE", "20100316", "20110326", "20100318",
                 "20100318", "20100324", "20100325", "20100331",  0);
    
    check_period("FREQ=MONTHLY;INTERVAL=2;BYDAY=WE,FR", "20100317", "20100512", "20100316",
                 "20100317", "20100319", "20100324", "20100326", "20100331",
                 "20100505", "20100507", "20100512",  0);

    check_period("FREQ=MONTHLY;BYMONTHDAY=1,2,3", "20100101", "20100102", "20100101",
                 "20100101", "20100102", 0);

}
END_TEST

START_TEST(period_test_3)
{
    check_period_2("FREQ=WEEKLY;COUNT=2;BYDAY=WE", "FREQ=WEEKLY;COUNT=2;BYDAY=TH",
                   "20100406", "20100414", "20100405",
                   "20100407", "20100408", "20100414", 0);
                   
    check_period_2("FREQ=WEEKLY;COUNT=2;BYDAY=WE", "FREQ=WEEKLY;COUNT=2;BYDAY=WE",
                   "20100406", "20100416", "20100405",
                   "20100407", "20100414", 0);
}
END_TEST

START_TEST(settings_test)
{
    ConfigModule* cm = kimi_conf_create_config_module(kimi, MOD_ID, "Test module", NULL);
    
    kimi_conf_register_option(cm, "opt1", "Option 1", OT_STRING, 0, NULL);
    kimi_conf_register_option(cm, "opt2", "Option 2", OT_INTEGER, 0, NULL);
    kimi_conf_register_option(cm, "opt4", "Option 4", OT_BOOLEAN, 0, NULL);

    Option* v1 = kimi_conf_get_option(cm, "opt1", NULL);
    Option* v2 = kimi_conf_get_option(cm, "opt2", NULL);
    Option* v3 = kimi_conf_get_option(cm, "opt3", NULL);
    Option* v4 = kimi_conf_get_option(cm, "opt4", NULL);
    fail_unless(kimi_conf_option_get_state(v1) == OS_UNSET);
    fail_unless(kimi_conf_option_get_state(v2) == OS_UNSET);
    fail_unless(v3 == NULL);
    fail_unless(kimi_conf_option_get_state(v4) == OS_UNSET);
    
    kimi_conf_option_set_string(v1, "VALUE");
    kimi_conf_option_set_int(v2, 5);
    kimi_conf_option_set_boolean(v4, TRUE);

    v1 = kimi_conf_get_option(cm, "opt1", NULL);
    v2 = kimi_conf_get_option(cm, "opt2", NULL);
    v4 = kimi_conf_get_option(cm, "opt4", NULL);

    fail_unless(strcmp(kimi_conf_option_get_string(v1), "VALUE") == 0);
    fail_unless(kimi_conf_option_get_int(v2) == 5);    
    fail_unless(kimi_conf_option_get_boolean(v4) == TRUE);
}
END_TEST

START_TEST(settings_test_2)
{
    /* This test too verbose. Mute its */
    disable_log();
    const static int size = 10000;
    int i, j;
    ConfigModule* cm = kimi_conf_create_config_module(kimi, MOD_ID, "Test module", NULL);
    char buf[BUF_SIZE];
    
    for (i = 0; i < size; i++) {
        sprintf(buf, "Test%d", i);
        Option* opt = kimi_conf_register_option(cm, buf, buf, OT_INTEGER, 0, NULL);
        kimi_conf_option_set_int(opt, i);
    }
    
    for (i = 0; i < size; i++) {
        sprintf(buf, "Test%d", i);
        Option* v = kimi_conf_get_option(cm, buf, NULL);
        fail_unless(kimi_conf_option_get_int(v) == i);
    }
    enable_log();
}
END_TEST

START_TEST(settings_test_3)
{
    ConfigModule* cm = kimi_conf_create_config_module(kimi, MOD_ID, "Test module", NULL);

    Option* opt1 = kimi_conf_register_option(cm, "opt1", "Option 1", OT_INTEGER, OF_DISPLAYABLE, NULL);
    Option* opt2 = kimi_conf_register_option(cm, "opt2", "Option 2", OT_INTEGER, 0, NULL);
    Option* opt3 = kimi_conf_register_option(cm, "opt3", "Option 3", OT_INTEGER, OF_DISPLAYABLE, NULL);

    kimi_conf_option_set_int(opt1, 256);
    kimi_conf_option_set_int(opt2, 521);
    kimi_conf_option_set_int(opt3, 1023);

    kimi_conf_show_all_options_to_user(cm, "Message", NULL);

    CHData* data = check_libkimi_get_data(kimi);

    fail_unless(data->options->len == 2);
    fail_unless(strcmp(data->opt_title, "Test module") == 0);
    fail_unless(strcmp(data->opt_message, "Message") == 0);
    
    fail_unless(data->options->pdata[0] == opt1
            || data->options->pdata[0] == opt3);
    fail_unless(data->options->pdata[1] == opt3
            || data->options->pdata[1] == opt1);
}
END_TEST

START_TEST(arg_test)   
{
    kimi_event_free(NULL, NULL);
    kimi_event_dup(NULL, NULL);
    kimi_free_array(NULL);
    kimi_register_module(NULL, NULL, NULL);
    kimi_get_module_data(NULL, NULL, NULL);
    kimi_get_module(NULL, NULL);
    kimi_store(NULL, NULL, NULL);
    kimi_event_remove(NULL, NULL, NULL, NULL);
    kimi_get_by_id("service", "id", NULL, NULL);
    fail_unless(-1 == kimi_get_by_period(1232, 3, kimi, NULL));
    kimi_get_by_period(0, 3, NULL, NULL);
    fail_unless(NULL == kimi_filter(NULL, NULL));
    kimi_show_banner(NULL, NULL, NULL);
    kimi_filter(NULL, NULL);
    kimi_get_date_event_list_array(NULL);
    kimi_date_event_list_free(NULL, 0);
    fail_unless(NULL == kimi_get_modules(NULL));
}
END_TEST

/* Simple check: set value then get it */
START_TEST(dynamic_field_test_1)
{
    char str[] = "test_string";
    Event* ev = kimi_event_new(NULL);
    kimi_event_set_field(ev, "test_field", str);
    char* str2 = (char*) kimi_event_get_field(ev, "test_field");
    fail_unless(strcmp(str, str2) == 0);

    kimi_event_free(ev, NULL);
}
END_TEST

static void dynamic_field_test_free_function(void* d)
{
    char* str = (char*) d;
    CHData* data = check_libkimi_get_data(kimi);
    
    data->test_value = atoi(str);
}
/* Check is free_function works correctly */
START_TEST(dynamic_field_test_2)
{
    CHData* data = check_libkimi_get_data(kimi);
    
    Event* ev = kimi_event_new(NULL);
    
    char str[] = "82023136";
    char str2[] = "1441";
    kimi_event_set_field(ev, "test_field", str);
    kimi_event_set_field_free_function(ev, "test_field", dynamic_field_test_free_function);
    char* tmp = (char*) kimi_event_set_field(ev, "test_field", str2);
    fail_unless(tmp == str);
    fail_unless(data->test_value == atoi(str));

    kimi_event_free(ev, NULL);
    fail_unless(data->test_value == atoi(str2));
}
END_TEST

/* Check is free_function works correctly with wrong argiments */
START_TEST(dynamic_field_test_3)
{
    CHData* data = check_libkimi_get_data(kimi);

    Event* ev = kimi_event_new(NULL);
    char str[] = "3414";    
    kimi_event_set_field(ev, NULL, str);
    char* str2 = (char*) kimi_event_get_field(ev, "test_field");
    fail_unless(str2 == NULL);
    
    kimi_event_set_field(ev, "test_field", str);
    kimi_event_set_field_free_function(ev, NULL, dynamic_field_test_free_function);

    kimi_event_free(ev, NULL);
    fail_unless(data->test_value != atoi(str));
}
END_TEST

/* Check is kimi_event_set_field_free_function creates new dynamic field */
START_TEST(dynamic_field_test_4)
{
    CHData* data = check_libkimi_get_data(kimi);
    Event* ev = kimi_event_new(NULL);
    char str[] = "82023136";

    kimi_event_set_field_free_function(ev, "test_field", dynamic_field_test_free_function);
    fail_unless(NULL == kimi_event_get_field(ev, "test_field"));
    kimi_event_set_field(ev, "test_field", str);
    kimi_event_free(ev, NULL);

    fail_unless(data->test_value == atoi(str));
}
END_TEST

START_TEST(dynamic_field_test_5)
{
    Event* ev = kimi_event_new(NULL);
    fail_unless(NULL == kimi_event_get_field(ev, NULL));
    kimi_event_free(ev, NULL);
}
END_TEST

START_TEST(module_init_test)
{
    GPtrArray* a = g_ptr_array_new();
    KimiUICallbacks back;
    back.show_options = check_libkimi_settings_callback;
    kimi = kimi_init(a, back, NULL);
    
    Module* mod = calloc(1, sizeof(Module));
    int code = kimi_register_module(mod, kimi, NULL);
    fail_unless(code != 0);
    free(mod);
    
    g_ptr_array_free(a, TRUE);
    kimi_deinit(kimi);

}
END_TEST

START_TEST(module_init_test_2)
{
    GPtrArray* a = g_ptr_array_new();
    KimiUICallbacks back;
    back.show_options = check_libkimi_settings_callback;
    kimi = kimi_init(a, back, NULL);
    
    check_libkimi_initialize(kimi);
    check_libkimi_initialize(kimi);
    
    g_ptr_array_free(a, TRUE);
    kimi_deinit(kimi);
}
END_TEST

Suite* test_suite(void)
{
    Suite* s = suite_create("Test");

    /* Core test case */
    TCase* tc_core = tcase_create("Core");
    tcase_add_checked_fixture(tc_core, setup, teardown);
    tcase_add_test(tc_core, check_init);
    tcase_add_test(tc_core, create_event);
    tcase_add_test(tc_core, create_event_long);
    tcase_add_test(tc_core, store_event);
    tcase_add_test(tc_core, store_event_to_wrong_module);
    tcase_add_test(tc_core, store_event_with_error);
    tcase_add_test(tc_core, remove_event_from_wrong_module);
    tcase_add_test(tc_core, remove_event);
    tcase_add_test(tc_core, free_event_array);
    tcase_add_test(tc_core, get_wrong_module);
    
    /* Period test case */
    TCase* tc_period = tcase_create("Period");
    tcase_add_checked_fixture(tc_period, setup, teardown);
    tcase_add_test(tc_period, period_test);
    tcase_add_test(tc_period, period_test_2);
    tcase_add_test(tc_period, period_test_3);

    /* Config test case */    
    TCase* tc_config = tcase_create("Config");
    tcase_add_checked_fixture(tc_config, setup, teardown);
    tcase_add_test(tc_config, settings_test);
    tcase_add_test(tc_config, settings_test_2);
    tcase_add_test(tc_config, settings_test_3);

    /* Wrong argument tests */
    TCase* tc_arg = tcase_create("Arg");
    tcase_add_checked_fixture(tc_arg, setup, teardown);
    tcase_add_test(tc_arg, arg_test);

    /* Module initialize test */
    TCase* tc_mod_init = tcase_create("Module init");
    tcase_add_test(tc_mod_init, module_init_test);
    tcase_add_test(tc_mod_init, module_init_test_2);
    
    /* Dynamic property */
    TCase* tc_d = tcase_create("Event dynamic property");
    tcase_add_checked_fixture(tc_d, setup, teardown);
    tcase_add_test(tc_d, dynamic_field_test_1);
    tcase_add_test(tc_d, dynamic_field_test_2);
    tcase_add_test(tc_d, dynamic_field_test_3);
    tcase_add_test(tc_d, dynamic_field_test_4);
    tcase_add_test(tc_d, dynamic_field_test_5);

    suite_add_tcase(s, tc_core);
    suite_add_tcase(s, tc_period);
    suite_add_tcase(s, tc_config);
    suite_add_tcase(s, tc_arg);
    suite_add_tcase(s, tc_mod_init);
    suite_add_tcase(s, tc_d);
    
    return s;
}

int main()
{
    int number_failed;
    
    Suite* s = test_suite();
    
    SRunner *sr = srunner_create(s);
    
    srunner_run_all (sr, CK_NORMAL);
    number_failed = srunner_ntests_failed(sr);
    srunner_free(sr);
    return (number_failed == 0) ? EXIT_SUCCESS : EXIT_FAILURE;
}

