#define CTF_LOG_FORMAT_USE_TIME
#define CTF_THROW_EXIT

#include "../CTF/src/ctf.h"
#include "settings/settings.h"
#include "appdata.h"
#include "event_manager/event_manager.h"
#include "modules/test_module/test_module.h"
#include <string.h>
#include "event_manager/period.h"
#include <glib.h>

GLOBAL_DECL();
CTF_TEST_DATA_FILE_NAME_DECL;
CTF_STAT_FILE_NAME_DECL;

CTF_TEST_CASE(initialize_module_test) {
    CTF_INIT_CASE("initialize_module_test");
    
    test_module_data* d = ev_mgr_get_module_data("tmod", data, NULL);
    
    CTF_CHECK(d);
    CTF_CHECK(d->a == 5 && d->b == 445);

    GError* err = NULL;
    d = ev_mgr_get_module_data("tmоd", data, &err); /* 'о' - in russian keyboard layout */
    
    CTF_CHECK(!d);
    CTF_CHECK(err);
    
    g_error_free(err);

    CTF_DEINIT_CASE();
}

CTF_TEST_CASE(get_event_module_test) {
    CTF_INIT_CASE("get_event_module_test");
    GError* err = NULL;

    test_module_data* d = ev_mgr_get_module_data("tmod", data, NULL);
    event* ev = NULL;

    CTF_CHECK(!ev_mgr_get_by_id("id", "tmd", data, NULL));

    d->imitate_error = TRUE; 
    CTF_CHECK(!ev_mgr_get_by_id("id", "tmod", data, NULL));
    d->imitate_error = FALSE;
    
    CTF_DEINIT_CASE();
}

CTF_TEST_CASE(store_end_get_test) {
    CTF_INIT_CASE("store_end_get_test");
    event* ev = ev_mgr_new(NULL);

    ev->title = strdup("Test event");
    ev->location = strdup("Test location");
    ev->service_event_id = strdup("tmod");

    ev_mgr_store(ev, data, NULL);
    char* id = strdup(ev->id);
    ev_mgr_free(ev, NULL);
    
    ev = ev_mgr_get_by_id(id, "tmd", data, NULL);
    CTF_CHECK(!ev);
   
    ev = ev_mgr_get_by_id(id, "tmod", data, NULL);
    
    CTF_REQUIRE(ev);
    CTF_CHECK(!strcmp(ev->title, "Test event") && !strcmp(ev->location, "Test location"));
    
    free(id);
    CTF_DEINIT_CASE();
}

CTF_TEST_CASE(remove_event_test) {
    CTF_INIT_CASE("remove_event_test");

    event* ev = ev_mgr_new(NULL);

    ev->title = strdup("Test event");
    ev->location = strdup("Test location");
    ev->service_event_id = strdup("tmod"); 

    ev_mgr_store(ev, data, NULL);
    char* id = strdup(ev->id);
    ev_mgr_free(ev, NULL);

    ev_mgr_remove(id, "tmd", data, NULL);
    CTF_CHECK(ev = ev_mgr_get_by_id(id, "tmod", data, NULL));
    ev_mgr_free(ev, NULL);

    ev_mgr_remove(id, "tmod", data, NULL);
    CTF_CHECK(!(ev = ev_mgr_get_by_id(id, "tmod", data, NULL)));
    ev_mgr_free(ev, NULL);
    
    CTF_DEINIT_CASE();
}

CTF_TEST_CASE(date_event_list_test) {
    CTF_INIT_CASE("date_event_list_test");

    ev_mgr_date_event_list_free(data, TRUE);
    CTF_CHECK(data->date_event_list);
    ev_mgr_date_event_list_free(data, FALSE);
    CTF_CHECK(!data->date_event_list);
    
    ev_mgr_date_event_list_free(data, TRUE);
    int i;
    for (i = 0; i < 1; i++) {
        date_event_list dil;
        dil.events = g_ptr_array_new();
    
        event* ev = ev_mgr_new(NULL);
        g_ptr_array_add(dil.events, ev);
        g_array_append_val(data->date_event_list, dil);
        ev_mgr_date_event_list_free(data, TRUE);
    }
    ev_mgr_date_event_list_free(data, FALSE);
    
    CTF_DEINIT_CASE();
}

CTF_TEST_CASE(period_test) {
    CTF_INIT_CASE("period_test");
    
    rule* rul = ev_per_create_rule(0, REC_DAY, 1);
    
    rul->count = 10;
    rul->interval = 2;
    
    GArray* arr = g_array_new(FALSE, FALSE, sizeof(short));
    
    short i = 2;
    short j = 4;
    
    g_array_append_val(arr, i);
    g_array_append_val(arr, j);
    
    rul->week_days = arr;
    
    const char* res = ev_per_get_ical(rul);
    //g_debug("%s\n", res);
    CTF_CHECK(!strcmp("FREQ=DAILY;COUNT=10;INTERVAL=2;BYDAY=TU,TH", res));
    
    ev_per_set_ical(rul, res);
    
    CTF_CHECK(rul->count == 10
            && rul->freq == REC_DAY
            && rul->interval == 2
            && g_array_index(rul->week_days, short, 0) == 2
            && g_array_index(rul->week_days, short, 1) == 4);
    
    CTF_CHECK(rul->week_days != NULL
            && rul->hours == NULL
            && rul->set_pos == NULL);
    
    ev_per_free_rule(rul);
    
    CTF_CHECK(rul->week_days == NULL
            && rul->hours == NULL
            && rul->set_pos == NULL);
               
    CTF_DEINIT_CASE();
}

CTF_TEST_CASE(period_test_2) {
    CTF_INIT_CASE("period_test_2");
    int i = 0;
    
    time_t expected_results[] = {
        1264982400,
        1265587200,
        1266192000
    };
    
    time_t event_time = 1264982400;
    time_t start = 1264982400; /* FEB 1 0:00:00 UTC 2010 */
    time_t end = 1266192000; /* FEB 15 0:00:00 UTC 2010 */
    
    period* per = ev_per_create_period();
    rule* rul = ev_per_create_rule(0, REC_WEEK, 1);

    rul->interval = 1;
    ev_per_update_rule(rul);
    
    g_ptr_array_add(per, rul);
    
    GArray* res = ev_per_generate_instance_times(per, start, end, event_time, 3600, 0);
    
    int ok = 1;
    for (i = 0; i < res->len; i++) {
        if (expected_results[i] != g_array_index(res, time_t, i))
            ok = 0;
    }
    CTF_CHECK(ok);
    
    g_array_free(res, TRUE);
    ev_per_free_period(per);
    
    CTF_DEINIT_CASE();
}

CTF_TEST_CASE(period_test_3) {
    CTF_INIT_CASE("period_test_3");

    int i = 0;
    
    time_t expected_results[] = {
        1241582644, /* 6.05.2009 4:4:4 */ 
        1241755444, /* 8 */
        1241928244, /* 10 */
        1242533044, /* 17 */
        1242705844, /* 19 */
        1242965044, /* 22 */ 
        1243137844 /* 24 */
    };
    
    time_t event_time = 1241582644; /* MAY 6 4:04:04 UTC 2009 */
    time_t start = 1241496244; /* MAY 5 4:04:04 UTC 2009 */
    time_t end = 1243397044; /* MAY 27 4:04:04 UTC 2009 */
    
    period* per = ev_per_create_period();
    
    /* Rule: tuesday and friday, every 2 weeks */
    rule* rul = ev_per_create_rule(0, REC_WEEK, 1);
    rul->freq = REC_WEEK;
    rul->interval = 2;
    
    GArray* arr = g_array_new(FALSE, FALSE, sizeof(short));
    short sarr[] = {2, 5};
    g_array_append_vals(arr, sarr, 2);
    rul->week_days = arr;
    
    
    /* Rule: sunday, every weeks */
    rule* rul2 = ev_per_create_rule(0, REC_WEEK, 1);
    rul2->freq = REC_WEEK;
    rul2->interval = 1;
    
    GArray* arr2 = g_array_new(FALSE, FALSE, sizeof(short));
    short sarr2[] = {0};
    g_array_append_vals(arr2, sarr2, 1);
    rul2->week_days = arr2;
    
    ev_per_update_rule(rul);
    ev_per_update_rule(rul2);
    
    g_ptr_array_add(per, rul);
    g_ptr_array_add(per, rul2);
    
    GArray* res = ev_per_generate_instance_times(per, start, end, event_time, 3600, 0);
    
    int ok = 1;
    for (i = 0; i < res->len; i++) {
        if (expected_results[i] != g_array_index(res, time_t, i))
            ok = 0;
    }
    CTF_CHECK(ok);
    
    g_array_free(res, TRUE);
    ev_per_free_period(per);
    
    CTF_DEINIT_CASE();
}

CTF_TEST_CASE(period_test_4_weekstart) {
    CTF_INIT_CASE("period_test_4_weekstart");
    int i;
    
    time_t expected_results[] = {
        1241496244, /* 5.05.2009 4:4:4 */ 
        1242533044, /* 17 */
        1242705844, /* 19 */
        1243742644 /* 31 */
    };
    time_t expected_results2[] = {
        1241496244, /* 5.05.2009 4:4:4 */ 
        1241928244, /* 10 */
        1242705844, /* 19 */
        1243137844 /* 24 */
    };
    
    time_t event_time = 1241496244; /* MAY 5 4:04:04 UTC 2009 */
    time_t start = 1241496244; /* MAY 3 4:04:04 UTC 2009 */
    time_t end = 1243742644; /* MAY 31 4:04:04 UTC 2009 */
    
    period* per = ev_per_create_period();
    
    /* Rule: tuesday and friday, every 2 weeks */
    rule* rul = ev_per_create_rule(0, REC_WEEK, 0);
    rul->interval = 2;
    rul->count = 4;
    
    GArray* arr = g_array_new(FALSE, FALSE, sizeof(short));
    short sarr[] = {0, 2}; /* Repeat by sundays */
    g_array_append_vals(arr, sarr, 2);
    rul->week_days = arr;
    
    g_ptr_array_add(per, rul);
    
    /* Week starts sunday check */
    ev_per_update_rule(rul);
    GArray* res = ev_per_generate_instance_times(per, start, end, event_time, 3600, 0);
    
    int ok = 1;
    for (i = 0; i < res->len; i++) {
        if (expected_results[i] != g_array_index(res, time_t, i))
            ok = 0;
    }
    CTF_CHECK(ok);
    g_array_free(res, TRUE);
    CTF_DEINIT_CASE();
}



CTF_TEST_CASE(get_by_period_test) {
    CTF_INIT_CASE("get_by_period_test");
   
    test_module_data* d = ev_mgr_get_module_data("tmod", data, NULL);

    ev_mgr_get_by_period(1265709600, 1265709600 + 12 * 24 * 60 * 60, data, NULL);
    
    CTF_CHECK(g_array_index(data->date_event_list, date_event_list, 0).date == 1265673600);
    CTF_CHECK(g_array_index(data->date_event_list, date_event_list, 1).date == 1266278400);

    time_t expected_results[] = {
        1265673600, /* 9.02.2010 10:00:00 */
        1265846400, /* 11 */
        1266105600, /* 14 */
    };

    d->st = 1;
    ev_mgr_get_by_period(1265709600, 1265709600 + 9 * 24 * 60 * 60, data, NULL);
    int i, ok = 1;
    for (i = 0; i < data->date_event_list->len; i++) {
        if (expected_results[i] != g_array_index(data->date_event_list, date_event_list, i).date)
            ok = 0;
    }
    CTF_CHECK(ok);
    CTF_CHECK(data->date_event_list->len == 3);
    

    CTF_DEINIT_CASE();
}









