#include "kimi.h"
#include "period.h"
#include "settings.h"

#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <time.h>
#include <dlfcn.h>

/**
 * Function creates a copy of GDate date
 */
#define GDATE_DUP(date) g_date_new_dmy(g_date_get_day(date), g_date_get_month(date), g_date_get_year(date))
#define NOT_NULL "Function requires not null arguments"

typedef void (*InitializeFunction)(Kimi*, Module*);

Event* kimi_event_new(GError** error)
{
    Event* ev = (Event*) calloc(1, sizeof(Event));
    CHECK_ALLOC(ev, error);
    return ev;
}

int kimi_event_free(Event* ev, GError** error)
{
    /* Clear fields, if they exists */
    if (!ev)
        return 0;
    if (ev->id) free(ev->id);
    if (ev->title) free(ev->title);
    if (ev->description) free(ev->description);
    if (ev->location) free(ev->location);
    if (ev->service_event_id) free(ev->service_event_id);
    if (ev->per) {
        kimi_per_free_period(ev->per);
    }
    if (ev->dynamic_fields) {
        GHashTableIter iter;
        g_hash_table_iter_init(&iter, ev->dynamic_fields);
        gpointer key, value;

        while (g_hash_table_iter_next (&iter, &key, &value)) {
            DynamicField* dv = (DynamicField*) value;
            free(key);
            if (dv->free_function) {
                dv->free_function(dv->data);
            }
        }
    }
    free(ev);
    return 0;
}

void kimi_free_array(GPtrArray* arr)
{
    if (!arr)
        return;
    int i;
    for (i = 0; i < arr->len; i++) {
        kimi_event_free(g_ptr_array_index(arr, i), NULL);
    }
    g_ptr_array_free(arr, TRUE);
}

Event* kimi_event_dup(Event* ev, GError** error)
{
    if (!ev)
        return NULL;
    Event* newev = kimi_event_new(NULL);
    CHECK_ALLOC(newev, error);
    memcpy(newev, ev, sizeof(Event));
    
    if (newev->title)
        newev->title = strdup(ev->title);
    if (newev->description)
        newev->description = strdup(ev->description);
    if (newev->location)
        newev->location = strdup(ev->location);
    if (newev->service_event_id) 
        newev->service_event_id = strdup(ev->service_event_id);
    if (newev->id) 
        newev->id = strdup(ev->id);

    if (newev->per) {
        newev->per = kimi_per_dup(ev->per); /* TODO: This should be copy of period */ 
    }
    return newev;
}

int kimi_event_fill(Event* ev, 
        const char* service_id, 
        const char* id, 
        const char* title,
        const char* desc,
        const char* location,
        time_t start,
        time_t end,
        Period* period,
        GError** error) {

    if (service_id)
        ev->service_event_id = strdup(service_id);
    if (id)
        ev->id = strdup(id);
    if (title)
        ev->title = strdup(title);
    if (desc)
        ev->description = strdup(desc);
    if (location)
        ev->location = strdup(location);
    ev->start_time = start;
    ev->end_time = end;

    ev->per = period;
    return 0;
}

void kimi_event_set_field_free_function(Event* ev, const char* name, void (*free_function)(void*))
{
    if (name == NULL) {
        g_critical(NOT_NULL);
        return;
    }
    if (!ev->dynamic_fields)
        ev->dynamic_fields = g_hash_table_new(g_str_hash, g_str_equal);
    
    DynamicField* field = g_hash_table_lookup(ev->dynamic_fields, name);
    /* If field doesn't exist create it */
    if (!field) {
        field = calloc(1, sizeof(DynamicField));
        g_hash_table_insert(ev->dynamic_fields, strdup(name), field);
    }
    field->free_function = free_function;
}

void* kimi_event_set_field(Event* ev, const char* name, void* data)
{
    if (name == NULL) {
        g_critical(NOT_NULL);
        return NULL;
    }

    if (!ev->dynamic_fields)
        ev->dynamic_fields = g_hash_table_new(g_str_hash, g_str_equal);
    
    /* Look is field already exists */
    DynamicField* field = g_hash_table_lookup(ev->dynamic_fields, name);
    void* last_value = field ? field->data : NULL;
    
    if (!field) {
        field = calloc(1, sizeof(DynamicField));
        g_hash_table_insert(ev->dynamic_fields, strdup(name), field);
    } else if (field->free_function && field->data) {
        field->free_function(field->data);
    }
    field->data = data;
    return last_value;
}

void* kimi_event_get_field(Event* ev, const char* name)
{
    if (name == NULL) {
        g_critical(NOT_NULL);
        return NULL;
    }

    if (ev->dynamic_fields) {
        DynamicField* field = g_hash_table_lookup(ev->dynamic_fields, name);
        return field->data;
    } else {
        return NULL;
    }

}

static Module* load_module(const char* file, Kimi* data)
{
    Module* mod = NULL;
    GError* err = NULL;
    /* Pointer to initialization function */
    InitializeFunction init;
    /* Open library */
    void* dl = dlopen(file, RTLD_LAZY);
    /* Check, is library opened correctly */
    if (dl) {
        /* Load symbol of initialization function */
        init = (InitializeFunction) dlsym(dl, MODULE_INITIALIZE_FUNCTION);
        /* If symbol exists, register new module */
        if (init) {
            mod = calloc(1, sizeof(Module));
            
            /* Call initialize function */
            data->current_module = mod;
            init(data, mod);
            data->current_module = NULL;
            
            kimi_register_module(mod, data, &err);
            if (!err) {
                g_message("Module %s successfully initialized", file);
            } else {
                g_warning("Error loading module %s", file);
                g_error_free(err);
                err = NULL;
            }
        } else {
            g_warning("Module %s doesn't contain symbol '%s'", file, MODULE_INITIALIZE_FUNCTION);
        }
    } else {
        g_message("File %s isn't .so library", file);
    }
    return mod;
}

Kimi* kimi_init(GPtrArray* dirs_, KimiUICallbacks callbacks, GError** error)
{
    int i = 0;
    GPtrArray* dirs = dirs_;
    GString* fullname = g_string_sized_new(512);
    CHECK_ALLOC(fullname, error);

    g_message("Inititalize Event Manager\n");

    Kimi* data = (Kimi*) calloc(1, sizeof(Kimi));
    CHECK_ALLOC(data, error, g_string_free(fullname, TRUE)); 
    
    data->ui_callbacks = callbacks;
    kimi_conf_initialize(data, g_getenv("HOME"));
    
    data->modules = g_hash_table_new_full(g_str_hash, g_str_equal, NULL, NULL);
    CHECK_ALLOC(data->modules, error,
            g_string_free(fullname, TRUE));

    /* If array doesn't exist, create it */
    if (!dirs) {
        dirs = g_ptr_array_new();
        CHECK_ALLOC(dirs, error,
                g_string_free(fullname, TRUE),
                g_hash_table_destroy(data->modules));

        /* Add default path */
        g_ptr_array_add(dirs, strdup(MODULESDIR));
    }
    /* For each directory get list of files */ 
    for (i = 0; i < dirs->len; i++) {
        const char* dir = g_ptr_array_index(dirs, i);
        GDir* dir_handle = g_dir_open(dir, 0, NULL); 
        
        /* If directory doesn't exist - skip it */
        if (!dir_handle) {
            continue;
        }

        const char* file;
        //file = g_dir_read_name(dir_handle);
        /* For each file, get it fullname, and try load as library */
        while ((file = g_dir_read_name(dir_handle))) {
            g_string_printf(fullname, "%s/%s", dir, file);
            const char* fullname_c = fullname->str;
            load_module(fullname_c, data);
        }
        
        g_dir_close(dir_handle);
    }
    g_string_free(fullname, TRUE); 
    
    /* Free dirs array if it allocated in this function */
    if (!dirs_) {
        for (i = 0; i < dirs->len; i++)
            free(g_ptr_array_index(dirs, i));
        g_ptr_array_free(dirs, TRUE);
    }
    return data;
}

void kimi_deinit(Kimi* data)
{
    g_message("Deinititalize Event Manager\n");
    /* Remove all modules */

    g_hash_table_foreach(data->modules, kimi_deinitialize_module, data);

    kimi_date_event_list_free(data, FALSE);
    
    g_hash_table_destroy(data->modules);
    
    kimi_conf_deinitialize(data);
    free(data);
}

int kimi_register_module(Module* mod, Kimi* data, GError** error)
{
    if (!data || !mod) {
        g_critical(NOT_NULL);
        return -1;
    }

    GHashTable* modules = data->modules;    

    if (!mod->service_string) {
        g_warning("Module has returned NULL service string");
        g_set_error(error, KIMI_ERROR, KIMI_MODULE_LOADING_ERROR, MODULE_LOADING_ERROR_STRING, "null");
        return -1;
    }

    if (kimi_get_module(mod->service_string, data)) {
        g_warning("Module %s has already been registered", mod->service_string);
        g_set_error(error, KIMI_ERROR, KIMI_MODULE_LOADING_ERROR, MODULE_LOADING_ERROR_STRING, mod->service_string);
        return -1;
    }

    char* key = strdup(mod->service_string);
    
    if (!key) { 
        g_set_error(error, KIMI_ERROR, KIMI_ALLOCATE_ERROR,
                ALLOCATE_ERROR_STRING, strlen(mod->service_string));
        return -1;
    }

    g_hash_table_insert(modules, key, mod);
    return 0;
}

void kimi_deinitialize_module(gpointer key_, gpointer val_, gpointer data_)
{
    char* key = (char*) key_;
    Module* mod = (Module*) val_;
    Kimi* data = (Kimi*) data_;
    
    data->current_module = mod;
    if (mod->deinitialize)
        mod->deinitialize(data, mod);
    else 
        g_warning("Module %s doesn't provides function \"deinitialize\"", mod->service_string);
    data->current_module = NULL;
    
    free(key);
    free(mod);
}

void* kimi_get_module_data(const char* service_string, Kimi* data, GError** error)
{
    if (!data) {
        g_critical(NOT_NULL);
        return NULL;
    }

    Module* mod = kimi_get_module(service_string, data);
    
    if (!mod) {
        g_warning("Someone trying get module data with id '%s'. No such module.", service_string); 
        g_set_error(error, KIMI_ERROR, KIMI_NO_SUCH_MODULE, NO_SUCH_MODULE_STRING, service_string);
        return NULL;
    }

    return mod->data;
}

Module* kimi_get_module(const char* service_string, Kimi* data)
{
    if (!data) {
        g_critical(NOT_NULL);
        return NULL;
    }   

    if (service_string)
        return (Module*) g_hash_table_lookup(data->modules, service_string);
    else
        return data->current_module;
}

GPtrArray* kimi_get_modules(Kimi* data)
{
    if (!data) {
        g_critical(NOT_NULL);
        return NULL;        
    }                           

    GPtrArray* arr = g_ptr_array_new();
    gpointer key, value;
    GHashTableIter iter;
    g_hash_table_iter_init(&iter, data->modules);
    
    while (g_hash_table_iter_next(&iter, &key, &value)) {
        g_ptr_array_add(arr, value);
    }
    return arr;
}

int kimi_store(Event* ev, Kimi* data, GError **error)
{
    if (!data || !ev) {
        g_critical(NOT_NULL);
        return -1;        
    }                           
    char* old_id = ev->id;

    GError* err = NULL;
    g_assert(ev->service_event_id);
    const char* service = ev->service_event_id;

    Module* mod = kimi_get_module(service, data);
    if (!mod) {
        g_set_error(error, KIMI_ERROR, KIMI_NO_SUCH_MODULE, NO_SUCH_MODULE_STRING, service);
        return -1;
    }
    
    data->current_module = mod;
    int code = mod->store_event(ev, data, &err);
    data->current_module = NULL;
    
    if (code && err) {
        g_set_error(error, KIMI_ERROR, KIMI_STORE_EVENT_FAILED, STORE_EVENT_FAILED_STRING, (ev->id ? ev->id : "new event"), ev->service_event_id, err->message);
        g_clear_error(&err);
        return -1;
    }

    /* Add event to current view */
    if (old_id)
        kimi_remove_event_from_date_event_list(data, old_id, ev->service_event_id);

    kimi_add_event_to_date_event_list(data, ev);

    return 0;
}

int kimi_event_remove(const char* id, const char* service, Kimi* data, GError** error)
{
    if (!data || !id || !service) {
        g_critical(NOT_NULL);
        return -1;
    }

    GError* err = NULL;
    
    Module* mod = kimi_get_module(service, data);

    if (!mod) {
        g_set_error(error, KIMI_ERROR, KIMI_NO_SUCH_MODULE, NO_SUCH_MODULE_STRING, service);
        return -1;
    }
    
    data->current_module = mod;
    int code = mod->remove_event(id, data, &err);
    data->current_module = NULL;
    
    if (code) {
        g_set_error(error, KIMI_ERROR, KIMI_REMOVE_EVENT_FAILED, REMOVE_EVENT_FAILED_STRING, id, service, err->message);
        g_error_free(err);
        err = NULL;

        return -1;
    }
    kimi_remove_event_from_date_event_list(data, id, service);
    return 0;
}

Event* kimi_get_by_id(const char* id, const char* service, Kimi* data, GError **error)
{
    #if 0
    if (!data || !id || !service) {
        g_critical(NOT_NULL);
        return NULL;
    }

    GError* err = NULL;
    Module* mod = kimi_get_module(service, data);
    
    if (!mod) {
        g_set_error(error, KIMI_ERROR, KIMI_NO_SUCH_MODULE, NO_SUCH_MODULE_STRING, service);
        return NULL;
    }
    
    data->current_module = mod;
    Event* ev = mod->get_event_by_id(id, data, &err);
    data->current_module = NULL;
    
    /* If no event returns and error */
    if (!ev && err) {
        g_set_error(error, KIMI_ERROR, KIMI_CANNOT_GET_EVENT, CANNOT_GET_EVENT_STRING, id, service, err->message);
        
        g_error_free(err);
        err = NULL;
        
        return NULL;
    }
        return ev;
#endif

return NULL;
}

/**
 * @brief Hash function for GDate
 *
 * @param p GDate pointer
 * @return data hash
 */
static guint date_hash_func(gconstpointer p)
{
    GDate* date = (GDate*) p;
    /* hash = year * 366 + day */
    return g_date_get_day_of_year(date) + g_date_get_year(date) * 366;
}

/**
 * @brief Compare two dates
 *
 * @param a first date
 * @param b second date
 * @return result
 * @retval TRUE if dates equals
 * @retval FALSE otherwise
 */
static gboolean date_equal_func(gconstpointer a, gconstpointer b)
{
    if (g_date_compare((GDate*) a, (GDate*) b) == 0) {
        return TRUE;
    } else {
        return FALSE;
    }
}

/**
 * @brief Add element into hashtable
 *
 * If date in hashtable then elem will be added to
 * end of value array,
 * else new record will be created
 *
 * @param hash GHashtable
 * @param date data
 * @param elem value
 */
static void add_event_to_hash(GHashTable* hash, const GDate* date, Event* elem)
{
    gpointer val = g_hash_table_lookup(hash, date);

    /* Key already in hashtable */
    if (val) {
        GPtrArray *arr = (GPtrArray*) val;
        g_ptr_array_add(arr, elem);
    } else {
        GPtrArray *arr = g_ptr_array_new();
        g_ptr_array_add(arr, elem);
        GDate *date_to_store =  GDATE_DUP(date);
        g_hash_table_insert(hash, date_to_store, arr);
    }
}

static time_t my_timegm(struct tm *tm)
{   
    time_t ret;
    char *tz;

    tz = getenv("TZ");
    setenv("TZ", "", 1);
    tzset();
    ret = mktime(tm);
    if (tz)
        setenv("TZ", tz, 1);
    else
        unsetenv("TZ");
    tzset();
    return ret;
}

/**
 * @brief Convert GDate to time_t
 * 
 * @param d GDate
 * @return time_t
 */
static time_t gdate2time_t(GDate* d)
{
    struct tm t;
    g_date_to_struct_tm(d, &t);
    
    t.tm_isdst = -1;
    return my_timegm(&t);
}

int kimi_get_by_period(time_t start, time_t end, Kimi* data, GError** error)
{
    if (end < start) {
        g_warning("kimi_get_by_period: end time must be more than start time");
        return -1;
    } 
    if (!data) {
        g_critical(NOT_NULL);
        return -1;
    }

    int i, j;
    GHashTable* modules = data->modules;
    GHashTableIter iter;
    g_hash_table_iter_init(&iter, modules);
    
    const char* id;
    Module* mod;
    
    GHashTable* dates = g_hash_table_new(date_hash_func, date_equal_func);
    
    /* Iterate over modules */
    while (g_hash_table_iter_next(&iter, (gpointer)&id, (gpointer)&mod)) {
        /* This is the list of events, at least one instance of which belongs interval [start, end] */
       data->current_module = mod;
       GPtrArray* events = mod->get_events_by_period(start, end, data, NULL);
       data->current_module = NULL;
       
/*       printf("Module: %s\n", mod->service_string);
       if (events)
           printf("Events: %d\n", events->len);
       else
           printf("Events: null\n");
*/
       if (!events)
           continue;
       
       /* Iterate over events and add them to hash table */
       for (i = 0; i < events->len; i++) {
            Event* ev = g_ptr_array_index(events, i);
            ev->service_event_id = strdup(mod->service_string);
            GArray* times = kimi_per_generate_instance_times(ev->per,
                                                           start,
                                                           end, 
                                                           ev->start_time);
            for (j = 0; j < times->len; j++) {
                GDate date;
                time_t t = g_array_index(times, time_t, j);
                g_date_clear(&date, 1);
                g_date_set_time_t(&date, t);
                
                add_event_to_hash(dates, &date, ev);
            }
            g_array_free(times, TRUE);                 
       }

       g_ptr_array_free(events, TRUE);
    }
    
    g_hash_table_iter_init(&iter, dates);
    
    kimi_date_event_list_free(data, TRUE);
    data->date_id_list_start = start;
    data->date_id_list_end = end; 
    /* Convert hash table to date_event_list* */
    GDate* date;
    GPtrArray* evs;
    while (g_hash_table_iter_next(&iter, (gpointer)&date, (gpointer)&evs)) {
        time_t t = gdate2time_t(date);
        DateEventList dil;
        dil.date = t;
        dil.events = evs;
        g_array_append_val(data->date_event_list, dil);
        g_date_free(date);
    }
    g_hash_table_destroy(dates);
    return 0;
}

static int kimi_compare_fields(const char* field, const char* mask)
{
    int ret = 0;
    if (field && mask) {
        if (strstr(field, mask)) {
            ret = 1;
        }
    } else if (mask == NULL) { // If no mask, then return true
        ret = 1;
    }
    
    return ret;
}

GPtrArray* kimi_filter(Event* mask, Kimi* data)
{
    if (!data) {
        g_critical(NOT_NULL);
        return NULL;
    }
    int i;
    GPtrArray* view = kimi_get_date_event_list_array(data);
    GPtrArray* res = g_ptr_array_new();
    
    for (i = 0; i < view->len; i++) {
        Event* ev = g_ptr_array_index(view, i);
        if (kimi_compare_fields(ev->title, mask->title)
         && kimi_compare_fields(ev->description, mask->description)
         && kimi_compare_fields(ev->location, mask->location)) {
                g_ptr_array_add(res, ev);
         }
    }
    
    g_ptr_array_free(view, TRUE);
    return res;
}

GPtrArray* kimi_get_date_event_list_array(Kimi* data)
{
    if (!data) {
        g_critical(NOT_NULL);
        return NULL;
    }

    int i, j;
    GArray* dl = data->date_event_list;
    GPtrArray* res = NULL;
    if (dl) {

        res = g_ptr_array_new();
        GHashTable* res_hash = g_hash_table_new_full(g_direct_hash, g_direct_equal, NULL, NULL);
        for (i = 0; i < dl->len; i++) {
            GPtrArray* events = g_array_index(dl, DateEventList, i).events;
            for (j = 0; j < events->len; j++) {
                Event* ev = (Event*) g_ptr_array_index(events, j);
                if (!g_hash_table_lookup(res_hash, ev)) {
                    g_hash_table_insert(res_hash, ev, NULL);
                    g_ptr_array_add(res, ev);
                }
            }
        }
        g_hash_table_destroy(res_hash);
    }
    return res;
}

void kimi_date_event_list_free(Kimi* data, int is_create_new)
{
    if (!data) {
        g_critical(NOT_NULL);
        return;
    }

    int i, j;
    GArray* dl = data->date_event_list;
    
    if (dl) {
        GHashTable* freed = g_hash_table_new_full(g_direct_hash, g_direct_equal, NULL, NULL);
        for (i = 0; i < dl->len; i++) {
            GPtrArray* events = g_array_index(dl, DateEventList, i).events;
            for (j = 0; j < events->len; j++)
            {
                Event* ev = (Event*)g_ptr_array_index(events, j);
                /* If key hasn't been freed */
                if (!g_hash_table_lookup(freed, ev))
                {
                    /* Add pointer to hash table, it will be free later */
                    g_hash_table_insert(freed, ev, ev);
                }
            }
            
            g_ptr_array_free(events, TRUE);
        }
        if (is_create_new) 
            g_array_set_size(dl, 0);
        else {
            g_array_free(dl, TRUE);
            dl = NULL;
        }
        
        /* Iterate over hash of unique events and free them */
        GHashTableIter iter;
        g_hash_table_iter_init(&iter, freed);
        
        gpointer iter_ev;
        gpointer iter_val;

        while (g_hash_table_iter_next(&iter, &iter_ev, &iter_val)) {
            kimi_event_free((Event*)iter_ev, NULL);
        }

        /* Destroy table */
        g_hash_table_destroy(freed);
    }
    else if (is_create_new) {
        dl = g_array_new(FALSE, FALSE, sizeof(DateEventList));
    }
    
    data->date_event_list = dl;
}

static void kimi_add_occurence_to_date_event_list(Kimi* data, Event* ev, time_t date)
{
    int i;
    GDate dates[2];
    GDate* tmp_day = &dates[0];
    GDate* date_day = &dates[1];

    g_date_clear(date_day, 1);
    g_date_set_time_t(date_day, date);

    GArray* dl = data->date_event_list;
    GPtrArray* arr = NULL;
    for (i = 0; i < dl->len; i++) {
        DateEventList events = g_array_index(dl, DateEventList, i);
        g_date_clear(tmp_day, 1);
        g_date_set_time_t(tmp_day, events.date);

        if (g_date_compare(date_day, tmp_day) == 0) {
            arr = events.events;
            break;
        }
    }

    if (arr == 0) {
        arr = g_ptr_array_new();
        DateEventList new_dl;
        new_dl.date = date;
        new_dl.events = arr;
        g_array_append_val(dl, new_dl);
    }

    g_ptr_array_add(arr, ev);
}

void kimi_add_event_to_date_event_list(Kimi* data, Event* ev)
{
    int i;
    GArray* dl = data->date_event_list;
    if (!dl || !ev) 
        return;
    
    Event* new_ev = kimi_event_dup(ev, NULL);
    GDate ev_day;
    GArray* inst = kimi_per_generate_instance_times(ev->per, data->date_id_list_start, data->date_id_list_end, ev->start_time);
    
    for (i = 0; i < inst->len; i++) {
        time_t t = g_array_index(inst, time_t, i);
        /*g_date_clear(&ev_day, 1);
        g_date_set_time_t(&ev_day, t);
        */
        kimi_add_occurence_to_date_event_list(data, new_ev, t);
    }
}

void kimi_remove_event_from_date_event_list(Kimi* data, const char* id, const char* service)
{
    int i, j;
    GArray* dl = data->date_event_list;
    if (!dl || !id || !service)
        return;

    for (i = 0; i < dl->len; i++) {
        DateEventList events = g_array_index(dl, DateEventList, i);
        for (j = 0; j < events.events->len; j++) {
            Event* ev = g_ptr_array_index(events.events, j);
            if (strcmp(ev->id, id) == 0
                    && strcmp(ev->service_event_id, service) == 0) {
                g_ptr_array_remove_index_fast(events.events, j);
                break;
            }
        }
    }

    for (i = 0; i < dl->len; i++) {
        DateEventList del = g_array_index(dl, DateEventList, i);
        if (del.events == NULL || del.events->len == 0) {
            g_array_remove_index(dl, i);
            i--;
        }
    }
}

void kimi_show_banner(Kimi* kimi, const char* title, const char* text_message)
{
    if (!kimi) {
        g_critical(NOT_NULL);
        return;
    }

    kimi->ui_callbacks.show_banner(title, text_message);
}

