#ifndef SETTINGS_H
#define SETTINGS_H

#include <glib.h>

#define KIMI_CONF_ERROR g_quark_from_static_string("kimi_conf_error_quark")

/**
 * Errors, that may occurs is libkimi_config
 */
enum {
    KIMI_CONF_ERROR_NO_SUCH_OPTION
};

typedef struct _Option Option;
typedef struct _ConfigModule ConfigModule;
typedef struct _OptionValue OptionValue;


/**
 * Settings type structure
 */
enum _OptionType {
    OT_STRING, /**< Text input field */
    OT_PASSWORD, /**< Password input field (gui should display entering symbols as "*") */
    OT_BOOLEAN, /**< Check filed, that has two states - enabled, disabled */
    OT_INTEGER, /**< Integer tyoe */
    OT_SELECT /**< Choose from several options */
};

enum _OptionState {
    OS_UNSET,
    OS_SET,
    OS_INVALID
};

/**
 * Option flags
 */
enum _OptionFlags {
    OF_DISPLAYABLE = 1 << 0, /**< Option will be displayed to user */
    OF_STORABLE = 1 << 1 /**< Option will be stored to disk */
};

typedef enum _OptionType OptionType;
typedef enum _OptionState OptionState;
typedef enum _OptionFlags OptionFlags;

struct _OptionValue {
    OptionState state;
    union {
        char* v_text;
        int v_boolean;
        int v_int;
        float v_float;
        struct {
            GPtrArray* v_sel_list;
            int v_sel_index;
        };
    };
};

struct _Option {
    ConfigModule* cm;
    char* id;
    char* string;
    OptionType type;
    OptionValue value;
    int flags;
};

struct _ConfigModule {
    Kimi* kimi;
    GHashTable* options;
    char* name;
    char* id;
};

void kimi_conf_initialize(Kimi* kimi, const char* dir);

void kimi_conf_deinitialize(Kimi* kimi);

/**
 * @brief Create new config module
 * @para
 * @param kimi Kimi context
 * @param name Human-readable name of module
 * @return Config module handle
 */
ConfigModule* kimi_conf_create_config_module(Kimi* kimi, const char* id, const char* name, GError** error);

/**
 * @brief Create new option in config module
 * This function creates new option with given id and type.
 * To set value of option kimi_conf_set_option should be used;
 * @param cm Config module
 * @param id If of options
 * @param type Type of option
 */
Option* kimi_conf_register_option(ConfigModule* cm,
                              const char* id,
                              const char* string,
                              OptionType type,
                              int flags,
                              GError** error);

/**
 * @brief Set value to option with given id
 * @param cm ConfigModule, that contains options
 * @param id Id of options
 * @param opt value
 * @param error GError
 * @return 0 on success, -1 on error
 * @deprecated Use kimi_conf_get_option + kimi_conf_set_* instead
 */
int kimi_conf_set_option(ConfigModule* cm, const char* id, OptionValue opt, GError** error);

/**
 * @brief Set value to option with given id
 * @param cm ConfigModule, that contains options
 * @param id Id of options
 * @param opt value
 * @param error GError
 * @return Value of option. Option will be invalid if id doesn't exist
 */
Option* kimi_conf_get_option(ConfigModule* cm, const char* id, GError** error);

/** 
 * @brief Make user set option
 * @param cm ConfigModule, that contains options
 * @param error GError
 */
int kimi_conf_show_all_options_to_user(ConfigModule* cm, const char* message,  GError** error);

/** 
 * @brief Make user set option (with array)
 * @param cm ConfigModule, that contains options
 * @param options Array of options
 * @param error GError
 */
int kimi_conf_show_all_options_to_user_array(ConfigModule* cm, const char* message, GPtrArray* options, GError** error);

/* ===================================== */
/* ========== Getters/setters ========== */
/* ===================================== */
OptionState kimi_conf_option_get_state(Option* opt);

int kimi_conf_option_get_int(Option* opt);

const char* kimi_conf_option_get_string(Option* opt);

int kimi_conf_option_get_boolean(Option* opt);


void kimi_conf_option_sel_init(Option* opt);

void kimi_conf_option_sel_add(Option* opt, const char* string);


void kimi_conf_option_set_int(Option* opt, int);

void kimi_conf_option_set_string(Option* opt, const char*);

void kimi_conf_option_set_boolean(Option* opt, int);



const GPtrArray* kimi_conf_get_config_modules(Kimi* kimi);

GPtrArray* kimi_conf_get_options(ConfigModule* cm);

#endif

