#ifndef EVENT_MANAGER_H
#define EVENT_MANAGER_H

#include <glib.h>
#include "period.h"
typedef struct _Kimi Kimi;
#include "settings.h"
#define BUF_SIZE 1024
#define MODULE_INITIALIZE_FUNCTION "kimi_module_initialize"

#define ENTER g_debug("Enter")
#define EXIT g_debug("Exit")

#define CHECK_ALLOC(c, e, ...) \
    if (!c) { \
        g_set_error(e, KIMI_ERROR, KIMI_ALLOCATE_ERROR, ALLOCATE_ERROR_STRING, sizeof(c)); \
        __VA_ARGS__; \
        return NULL; \
    }
    
#define CHECK_ALLOC2(c, e, ...) \
    if (!c) { \
        g_set_error(e, KIMI_ERROR, KIMI_ALLOCATE_ERROR, ALLOCATE_ERROR_STRING, sizeof(c)); \
        __VA_ARGS__; \
        return -1; \
    }
 
#define KIMI_REPLACE_TEXT_FIELD(ev, field, value) \
    if (ev->field) free(ev->field); \
    ev->field = strdup(value);
    

//typedef struct _Kimi Kimi;
typedef struct _Event Event;
typedef struct _Module Module;
typedef struct _DateEventList DateEventList;
typedef struct _KimiUICallbacks KimiUICallbacks;
typedef struct _DynamicField DynamicField;

struct _KimiUICallbacks {
    void (*show_banner)(const char* title, const char* text_message);    
    void (*show_options)(ConfigModule* cm, const char* title, const char* message, GPtrArray* options);
};

/**
 * Libkimi context
 */
struct _Kimi {
    KimiUICallbacks ui_callbacks;
    GHashTable* modules; /**< Hashtable, that contains pairs <char* service_id, Module* mod> */
    
    GArray* date_event_list; /**< Current view */
    time_t date_id_list_start;
    time_t date_id_list_end;
    
    Module* current_module; /**<Last called module */
    
    GPtrArray* config_modules;
    GKeyFile* config;
    char* config_file;   
};

/**
 * A structure to represent event
 */
struct _Event {
    char* id; /**< Event id */
    int id_type; /**< Event type */
    char* title;/**< Event title (or summary descriptio) */
    time_t start_time; /**< Start time of event in seconds 01.01.1970 UTC */
    time_t end_time; /**< End time of event in seconds 01.01.1970 UTC */
    char* description; /**< Description of event */
    time_t remind;
    Period* per; /**< Pointer to structure Period, that describes recurrence of event */
    char* location; /**< Location of event */
    char* service_event_id; /**< Id of service, to that event belongs */
    
    GHashTable* dynamic_fields; /**< Hash table for custom fields */
};

/**
 * A structure, thas contains date and list of events, which happens this day. 
 */
struct _DateEventList {
    time_t date; /**< Date */
    GPtrArray* events; /**< Event array */
};

/**
 * A structure, that contains information about service
 */
struct _Module {
    char* service_name; /**< Human-readable name of service. */
    char* service_string; /**< Module id */
    void* data; /**< Module data. Module can store anything in this field */
    void* service_pixmap;
//    GtkWidget* (*get_settings_widget)(Kimi* app, GError** error);
    Event* (*get_event_by_id)(const char* id, Kimi* app, GError** error); /**< Function, thas should return event with given id */
    int (*remove_event)(const char* id, Kimi* app_data, GError** error); /**< Remove event */
    GPtrArray* (*get_events_by_period)(time_t start, time_t end, Kimi* app, GError** error); /**< This function must return an array
        of events, that have instance in interval (start, end). Each event should present in array only once. If module can't return event in period,
        it may return all events, but it requires more memory and CPU */
    int (*store_event)(Event* event, Kimi* app, GError** error); /**< Store or replace event into service. If event->id == NULL, then new event will be created */ 
    GPtrArray* (*filter)(Event*);
    void (*deinitialize)(Kimi* data, Module* mod); /**< Deinitialize module */
    
    /* Fields, that modules shouldn't access */
    
};

struct _DynamicField {
    void* data;
    void (*free_function)(void*);
};

#define ALLOCATE_ERROR_STRING "Failed allocate %d bytes"
#define CANNOT_GET_EVENT_STRING "Failed get event with id \"%s\" from service with id \"%s\": %s"
#define STORE_EVENT_FAILED_STRING "Failed to store event with id \"%s\" to service with id \"%s\": %s"
#define NO_SUCH_MODULE_STRING "Module with name \"%s\" hasn't been registered"
#define REMOVE_EVENT_FAILED_STRING "Failed to remove event with id \"%s\" to service with id \"%s\": %s"
#define MODULE_LOADING_ERROR_STRING "Cannot load module: %s"

#define KIMI_ERROR g_quark_from_static_string("kimi_error_quark")

/**
 * Errors, that may occurs is libkimi
 */
enum {
    KIMI_ALLOCATE_ERROR = 1, /**< Allocation error */
    KIMI_CANNOT_GET_EVENT, /**< Error, while trying to get event from module */
    KIMI_STORE_EVENT_FAILED, /**< Store event error */
    KIMI_NO_SUCH_MODULE, /**< Module not found. May occurs in kimi_get_module_data */
    KIMI_REMOVE_EVENT_FAILED, /**< Cannot remove event */
    KIMI_MODULE_LOADING_ERROR /**< Module initialization error */
};

/*#define KIMI_ALLOCATE_ERROR 1
#define KIMI_CANNOT_GET_EVENT 2
#define KIMI_STORE_EVENT_FAILED 3*/

/**
 * @brief Initialize event manager subsystem
 * This function should be called before any module initialization
 * @param dirs list of directories, where search modules
 * @param error Double pointer to GError
 * @return calendar handle
 */
Kimi* kimi_init(GPtrArray* dirs, KimiUICallbacks callbacks, GError** error);

/**
 * @brief Deinitialize event manager subsystem
 *
 * @param data Application data
 */
void kimi_deinit(Kimi* data);

/**
 * @brief Creates new event struct.
 *
 * Function creates new event structure and initializes it.
 *
 * @param error Double pointer to GError
 * @return struct event
 * @retval !NULL success
 * @retval NULL error
 */
Event* kimi_event_new(GError** error);

/**
 * @brief free event structure
 *
 * Function frees memory, that has allocated for event structure
 *
 * @param event event structure
 * @param error Double pointer to GError
 * @return return code
 * @retval 0 success
 * @retval !=0 error
 */
int kimi_event_free(Event* ev, GError** error);

/**
 * @brief free event array structure
 *
 * Function frees memory, that has allocated for event array
 *
 * @param event event array
 * @param error Double pointer to GError
 * @return return code
 * @retval 0 success
 * @retval !=0 error
 */
void kimi_free_array(GPtrArray* arr);

/**
 * @brief copy event structure
 * @param event
 * @param GError
 * @return copy of event
 *
 */
Event* kimi_event_dup(Event* ev, GError** error);

/**
 * @brief Fill structure with given values
 *
 *
 * @return -1 if error
 */
int kimi_event_fill(Event* ev,
        const char* service_id,
        const char* id, 
        const char* title,
        const char* desc,
        const char* location,
        time_t start,
        time_t end,
        Period* period,
        GError** error);

void kimi_event_set_field_free_function(Event* ev, const char* name, void (*free_function)(void*));

/**
 * @brief This function set the field with name 'name'
 * Last value of field will be freed with free_function (if it is not null)
 * @param ev Event
 * @param name Name of field
 * @param data Data to store
 * @return Previous value of field (to allow user free it)
 */
void* kimi_event_set_field(Event* ev, const char* name, void* data);
        
/**
 * @brief This function get the field with name 'name'
 * @param ev Event
 * @param name Name of fields
 * @return Value of field
 */
void* kimi_event_get_field(Event* ev, const char* name);

/**
 * @brief Register service module 
 *
 * @param mod Module description
 * @param data Application data
 * @param GError
 */
int kimi_register_module(Module* mod, Kimi* data, GError** error);

/**
 * @brief Callback for GHashTable
 *
 */
void kimi_deinitialize_module(gpointer key, gpointer val, gpointer data);

/**
 * @brief Get module data
 * 
 * @param service identifier. Can be NULL, then function will return data of last called module
 * @param data Application data
 *
 */
void* kimi_get_module_data(const char* service_string, Kimi* data, GError** error);

/**
 * @brief Get module
 *
 * @param service identifier
 * @param data Application data
 *
 * @return module pointer, or NULL if not exists
 */
Module* kimi_get_module(const char* service_string, Kimi* data);

/**
 * @brief Get list of modules
 * 
 * @param Kimi handle
 * @return GPtrArray<Module*>. It should be freed by g_ptr_array_free(arr, TRUE)
 */
GPtrArray* kimi_get_modules(Kimi* data);

/**
 * @brief Store event structure into database
 *
 * Function stores event into database.
 * If ev->id = NULL, then new database record will be created.
 * Function returns errors: KIMI_DB_ERROR (database error)
 *
 * @param ev event, that will be overwrited
 * @param app Kimi
 * @param error Double pointer to GError
 * @return return code
 * @retval 0 success
 * @retval -1 error
 */
int kimi_store(Event *ev, Kimi* data, GError **error);

/**
 * @brief Get event from service
 * @param id event id
 * @param service service id
 * @param data Application data
 * @param GError
 */
Event* kimi_get_by_id(const char* id, const char* service, Kimi* data, GError **error);

/**
 * @brief Remove event from service
 * @param id event id
 * @param service service id
 * @param data Application data
 * @param GError
 */
int kimi_event_remove(const char* id, const char* service, Kimi* app_data, GError** error);

/**
 * @brief Get events by period
 * @param start begin of time interval
 * @param end end of time interval
 * @param data Application data
 * @param GError
 */
int kimi_get_by_period(time_t start, time_t end, Kimi *app_data, GError **error);

/**
 * @brief
 * @param mask Condition for filter events
 * @param data Application data
 * @return Events array
 */
GPtrArray* kimi_filter(Event* mask, Kimi* data);

/**
 * @brief Insert event to DateEventList
 * @param data Kimi handle
 * @param ev Event to store
 * @return void
 */
void kimi_add_event_to_date_event_list(Kimi* data, Event* ev);

/**
 * @brief Remove event form DateEventList
 * @param data Kimi handle
 * @param id Event id
 * @param service Service id
 */
void kimi_remove_event_from_date_event_list(Kimi* data, const char* id, const char* service);

/**
 * @brief Function returns all events in current date_event_list
 * @return Events. Array should be freed: g_ptr_array_free(arr, TRUE);
 */
GPtrArray* kimi_get_date_event_list_array(Kimi* data);

/**
 * @brief Initialize date_event_list
 * @param data Application data
 * @param is_create_new if TRUE, then new date_event_list will be created.
 */
void kimi_date_event_list_free(Kimi* data, int is_create_new);

/**
 * @brief Send message to UI
 * @param title 
 * @param text_message
 */
void kimi_show_banner(Kimi* kimi, const char* title, const char* text_message);

/* ================================================== */
/* =============== Setters / Getters ================ */
/* ================================================== */

void kimi_event_set_id(Event* ev, const char* str);
void kimi_event_set_title(Event* ev, const char* str);
void kimi_event_set_description(Event* ev, const char* str);
void kimi_event_set_location(Event* ev, const char* str);
void kimi_event_set_period(Event* ev, Period* per);

const char* kimi_event_get_id(Event* ev);
const char* kimi_event_get_title(Event* ev);
const char* kimi_event_get_description(Event* ev);
const char* kimi_event_get_location(Event* ev);
Period* kimi_event_get_period(Event* ev);
#endif /* EVENT_MANAGER_H */

