#include <glib.h>
#include <glib-object.h>
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <readline/readline.h>
#include <readline/history.h>

#include "process.h"
#include "kimiconsole.h"
#include "kimi.h"

#define BUF 1024
/**
 * @brief Check if string contains only space, tabs, newlines
 *
 * @param str String
 * @return result (0,1)
 */
static int is_blank(const char* str)
{
    static const char blank_symbols[] = {' ', '\t', '\n', '\0'};
    static const int blank_symbols_count = sizeof(blank_symbols) / sizeof(char);
    char c = 0;
    int i = 0, j = 0;
    for (i = 0; (c = str[i]) != '\0'; i++) {
        int found = 0;
        for (j = 0; j < blank_symbols_count; j++) {
            if (c == blank_symbols[j]) {
                found = 1;
                break;
            }
        }
        if (!found)
            return 0;
    }
    return 1;
}

static int parse_getopt(int argc, char** argv, KimiConsole* kc)
{
    int c;
    int index;
    kc->isInteractive = 1;
    static struct option long_options[] = 
    {
        {"file", 1, 0, 'f'},
        {"output", 1, 0, 'o'},
        {0, 0, 0, 0}
    };
    
    for (;;) {
        index = 0;
        c = getopt_long(argc, argv, "f:o:", long_options, &index);
        
        if (c == -1)
            return 0;

        switch (c) {
        case 0:   
            break;
            
        case 'f':
            kc->isInteractive = 0;
            kc->file = strdup(optarg); 
            break;
            
        case 'o':
            kc->outfile = strdup(optarg);
            break;

        case '?':
            return -1;

        default:
            return 0;
        }
    }
}

/**
 * @brief Perform one step.
 * If current mode is interactive, readline end execute it.
 * Else read next line from file handle and execut it.
 * @return -1 if EOF, or user has entered C-d
 */
static int step(KimiConsole* kc)
{
    if (kc->isInteractive) {
        char* line = readline(">>> ");
        if (line == NULL) {
            return -1;
        }
        if (!is_blank(line)) {
            add_history (line);
            process_command(kc, line);
        }
        free(line);
    } else {
        char line[BUF];
        char* res = fgets(line, BUF, kc->handle);
        if (res == NULL) {
            return -1;
        }
        line[strlen(line) - 1] = '\0';
        if (!is_blank(line)) {
            add_history(line);
            process_command(kc, line);
        }
    }
    return kc->stop;
}

int main(int argc, char** argv)
{
    g_type_init();
    int i;
    KimiConsole kc;
    memset(&kc, 0, sizeof(KimiConsole));

    if (parse_getopt(argc, argv, &kc) == -1) {
        if (kc.file)    
            free(kc.file);
        
        return EXIT_FAILURE;   
    }   
    
    /* If mode == interactive, trying to open script */
    if (!kc.isInteractive) {
        if (open_and_execute_script(&kc, kc.file) == -1) {
            if (kc.file)    
                free(kc.file);
            return EXIT_FAILURE;
        }
    }
    
    if (kc.outfile) {
        kc.out = fopen(kc.outfile, "w");
        free(kc.outfile); /* Name of file no more needed */
        if (!kc.out) {
            fprintf(stderr, "Cannot open output file\n");
            if (kc.file) 
                free(kc.file);
            if (kc.handle)
                fclose(kc.handle);
            return EXIT_FAILURE;
        }
    } else {

        kc.out = stdout;
    }
    
    kc.module_paths = g_ptr_array_new();
        
    while(step(&kc) != -1)
        ;
    
    if (kc.kimi)
        kimi_deinit(kc.kimi);

    printf("\n");

    if (kc.handle)
        fclose(kc.handle);
    if (kc.out != stdout)
        fclose(kc.out);
    if (kc.file)    
        free(kc.file);
    if (kc.module_paths) {
        for (i = 0; i < kc.module_paths->len; i++) {
            free(g_ptr_array_index(kc.module_paths, i));
        }
        g_ptr_array_free(kc.module_paths, TRUE);
    }
    return EXIT_SUCCESS;
}

int open_and_execute_script(KimiConsole* kc, const char* file)
{
    /* If file already opened, close it */
    if (kc->handle)
        fclose(kc->handle);

    /* Open new file */
    kc->handle = fopen(file, "r");

    /* Wrong file */
    if (!kc->handle) {
        fprintf(stderr, "No such file\n");
        return -1;
    } else {
        kc->isInteractive = 0;
        return 0;
    }

}
